package jp.co.sra.jun.goodies.stopwatch;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.awt.image.IndexColorModel;
import java.io.IOException;
import java.io.Writer;
import java.util.Timer;
import java.util.TimerTask;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StOpaqueImage;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.gauge.JunLevelGaugeModel;
import jp.co.sra.jun.goodies.utilities.JunControlUtility;

/**
 * JunStopwatch class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/04/19 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun652 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunStopwatch.java,v 8.7 2008/02/20 06:32:03 nisinaka Exp $
 */
public class JunStopwatch extends JunLevelGaugeModel implements ClipboardOwner {
	protected JunButtonModel resetButton;
	protected JunButtonModel toggleButton;
	protected Timer clockProcess;
	protected KeyListener _keyListener;
	protected transient StPopupMenu _popupMenu;

	/**
	 * Create a new instance of <code>JunStopwatch</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunStopwatch() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		resetButton = null;
		toggleButton = null;
		clockProcess = null;
		_keyListener = null;
		_popupMenu = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @category initialize-release
	 */
	public void release() {
		super.release();
		this.terminateClockProcess();
	}

	/**
	 * Start a clock process.
	 * 
	 * @category actions
	 */
	public void start() {
		if (this.toggleButton().value() == false) {
			this.toggleButtonAction();
		}
	}

	/**
	 * Stop a clock process.
	 * 
	 * @category actions
	 */
	public void stop() {
		if (this.toggleButton().value()) {
			this.toggleButtonAction();
		}
	}

	/**
	 * Reset a clock process.
	 * 
	 * @category actions
	 */
	public void zero() {
		this.resetButtonAction();
	}

	/**
	 * Answer the reset button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel resetButton() {
		if (resetButton == null) {
			StComposedText text = new StComposedText("0");
			StDisplayable image = new StImage(text.width() + 2, text.height() + 2);
			Graphics gc = image.asImage().image().getGraphics();
			try {
				gc.setColor(Color.black);
				text.displayOn_at_(gc, new Point(1, 1));
			} finally {
				if (gc != null) {
					gc.dispose();
					gc = null;
				}
			}
			StImage figure = image.asImage().convertToPalette_(new IndexColorModel(1, 2, new byte[] { (byte) 255, (byte) 0 }, new byte[] { (byte) 255, (byte) 0 }, new byte[] { (byte) 255, (byte) 0 }));

			image = new StImage(text.width() + 2, text.height() + 2);
			gc = image.asImage().image().getGraphics();
			try {
				gc.setColor(Color.black);
				Point[] points = new Point[] { new Point(0, 1), new Point(0, -1), new Point(1, 0), new Point(-1, 0) };
				for (int i = 0; i < points.length; i++) {
					text.displayOn_at_(gc, new Point(1 + points[i].x, 1 + points[i].y));
				}
			} finally {
				if (gc != null) {
					gc.dispose();
					gc = null;
				}
			}
			StImage shape = new StImage(image.asImage().width(), image.asImage().height(), StImage.MonoMaskColorModel());
			shape.copy_from_in_rule_(new Rectangle(0, 0, image.asImage().width(), image.asImage().height()), new Point(0, 0), image.asImage(), StImage.Over);
			image = new StOpaqueImage(figure, shape);
			JunButtonModel button = new JunButtonModel(false, image, new StBlockClosure() {
				public Object value_(Object model) {
					JunStopwatch.this.resetButtonAction();
					return null;
				}
			});
			resetButton = button;
		}
		return resetButton;
	}

	/**
	 * Action for the event when the reset button is pressed.
	 * 
	 * @category buttons
	 */
	public void resetButtonAction() {
		this.resetButton().value_(false);
		if (this.toggleButton().value()) {
			this.toggleButton().value_(false);
			this.terminateClockProcess();
		}
		this.initial_(this.nextMillisecondClockValue());
	}

	/**
	 * Answer the toggle button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel toggleButton() {
		if (toggleButton == null) {
			BufferedImage image = JunCursors.StopwatchCursorImage();
			JunButtonModel button = new JunButtonModel(false, image, new StBlockClosure() {
				public Object value_(Object model) {
					JunStopwatch.this.toggleButtonAction();
					return null;
				}
			});
			toggleButton = button;

		}
		return toggleButton;
	}

	/**
	 * Action for the event when the toggle button is pressed.
	 * 
	 * @category buttons
	 */
	public void toggleButtonAction() {
		this.toggleButton().value_(!this.toggleButton().value());
		if (this.toggleButton().value()) {
			this.resumeClockProcess();
		} else {
			this.terminateClockProcess();
		}
	}

	/**
	 * Notifies this object that it is no longer the owner of
	 * the contents of the clipboard.
	 *
	 * @param clipboard the clipboard that is no longer owned
	 * @param contents the contents which this owner had placed on the clipboard
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
	 * @category clipboards
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}

	/**
	 * Reset the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.gauge.JunLevelGaugeModel#reset()
	 * @category controlling
	 */
	public void reset() {
		super.reset();
		value = 0;
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param evt java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent evt) {
		this.release();
		super.noticeOfWindowClose(evt);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunStopwatchViewAwt(this);
		} else {
			return new JunStopwatchViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Stopwatch");
	}

	/**
	 * Answer my KeyListener which handles my keyboard events.
	 *
	 * @return java.awt.event.KeyListener
	 * @category keyboard
	 */
	public KeyListener _keyListener() {
		if (_keyListener == null) {
			_keyListener = new KeyAdapter() {
				public void keyPressed(KeyEvent evt) {
					if (JunStopwatch.this._keyboardEvent_(evt)) {
						evt.consume();
					}
				}
			};
		}
		return _keyListener;
	}

	/**
	 * Process the key event.
	 *
	 * @param evt java.awt.event.KeyEvent
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean _keyboardEvent_(KeyEvent evt) {
		if (evt.isConsumed()) {
			return false;
		}

		int code = evt.getKeyCode();
		switch (code) {
			case KeyEvent.VK_SPACE:
				this.toggleButtonAction();
				return true;
			case KeyEvent.VK_ENTER:
				this.resetButtonAction();
				return true;
			case KeyEvent.VK_S:
				this.toggleButtonAction();
				return true;
			case KeyEvent.VK_R:
				this.resetButtonAction();
				return true;
			case KeyEvent.VK_1:
				this.toggleButtonAction();
				return true;
			case KeyEvent.VK_0:
				this.resetButtonAction();
				return true;
			case KeyEvent.VK_NUMPAD1:
				this.toggleButtonAction();
				return true;
			case KeyEvent.VK_NUMPAD0:
				this.resetButtonAction();
				return true;
		}

		return false;
	}

	/**
	 * Copy the receiver's value to clipboard.
	 * 
	 * @category menu messages
	 */
	public void copyValue() {
		Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		StringSelection contents = new StringSelection(this.displayString());
		clipboard.setContents(contents, this);
	}

	/**
	 * Answer the background color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.goodies.gauge.JunLevelGaugeModel#backgroundColor()
	 * @category preferences
	 */
	public Color backgroundColor() {
		Color aColor = (Color) this.attributes().at_($("backgroundColor"));
		if (aColor == null) {
			aColor = StColorValue.Brightness_(1);
			this.attributes().at_put_($("backgroundColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the tick time.
	 * 
	 * @return int
	 * @category preferences
	 */
	public int tickTime() {
		Integer anInteger = (Integer) this.attributes().at_($("tickTime"));
		if (anInteger == null) {
			anInteger = new Integer(10);
			this.attributes().at_put_($("tickTime"), anInteger);
		}
		return anInteger.intValue();
	}

	/**
	 * Set the tick time.
	 * 
	 * @param anInteger int
	 * @category preferences
	 */
	public void tickTime_(int anInteger) {
		this.attributes().at_put_($("tickTime"), new Integer(Math.max(anInteger, 10)));
	}

	/**
	 * Answer the edge color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.goodies.gauge.JunLevelGaugeModel#valueColor()
	 * @category preferences
	 */
	public Color valueColor() {
		Color aColor = (Color) this.attributes().at_($("valueColor"));
		if (aColor == null) {
			aColor = Color.black;
			this.attributes().at_put_($("valueColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the value size.
	 * 
	 * @return int
	 * @category preferences
	 */
	public int valueSize() {
		Number aNumber = (Number) this.attributes().at_($("valueSize"));
		if (aNumber == null) {
			aNumber = new Integer(16);
			this.attributes().at_put_($("valueSize"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set the value size.
	 * 
	 * @param aNumber int
	 * @category preferences
	 */
	public void valueSize_(int aNumber) {
		this.attributes().at_put_($("valueSize"), new Integer(Math.max(4, Math.min(aNumber, 72))));
	}

	/**
	 * Answer the value style.
	 * 
	 * @return int
	 * @category preferences
	 */
	public int valueStyle() {
		Number aNumber = (Number) this.attributes().at_($("valueStyle"));
		if (aNumber == null) {
			aNumber = new Integer(Font.BOLD);
			this.attributes().at_put_($("valueStyle"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set the value style.
	 * 
	 * @param aNumber int
	 * @category preferences
	 */
	public void valueStyle_(int aNumber) {
		this.attributes().at_put_($("valueStyle"), new Integer(aNumber));
	}

	/**
	 * Answer the receiver's string representation.
	 * 
	 * @return java.lang.String
	 * @category printing
	 */
	public String displayString() {
		long clockValue = this._longValue();
		int msecValue = (int) (clockValue % 1000 / 10);
		int secondValue = (int) (clockValue / 1000 % 60);
		int miniteValue = (int) (clockValue / (1000 * 60) % 60);
		int hourValue = (int) (clockValue / (1000 * 60 * 60));

		StringBuffer buffer = new StringBuffer();
		buffer.append(hourValue);
		buffer.append(":");
		if (miniteValue < 10) {
			buffer.append("0");
		}
		buffer.append(miniteValue);
		buffer.append(":");
		if (secondValue < 10) {
			buffer.append("0");
		}
		buffer.append(secondValue);
		buffer.append(".");
		if (msecValue < 10) {
			buffer.append("0");
		}
		buffer.append(msecValue);

		return buffer.toString();
	}

	/**
	 * Print the receiver's string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write(this.displayString());
	}

	/**
	 * Answer the receiver's next millisecond clock value.
	 * 
	 * @return long
	 * @category private
	 */
	public long nextMillisecondClockValue() {
		return JunControlUtility.NextMillisecondClockValue_(this.tickTime());
	}

	/**
	 * Resume a clock process.
	 * 
	 * @category private
	 */
	protected synchronized void resumeClockProcess() {
		if (clockProcess != null) {
			clockProcess.cancel();
		}
		final JunStopwatch self = this;
		clockProcess = new Timer();
		self.resume_(this.nextMillisecondClockValue());
		clockProcess.scheduleAtFixedRate(new TimerTask() {
			public void run() {
				self.value_(self._doubleValue() + self.offset() + self.tickTime());
			}
		}, 0, this.tickTime());
	}

	/**
	 * Terminate a clock process.
	 * 
	 * @category private
	 */
	protected synchronized void terminateClockProcess() {
		if (clockProcess != null) {
			clockProcess.cancel();
		}
		clockProcess = null;
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem($String("Copy"), $("copyValue"), new MenuPerformer(this, "copyValue")));
		}
		return _popupMenu;
	}
}
