package jp.co.sra.jun.goodies.spirodesign;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;

import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StObject;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;

/**
 * JunSpiroObject class
 * 
 *  @author    m-asada
 *  @created   2006/03/28 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun676 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSpiroObject.java,v 8.11 2008/02/20 06:32:03 nisinaka Exp $
 */
public abstract class JunSpiroObject extends StObject implements StDisplayable {
	protected Color spiroColor;
	protected int spiroWidth;

	/**
	 * Create a new instance of <code>JunSpiroObject</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunSpiroObject() {
		super();
		this.initialize();
	}

	/**
	 * Initialize the receiver when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		spiroColor = null;
		spiroWidth = -1;
	}

	/**
	 * Answer the receiver's spiro color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color color() {
		if (spiroColor == null) {
			spiroColor = this.defaultColor();
		}
		return spiroColor;
	}

	/**
	 * Set the receiver's spiro color.
	 * 
	 * @param aColor java.awt.Color
	 * @category accessing
	 */
	public void color_(Color aColor) {
		spiroColor = aColor;
	}

	/**
	 * Answer the receiver's spiro width.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int width() {
		if (spiroWidth <= 0) {
			spiroWidth = this.defaultWidth();
		}
		return spiroWidth;
	}

	/**
	 * Set the receiver's spiro width.
	 * 
	 * @param aNumber int
	 * @category accessing
	 */
	public void width_(int aNumber) {
		spiroWidth = Math.max(aNumber, 1);
	}

	/**
	 * Answer the receiver's mark.
	 * 
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public abstract Rectangle mark();

	/**
	 * Answer the receiver's marks.
	 * 
	 * @return java.awt.Rectangle[]
	 * @category accessing
	 */
	public abstract Rectangle[] marks();

	/**
	 * Answer the receiver's pen points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public abstract Jun2dPoint[] points();
	
	/**
	 * Answer the receiver as StImage.
	 *
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		Rectangle bounds = this.bounds();
		StImage anImage = new StImage(bounds.x + bounds.width, bounds.y + bounds.height);
		Graphics aGraphics = anImage.image().getGraphics();
		try {
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Answer the default color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	public Color defaultColor() {
		return Color.black;
	}

	/**
	 * Answer the default mark extent.
	 * 
	 * @return java.awt.Dimension
	 * @category defaults
	 */
	public Dimension defaultMarkExtent() {
		return new Dimension(5, 5);
	}

	/**
	 * Answer the default width.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultWidth() {
		return 1;
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle boundingBox() {
		return new Rectangle(0, 0, 0, 0);
	}

	/**
	 * Answer the bounding box of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		return this.preferredBounds();
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		Rectangle preferredBounds = this.boundingBox();
		preferredBounds.grow(Math.max(this.defaultMarkExtent().width, this.width()), Math.max(this.defaultMarkExtent().height, this.width()));
		return preferredBounds;
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		this.displayOn_at_with_Marks_(graphicsContext, new Point(0, 0), true);
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param displayPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics graphicsContext, Point displayPoint) {
		this.displayOn_at_with_Marks_(graphicsContext, displayPoint, true);
	}

	/**
	 * Display the receiver with marks on the graphics at the specified point.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param displayPoint java.awt.Point
	 * @param aBoolean boolean
	 * @category displaying
	 */
	public abstract void displayOn_at_with_Marks_(Graphics graphicsContext, Point displayPoint, boolean aBoolean);

	/**
	 * Apply attributes to the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics2D
	 * @category private
	 */
	protected void applyAttributesToGraphicsContext_(Graphics2D graphicsContext) {
		graphicsContext.setColor(this.color());
		graphicsContext.setStroke(new BasicStroke(this.width(), BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		graphicsContext.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
	}
}
