package jp.co.sra.jun.goodies.sound;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.util.Enumeration;
import java.util.Hashtable;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunSoundWaveMeterViewAwt class
 * 
 *  @author    Hoshi Takanori
 *  @created   2003/08/29 (by Hoshi Takanori)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSoundWaveMeterViewAwt.java,v 8.10 2008/02/20 06:32:03 nisinaka Exp $
 */
public class JunSoundWaveMeterViewAwt extends JunAbstractViewCanvas implements JunSoundWaveMeterView {

	protected int[][] cacheTable;
	protected int previousPosition = 0;

	/**
	 * Create a new instance of <code>JunSoundWaveMeterViewAwt</code> and initialize it.
	 * 
	 * @param aSoundMeterModel jp.co.sra.jun.goodies.sound.JunSoundMeterModel
	 * @category Instance creation
	 */
	public JunSoundWaveMeterViewAwt(JunSoundMeterModel aSoundMeterModel) {
		super(aSoundMeterModel);
	}

	/**
	 * Answer my model as JunSoundMeterModel.
	 *
	 * @return jp.co.sra.jun.goodies.sound.JunSoundMeterModel
	 * @category model accessing
	 */
	public JunSoundMeterModel getSoundMeterModel() {
		return (JunSoundMeterModel) this.model();
	}

	/**
	 * Answer my cache table.
	 *
	 * @return int[][]
	 * @category accessing
	 */
	public int[][] cacheTable() {
		if (cacheTable == null) {
			cacheTable = this.makeCacheTable();
		}
		return cacheTable;
	}

	/**
	 * Display on a graphics context.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		if (this.isShowing() == false) {
			return;
		}

		this.flushCacheTable();
		Dimension size = this.getSize();
		this.displayOn_from_to_(graphicsContext, 0, size.width);
	}

	/**
	 * Display the current position on the graphics context.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param key java.lang.Double
	 * @see jp.co.sra.jun.goodies.sound.JunSoundWaveMeterView#displayOn_currentPosition_(java.awt.Graphics, java.lang.Double)
	 * @category displaying
	 */
	public void displayOn_currentPosition_(Graphics graphicsContext, Double key) {
		Dimension size = this.getSize();
		int x = (int) (key.doubleValue() * size.width);
		this.displayOn_from_to_(graphicsContext, Math.max(previousPosition - 1, 0), Math.min(previousPosition + 1, size.width));
		graphicsContext.setColor(Color.gray);
		graphicsContext.fillRect(x, 0, 1, size.height);
		int[][] table = this.cacheTable();
		double[] value = this.getSoundMeterModel().soundLevelDataAt_(key);
		if (value != null) {
			int low = (int) (value[0] * size.height);
			int high = (int) (value[1] * size.height);
			this.cacheTable_at_low_high_(table, x, low, high);
		}
		previousPosition = x;
	}

	/**
	 * Display from startX to endX on the graphics context.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param startX int
	 * @param endX int
	 * @see jp.co.sra.jun.goodies.sound.JunSoundWaveMeterView#displayOn_from_to_(java.awt.Graphics, int, int)
	 * @category displaying
	 */
	public void displayOn_from_to_(Graphics graphicsContext, int startX, int endX) {
		int[][] table = this.cacheTable();
		Dimension size = this.getSize();
		for (int x = startX; x < endX; ++x) {
			if (table[x] == null) {
				graphicsContext.setColor(Color.white);
				graphicsContext.fillRect(x, 0, 1, size.height);
			} else {
				int low = table[x][0];
				int high = table[x][1];
				graphicsContext.setColor(Color.white);
				graphicsContext.fillRect(x, 0, 1, low);
				graphicsContext.fillRect(x, high, 1, size.height - high);
				float darkness = 0.75f - (float) (high - low) / size.height / 2;
				graphicsContext.setColor(new Color(darkness, darkness, darkness));
				graphicsContext.fillRect(x, low, 1, Math.max(high - low, 1));
			}
		}
	}

	/**
	 * Action for the update notification.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("wave")) {
			Object parameter = evt.getParameter();
			if (parameter instanceof Double) {
				Graphics graphics = this.getGraphics();
				if (graphics != null) {
					this.displayOn_currentPosition_(graphics, (Double) parameter);
				}
			}
		} else if (aSymbol == $("level")) {
			return;
		} else {
			super.update_(evt);
		}
	}

	/**
	 * Flush my cache table.
	 * 
	 * @category private
	 */
	public void flushCacheTable() {
		cacheTable = null;
	}

	/**
	 * Put low and high to cache table.
	 * 
	 * @param table int[][]
	 * @param x int
	 * @param low int
	 * @param high int
	 * @category private
	 */
	public void cacheTable_at_low_high_(int[][] table, int x, int low, int high) {
		table[x] = new int[] { low, high };
	}

	/**
	 * Make my cache table.
	 * 
	 * @return int[][]
	 * @category private
	 */
	public int[][] makeCacheTable() {
		Dimension size = this.getSize();
		int[][] table = new int[size.width + 1][];
		Hashtable data = this.getSoundMeterModel().soundLevelData();
		for (Enumeration e = data.keys(); e.hasMoreElements();) {
			Object key = e.nextElement();
			double[] value = this.getSoundMeterModel().soundLevelDataAt_(key);
			if (key instanceof Double && value != null) {
				int x = (int) (((Double) key).doubleValue() * size.width);
				int low = (int) (value[0] * size.height);
				int high = (int) (value[1] * size.height);
				this.cacheTable_at_low_high_(table, x, low, high);
			}
		}
		return table;
	}
}
