package jp.co.sra.jun.goodies.prolog;

import jp.co.sra.smalltalk.*;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunPrologScanner class
 * 
 *  @author    kondo
 *  @created   1999/09/09 (by kondo)
 *  @updated   2003/04/25 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun301 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPrologScanner.java,v 8.11 2008/02/20 06:32:02 nisinaka Exp $
 */
public class JunPrologScanner extends JunAbstractObject {

	protected static final char CR = '\r';
	protected static final char LF = '\n';
	protected static final char EndChar = '^';

	protected StReadStream source;
	protected int mark;
	protected Object token;
	protected StSymbol tokenType;
	protected StSymbol[] typeTable;
	protected char endChar;
	protected StBlockClosure failBlock;

	/**
	 * Create a new instance of JunPrologScanner and initialize it.
	 */
	public JunPrologScanner() {
		super();
		this.initScanner();
	}

	/**
	 * Shorten a string by ellipsis if too long.
	 * 
	 * @return java.lang.String
	 * @param aString java.lang.String
	 * @param count int
	 */
	public static String ContractTo(String aString, int count) {
		int length = aString.length();
		if (length <= count) {
			return aString;
		}

		int half = (count / 2) - 1;
		return aString.substring(0, half) + "..." + aString.substring(length - count + half + 3);
	}

	/**
	 * Get a binary token.
	 */
	public void xBinary() {
		failBlock.value_("Syntax error" + source.peek());
	}

	/**
	 * Got a colon as a token.
	 */
	public void xColon() {
		source.next();
		char ch = source.peek();
		if (ch == '-') {
			tokenType = $("neck");
			this.singleChar_(tokenType);
		} else {
			this.unNextToken();
			tokenType = $("symbol");
			token = JunPrologSymbol.ReadFrom_(source);
		}
	}

	/**
	 * Got a comment as a token.
	 */
	public void xComment() {
		char ch;
		while ((ch = this.nextChar()) != CR) {
			if (ch == endChar) {
				this.tokenType = $("eof");
				return;
			}
		}
		this.nextToken();
	}

	/**
	 * Got a number as a token.
	 */
	public void xDigit() {
		tokenType = $("number");
		token = StNumber.ReadFrom_(source);
	}

	/**
	 * Got a dollar as a token.
	 */
	public void xDollar() {
		throw new SmalltalkException("Should not be called.  It is a symbol.");
	}

	/**
	 * Got a double quote as a token.
	 */
	public void xDoubleQuote() {
		tokenType = $("string");
		token = JunPrologString.ReadFrom_(source);
	}

	/**
	 * Got a sign as a token.
	 */
	public void xSign() {
		char sign = this.nextChar();
		char ch = this.peekChar();

		if (Character.isDigit(ch)) {
			tokenType = $("number");
			token = StNumber.ReadFrom_(source);
			if (sign == '-') {
				token = StNumber._Negate((Number) token);
			}
		} else {
			this.unNextChar();
			tokenType = $("symbol");
			token = JunPrologSymbol.ReadFrom_(source);
		}
	}

	/**
	 * Got a single quote as a token.
	 */
	public void xSingleQuote() {
		tokenType = $("symbol");
		token = JunPrologSymbol.ReadFrom_(source);
	}

	/**
	 * Got a symbol as a token.
	 */
	public void xSymbol() {
		tokenType = $("symbol");
		token = JunPrologSymbol.ReadFrom_(source);
	}

	/**
	 * Got a variable as a token.
	 */
	public void xVariable() {
		tokenType = $("variable");
		token = JunPrologVariable.ReadFrom_(source);
	}

	/**
	 * Initialize the prolog scanner.
	 */
	protected void initScanner() {
		typeTable = JunPrologScannerTable.ScannerTable();
		endChar = EndChar;

		failBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				String errorMessage = (String) anObject;
				String label = errorMessage + " near " + ContractTo(token.toString(), 10);
				String string = source.upToEnd();
				if (string.length() == 0) {
					string = "--> end of file";
				} else {
					string = "--> " + ContractTo(string, 30);
				}
				throw SmalltalkException.Error(label + "\n" + string);
			}
		};
	}

	/**
	 * Get a next token with multiple characters.
	 * 
	 * @param type jp.co.sra.smalltalk.StSymbol
	 * @throws SmalltalkException
	 */
	protected void multiChar_(StSymbol type) {
		try {
			this.perform_(type.toString());
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Get the next character from the source.
	 * 
	 * @return char
	 */
	protected char nextChar() {
		char c = source.next();
		if (c == LF) {
			return CR;
		} else if (c == CR) {
			if (source.peek() == LF) {
				source.next();
			}
			return CR;
		} else if (c == 0) {
			return endChar;
		}
		return c;
	}

	/**
	 * Get a next token.
	 * 
	 * @return java.lang.Object
	 */
	protected Object nextToken() {
		mark = source.position();
		char ch = this.peekChar();
		tokenType = typeTable[ch];
		while (this.tokenType == $("xDelimiter")) {
			this.nextChar();
			ch = this.peekChar();
			tokenType = typeTable[ch];
		}

		if (tokenType.toString().charAt(0) == 'x') {
			this.multiChar_(tokenType);
		} else {
			this.singleChar_(tokenType);
		}

		return token;
	}

	/**
	 * Initialize for the source stream to scan.
	 * 
	 * @param inputStream jp.co.sra.smalltalk.StReadStream
	 */
	protected void on_(StReadStream inputStream) {
		source = inputStream;
		mark = source.position();
	}

	/**
	 * Peek the next character from the source stream.
	 * 
	 * @return char
	 */
	protected char peekChar() {
		char c = source.peek();
		if (c == LF) {
			return CR;
		} else if (c == CR) {
			return CR;
		} else if (c == 0) {
			return endChar;
		}
		return c;
	}

	/**
	 * Get a next token which is just a single character.
	 * 
	 * @param type jp.co.sra.smalltalk.StSymbol
	 */
	protected void singleChar_(StSymbol type) {
		this.nextChar();
		token = type;
		if (tokenType != $("leftBrace")) {
			return;
		}

		tokenType = $("object");
		token = JunPrologObject.ReadFrom_(source);
	}

	/**
	 * Push back a character to the source stream.
	 */
	protected void unNextChar() {
		source.skip_(-1);
	}

	/**
	 * Push back a token to the source stream.
	 */
	protected void unNextToken() {
		source.position_(mark);
	}
}
