package jp.co.sra.jun.goodies.parameter;

import jp.co.sra.smalltalk.*;

/**
 * JunParameter class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/12/08 (by MATSUDA Ryouichi)
 *  @updated   2002/01/17 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunParameter.java,v 8.11 2008/02/20 06:32:01 nisinaka Exp $
 */
public class JunParameter extends StModel {

	protected String name;
	protected double value = Double.NaN;
	protected double minValue = Double.NaN;
	protected double maxValue = Double.NaN;
	protected double stepValue = Double.NaN;
	protected String parameterName;

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @category Instance creation
	 */
	public JunParameter(String aString) {
		this(aString, DefaultValue());
	}

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @param aNumber double
	 * @param maxNumber double
	 * @category Instance creation
	 */
	public JunParameter(String aString, double aNumber, double maxNumber) {
		this(aString, aNumber, DefaultMinValue(), maxNumber);
	}

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @param aNumber double
	 * @param minNumber double
	 * @param maxNumber double
	 * @category Instance creation
	 */
	public JunParameter(String aString, double aNumber, double minNumber, double maxNumber) {
		this.name_(aString);
		this.value_(aNumber);
		this.minValue_(minNumber);
		this.maxValue_(maxNumber);
	}

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @param aNumber double
	 * @param minNumber double
	 * @param maxNumber double
	 * @param stepNumber double
	 * @category Instance creation
	 */
	public JunParameter(String aString, double aNumber, double minNumber, double maxNumber, double stepNumber) {
		this(aString, aNumber, minNumber, maxNumber);
		this.stepValue_(stepNumber);
	}

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @param aNumber double
	 * @category Instance creation
	 */
	public JunParameter(String aString, double aNumber) {
		this(aString, aNumber, DefaultMaxValue());
	}

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @param anArrayOfValues double[]
	 * @category Instance creation
	 */
	public JunParameter(String aString, double[] anArrayOfValues) {
		this(aString);

		if (anArrayOfValues.length >= 1) {
			this.value_(anArrayOfValues[0]);
		}
		if (anArrayOfValues.length == 2) {
			this.minValue_(0);
			this.maxValue_(anArrayOfValues[1]);
		}
		if (anArrayOfValues.length >= 3) {
			this.minValue_(anArrayOfValues[1]);
			this.maxValue_(anArrayOfValues[2]);
		}
		if (anArrayOfValues.length >= 4) {
			this.stepValue_(anArrayOfValues[3]);
		}
	}

	/**
	 * Create a new instance of JunParameter and initialize it.
	 * 
	 * @param aString java.lang.String
	 * @param anArrayOfValues double[]
	 * @param displayString java.langString
	 * @category Instance creation
	 */
	public JunParameter(String aString, double[] anArrayOfValues, String displayString) {
		this(aString, anArrayOfValues);
		this.parameterName_(displayString);
	}

	/**
	 * Answer a default max value.
	 * 
	 * @return double
	 * @category Defaults
	 */
	public static final double DefaultMaxValue() {
		return 100;
	}

	/**
	 * Answer a default min value.
	 * 
	 * @return double
	 * @category Defaults
	 */
	public static final double DefaultMinValue() {
		return 0;
	}

	/**
	 * Answera default value.
	 * 
	 * @return double
	 * @category Defaults
	 */
	public static double DefaultValue() {
		return 0;
	}

	/**
	 * Answer the start value of the integer range.
	 *
	 * @return int
	 * @category accessing
	 */
	public final int _rangeStartValue() {
		return 0;
	}

	/**
	 * Answer the stop value of the integer range.
	 *
	 * @return int
	 * @category accessing
	 */
	public int _rangeStopValue() {
		if (Double.isNaN(stepValue)) {
			return 100;
		} else {
			return (int) Math.floor((this.maxValue() - this.minValue()) / this.stepValue());
		}
	}

	/**
	 * Answer the value int the integer range.
	 *
	 * @return int
	 * @category accessing
	 */
	public int _rangeValue() {
		return this._rangeStartValue() + (int) ((this._rangeStopValue() - this._rangeStartValue()) * (this.value() - this.minValue()) / (this.maxValue() - this.minValue()));
	}

	/**
	 * Set the value in the integer range.
	 *
	 * @param rangeValue int
	 * @category accessing
	 */
	public void _setRangeValue(int rangeValue) {
		double newValue = this.minValue + (this.maxValue() - this.minValue()) * (rangeValue - this._rangeStartValue()) / (this._rangeStopValue() - this._rangeStartValue());
		this.value_(newValue);
	}

	/**
	 * Answer the maxValue.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double maxValue() {
		return maxValue;
	}

	/**
	 * Set the maxValue.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void maxValue_(double aNumber) {
		maxValue = aNumber;
		this.changed_($("maxValue"));
	}

	/**
	 * Answer the minValue.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double minValue() {
		return minValue;
	}

	/**
	 * Set the minValue.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void minValue_(double aNumber) {
		minValue = aNumber;
		this.changed_($("minValue"));
	}

	/**
	 * Answer the name.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String name() {
		return name;
	}

	/**
	 * Set the name.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void name_(String aString) {
		name = aString;
		this.changed_($("name"));
	}

	/**
	 * Answer the stepValue.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double stepValue() {
		return stepValue;
	}

	/**
	 * Set the stepValue.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void stepValue_(double aNumber) {
		stepValue = aNumber;
		this.changed_($("stepValue"));
	}

	/**
	 * Answer the value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double value() {
		return value;
	}

	/**
	 * Set the value.
	 * 
	 * @param newValue double
	 * @category accessing
	 */
	public void value_(double newValue) {
		if (newValue < minValue) {
			newValue = minValue;
		}

		if (maxValue < newValue) {
			newValue = maxValue;
		}

		if (newValue != value) {
			value = newValue;
			this.changed_($("value"));
		}
	}

	/**
	 * Answer my current parameter name.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String parameterName() {
		if (parameterName == null) {
			return this.name();
		}
		return parameterName;
	}

	/**
	 * Set my new parameter name.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void parameterName_(String aString) {
		parameterName = aString;
	}
}
