package jp.co.sra.jun.goodies.nib;

import java.awt.Color;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.Arrays;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.track.JunSimpleSliderModel;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunNibChoice class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2004/03/19 (By Nobuto Matsubara)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun538 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunNibChoice.java,v 8.13 2008/02/20 06:31:51 nisinaka Exp $
 */
public class JunNibChoice extends JunApplicationModel implements StBlockValued {
	protected JunSimpleSliderModel simpleSlider;
	protected double[] nibCollection;
	protected double nibWidth;
	protected Color nibColor;
	protected StSymbol nibShape;

	/**
	 * Create a new instance of JunNibChoice.
	 * 
	 * @category Instance creation
	 */
	public JunNibChoice() {
		super();
	}

	/**
	 * Create an instance of JunNibChoice.
	 * 
	 * @param aCollection double[]
	 * @category Instance creation
	 */
	public JunNibChoice(double[] aCollection) {
		this.nibCollection_(aCollection);
	}

	/**
	 * Create an instance of JunNibChoice.
	 * 
	 * @param aCollection double[]
	 * @param nibWidth double
	 * @param nibColor java.awt.Color
	 * @param nibShape jp.co.sra.smalltalk.StSymbol
	 * @category Instance creation
	 */
	public JunNibChoice(double[] aCollection, double nibWidth, Color nibColor, StSymbol nibShape) {
		this.value_(new Object[] { aCollection, new Double(nibWidth), nibColor, nibShape });
	}

	/**
	 * Create an instance of JunNibChoice.
	 * 
	 * @param valueArray Object[]
	 * @category Instance creation
	 */
	public JunNibChoice(Object[] valueArray) {
		this.value_(valueArray);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		simpleSlider = null;
		nibCollection = null;
		nibWidth = 0.0;
		nibColor = null;
		nibShape = null;
	}

	/**
	 * Process the key event.
	 *
	 * @param ev java.awt.event.KeyEvent
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean _keyboardEvent(KeyEvent ev) {
		if (ev.isConsumed()) {
			return false;
		}

		int code = ev.getKeyCode();
		switch (code) {
			case KeyEvent.VK_DOWN:
			case KeyEvent.VK_LEFT:
				this.prevNibWidth();
				return true;
			case KeyEvent.VK_UP:
			case KeyEvent.VK_RIGHT:
				this.nextNibWidth();
				return true;
		}

		return false;
	}

	/**
	 * Set next nib width.
	 * 
	 * @category private
	 */
	protected void nextNibWidth() {
		int index = 0;
		for (int i = 0; i < this.nibCollection().length; i++) {
			if (this.nibCollection()[i] == this.nibWidth()) {
				index = i;
			}
		}
		if (index < this.nibCollection().length - 1) {
			this.nibWidth_(this.nibCollection()[index + 1]);
		}
	}

	/**
	 * Set previous nib width.
	 * 
	 * @category private
	 */
	protected void prevNibWidth() {
		int index = 0;
		for (int i = 0; i < this.nibCollection().length; i++) {
			if (this.nibCollection()[i] == this.nibWidth()) {
				index = i;
			}
		}
		if (index > 0) {
			this.nibWidth_(this.nibCollection()[index - 1]);
		}
	}

	/**
	 * Answer my KeyListener which handles my keyboard events.
	 *
	 * @return java.awt.event.KeyListener
	 * @category keyboard
	 */
	public KeyListener _keyListener() {
		return new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (_keyboardEvent(e)) {
					e.consume();
				}
			}
		};
	}

	/**
	 * Answer nib collection.
	 * 
	 * @return An array.
	 * @category accessing
	 */
	public double[] nibCollection() {
		if (nibCollection == null) {
			nibCollection = this.defaultNibCollection();
		}
		return nibCollection;
	}

	/**
	 * Set new nib collection.
	 * 
	 * @param aCollection double[]
	 * @category accessing
	 */
	public void nibCollection_(double[] aCollection) {
		double[] newCollection = new double[aCollection.length];
		System.arraycopy(aCollection, 0, newCollection, 0, aCollection.length);
		Arrays.sort(newCollection);
		nibCollection = newCollection;
	}

	/**
	 * Answer nib color.
	 * 
	 * @return java.awt.Color.
	 * @category accessing
	 */
	public Color nibColor() {
		if (nibColor == null) {
			nibColor = this.defaultNibColor();
		}
		return nibColor;
	}

	/**
	 * Set new color.
	 * 
	 * @param colorValue java.awt.Color.
	 * @category accessing
	 */
	public void nibColor_(Color colorValue) {
		if (this.nibColor().equals(colorValue)) {
			return;
		}
		nibColor = colorValue;
		this.changed_($("nibColor"));
	}

	/**
	 * Answer nib shape.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol.
	 * @category accessing
	 */
	public StSymbol nibShape() {
		if (nibShape == null) {
			nibShape = this.defaultNibShape();
		}
		return nibShape;
	}

	/**
	 * Set new shape.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol.
	 * @category accessing
	 */
	public void nibShape_(StSymbol aSymbol) {
		if (this.nibShape().equals(aSymbol)) {
			return;
		}
		if (!($("rectangle").equals(aSymbol) || $("circle").equals(aSymbol))) {
			return;
		}
		nibShape = aSymbol;
		this.changed_($("nibShape"));
	}

	/**
	 * Answer nib width.
	 * 
	 * @return nib width.
	 * @category accessing
	 */
	public double nibWidth() {
		if (nibWidth == 0) {
			if (this.nibCollection().length == 0) {
				this.nibCollection_(new double[] { this.defaultNibWidth() });
			}
			nibWidth = this.nibCollection()[0];
		}
		return nibWidth;
	}

	/**
	 * Set nib width.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void nibWidth_(double aNumber) {
		if (nibWidth == aNumber) {
			return;
		}
		ArrayList list = new ArrayList();
		for (int i = 0; i < this.nibCollection().length; i++) {
			list.add(new Double(this.nibCollection()[i]));
		}
		if (list.contains(new Double(aNumber))) {
			nibWidth = aNumber;
		} else {
			for (int i = 0; i < this.nibCollection().length; i++) {
				if (this.nibCollection()[i] > aNumber) {
					nibWidth = this.nibCollection()[i];
					break;
				}
			}
		}
		if (!(nibWidth == (this.computeNibValueFromSliderValue_(((Double) this.simpleSlider().value()).doubleValue())))) {
			this.simpleSlider().value_(this.computeSliderValueFromNibValue_(nibWidth));
		}
		this.changed_($("nibWidth"));
	}

	/**
	 * Set the receiver's new nib width and nib color.
	 * 
	 * @param aNumber double
	 * @param colorValue java.awt.Color
	 * @category accessing
	 */
	public void nibWidth_nibColor_(double aNumber, Color colorValue) {
		if (Arrays.binarySearch(this.nibCollection(), aNumber) >= 0) {
			nibWidth = aNumber;
		} else {
			double[] nibCollection = this.nibCollection();
			nibWidth = Double.NaN;
			for (int i = 0; i < nibCollection.length; i++) {
				if (nibCollection[i] > aNumber) {
					nibWidth = nibCollection[i];
					break;
				}
			}
			if (Double.isNaN(nibWidth)) {
				nibWidth = nibCollection[nibCollection.length - 1];
			}
		}
		nibColor = colorValue;
		if (nibWidth != this.computeNibValueFromSliderValue_(this.simpleSlider().doubleValue())) {
			this.simpleSlider().value_(this.computeSliderValueFromNibValue_(nibWidth));
		}
		this.changed();
	}

	/**
	 * Answer a JunSimpleSliderModel instance.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunSimpleSliderModel
	 * @category accessing
	 */
	public JunSimpleSliderModel simpleSlider() {
		if (simpleSlider == null) {
			simpleSlider = new JunSimpleSliderModel();
			simpleSlider.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					JunNibChoice.this.nibWidth_(JunNibChoice.this.computeNibValueFromSliderValue_(((Number) anObject).doubleValue()));
					return null;
				}
			});
		}
		return simpleSlider;
	}

	/**
	 * Answer nib collection, width, color and shape with array.
	 * 
	 * @return Object[]
	 * @see jp.co.sra.smalltalk.StValued#value()
	 * @category accessing
	 */
	public Object value() {
		return new Object[] { this.nibCollection(), new Double(this.nibWidth()), this.nibColor(), this.nibShape() };
	}

	/**
	 * Set nib collection, width, color and shape with array.
	 * 
	 * @param anObject Object[]{ nibCollectin, nibWidth, nibColor, nibShape }
	 * @category accessing
	 */
	public void value_(Object anObject) {
		Object[] anArray = (Object[]) anObject;
		this.nibCollection_((double[]) anArray[0]);
		this.nibWidth_(((Double) anArray[1]).doubleValue());
		this.nibColor_((Color) anArray[2]);
		this.nibShape_((StSymbol) anArray[3]);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunNibChoiceViewAwt.WithSlider(this);
		} else {
			return JunNibChoiceViewSwing.WithSlider(this);
		}
	}

	/**
	 * Answer default nib collection.
	 * 
	 * @return Default nib collection.
	 * @category defaults
	 */
	public double[] defaultNibCollection() {
		return new double[] { 1, 3, 5, 7, 9, 11, 13, 15, 17, 19, 21, 23, 25 };
	}

	/**
	 * Answer default nib color.
	 * 
	 * @return java.awt.Color.
	 * @category defaults
	 */
	public Color defaultNibColor() {
		return Color.black;
	}

	/**
	 * Answer default nib shape.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol.
	 * @category defaults
	 */
	public StSymbol defaultNibShape() {
		return $("rectangle");
	}

	/**
	 * Answer default nib width.
	 * 
	 * @return Default nib width.
	 * @category defaults
	 */
	public int defaultNibWidth() {
		return 1;
	}

	/**
	 * Compute a block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure.
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * compute nib value from slider value.
	 * 
	 * @param sliderValue double
	 * @return nib value.
	 * @category private
	 */
	protected double computeNibValueFromSliderValue_(double sliderValue) {
		int denominator = this.nibCollection().length - 1;
		int index = 0;
		if (denominator <= 0) {
			index = 0;
		} else {
			index = (int) (sliderValue * denominator + 0.5);
		}
		return this.nibCollection()[index];
	}

	/**
	 * Compute slider value from nib value.
	 * 
	 * @param nibValue double
	 * @return JunSimpleSlider's value
	 * @category private
	 */
	protected double computeSliderValueFromNibValue_(double nibValue) {
		int denominator = this.nibCollection().length - 1;
		double value = 0;
		double unit = 0;
		int index = 0;
		if (denominator <= 0) {
			value = this.simpleSlider().doubleValue();
		} else {
			unit = 1.0 / denominator;
			for (int i = 0; i < this.nibCollection().length; i++) {
				if (nibValue == this.nibCollection()[i]) {
					index = i;
					break;
				}
			}
			value = unit * Math.max(index, 0);
		}
		return value;
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Nib Choice");
	}

	/**
	 * Answer a view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunNibChoiceView && ((JunNibChoiceView) each).model() == this) {
				return (JunNibChoiceView) each;
			}
		}
		return null;
	}
}
