package jp.co.sra.jun.goodies.multimedia;

import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.movie.framework.JunMovieHandle;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMultiMediaViewer class
 * 
 *  @author    nisinaka
 *  @created   2003/03/18 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun528 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMultiMediaViewer.java,v 8.11 2008/02/20 06:31:50 nisinaka Exp $
 */
public class JunMultiMediaViewer extends JunApplicationModel {

	protected JunFileModel fileModel;
	protected JunMultiMediaImageModel imageModel;
	protected JunMultiMediaMovieModel movieModel;
	protected JunMultiMediaTextModel textModel;
	protected JunMultiMediaExtraModel extraModel;
	protected boolean isMoviePlay;

	/**
	 * Initialize the JunMultiMediaViewer when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		fileModel = null;
		imageModel = null;
		movieModel = null;
		textModel = null;
		extraModel = null;
		isMoviePlay = false;
	}

	/**
	 * Release the resources of the JunMultiMediaViewer.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.fileModel().release();
		this.movieModel().release();
		this.imageModel().release();
		this.textModel().release();
		this.extraModel().release();

		super.release();
	}

	/**
	 * Answer my file model.
	 *
	 * @return jp.co.sra.jun.goodies.files.JunFileModel
	 * @category accessing
	 */
	public JunFileModel fileModel() {
		if (fileModel == null) {
			Set aSet = new HashSet();
			String[] patterns = JunSystem.DefaultImageExtensionPatterns();
			for (int i = 0; i < patterns.length; i++) {
				aSet.add(patterns[i]);
			}
			patterns = JunSystem.DefaultMovieExtensionPatterns();
			for (int i = 0; i < patterns.length; i++) {
				aSet.add(patterns[i]);
			}
			patterns = JunSystem.DefaultSoundExtensionPatterns();
			for (int i = 0; i < patterns.length; i++) {
				aSet.add(patterns[i]);
			}
			patterns = JunSystem.DefaultTextExtensionPatterns();
			for (int i = 0; i < patterns.length; i++) {
				aSet.add(patterns[i]);
			}
			patterns = JunSystem.DefaultExtraExtensionPatterns();
			for (int i = 0; i < patterns.length; i++) {
				aSet.add(patterns[i]);
			}
			String[] extensionPatterns = (String[]) aSet.toArray(new String[aSet.size()]);
			JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("Multi-media files"), extensionPatterns), JunFileModel.FileType.All(JunSystem.$String("All files")) };
			fileModel = new JunFileModel();
			fileModel.fileTypes_(fileTypes);
			fileModel.compute_(new StBlockClosure() {
				public Object value_(Object value) {
					checkFile_((File) value);
					return null;
				}
			});
		}
		return fileModel;
	}

	/**
	 * Answer my image model.
	 *
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaImageModel
	 * @category accessing
	 */
	public JunMultiMediaImageModel imageModel() {
		if (imageModel == null) {
			imageModel = new JunMultiMediaImageModel();
		}
		return imageModel;
	}

	/**
	 * Answer my movie model.
	 *
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaMovieModel
	 * @category accessing
	 */
	public JunMultiMediaMovieModel movieModel() {
		if (movieModel == null) {
			movieModel = new JunMultiMediaMovieModel();
			movieModel.trackerModel().loop();
		}
		return movieModel;
	}

	/**
	 * Answer my text model.
	 *
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaTextModel
	 * @category accessing
	 */
	public JunMultiMediaTextModel textModel() {
		if (textModel == null) {
			textModel = new JunMultiMediaTextModel();
		}
		return textModel;
	}

	/**
	 * Answer my extra model.
	 *
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaExtraModel
	 * @category accessing
	 */
	public JunMultiMediaExtraModel extraModel() {
		if (extraModel == null) {
			extraModel = new JunMultiMediaExtraModel();
		}
		return extraModel;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunMultiMediaViewerViewAwt(this);
		} else {
			return new JunMultiMediaViewerViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Multi-Media Viewer");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose()
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		this.fileModel().closeRequest();
		this.movieModel().closeRequest();
		this.imageModel().closeRequest();
		this.textModel().closeRequest();
		this.extraModel().closeRequest();
		this.release();
		super.noticeOfWindowClose(e);
	}

	/**
	 * Check the file.
	 *
	 * @param aFile java.io.File
	 * @category private
	 */
	protected void checkFile_(File aFile) {
		isMoviePlay = this.movieModel().isPlay();

		this.imageModel().beInvisible();
		this.movieModel().beInvisible();
		this.textModel().beInvisible();
		this.extraModel().beInvisible();

		if (aFile == null) {
			return;
		}
		if (aFile.exists() == false) {
			return;
		}
		if (aFile.isDirectory()) {
			return;
		}

		String filename = aFile.getName();
		int index = filename.lastIndexOf('.');
		String extensionString = (index < 0) ? "" : filename.substring(index + 1).toLowerCase();

		if (Arrays.asList(JunSystem.DefaultMovieExtensions()).contains(extensionString) || Arrays.asList(JunSystem.DefaultSoundExtensions()).contains(extensionString)) {
			// Movie and Sound
			JunMovieHandle aMovieHandle = JunMovieHandle.Filename_(aFile);
			if (aMovieHandle == null) {
				return;
			}

			this.movieModel().movieHandle_(aMovieHandle);
			this.movieModel().normalSize();
			this.movieModel().beVisible();
			this.movieModel().goto_(this.movieModel().now());
			if (this.movieModel().isIntervalNotEmpty()) {
				this.movieModel().trackerModel().firstMarker_(this.movieModel().trackerModel().firstMarker());
				this.movieModel().trackerModel().lastMarker_(this.movieModel().trackerModel().lastMarker());
			}
			if (isMoviePlay) {
				this.movieModel().start();
			}
		} else if (Arrays.asList(JunSystem.DefaultImageExtensions()).contains(extensionString)) {
			// Image
			this.imageModel().visual_((new JunMultiMediaImageModel(aFile)).visual());
			this.imageModel().beVisible();
		} else if (Arrays.asList(JunSystem.DefaultTextExtensions()).contains(extensionString)) {
			// Text
			this.textModel().textHolder().value_((new JunMultiMediaTextModel(aFile)).textHolder().value());
			this.textModel().beVisible();
		} else if (StInputState.Default().altDown()) {
			// Unknown
			Object aString;
			int limitSize = 1024 * 1024 * 4;
			long fileSize = aFile.length();
			if (fileSize > limitSize && JunDialog.Confirm_($String("The file is large: <1p> bytes.", null, new Long(fileSize)) + "\n" + $String("Would you like to use dump on the head?"))) {
				aString = (new JunMultiMediaTextModel(aFile, limitSize)).textHolder().value();
			} else {
				aString = (new JunMultiMediaTextModel(aFile)).textHolder().value();
			}
			this.textModel().textHolder().value_(aString);
			this.textModel().beVisible();
		} else {
			this.extraModel().file_(aFile);
			this.extraModel().beVisible();
		}
	}

}
