package jp.co.sra.jun.goodies.movie.support;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemInterface;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.progress.JunProgress;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunScreenRecorder class
 * 
 *  @author    Hoshi Takanori
 *  @created   2002/10/18 (by Hoshi Takanori)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun448 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunScreenRecorder.java,v 8.13 2008/02/20 06:31:50 nisinaka Exp $
 */
public class JunScreenRecorder extends JunApplicationModel implements Runnable {

	protected File file;
	protected Rectangle movieArea;
	protected Dimension movieExtent;
	protected int watchTick = 1000;
	protected boolean watchState = false;
	protected volatile Thread watchThread = null;
	protected File imagesFile;
	protected int imagesCount;
	protected volatile boolean watchDone;

	/**
	 * Answer movie area.
	 * 
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public Rectangle getMovieArea() {
		if (movieArea == null) {
			movieArea = new Rectangle(Toolkit.getDefaultToolkit().getScreenSize());
		}
		return movieArea;
	}

	/**
	 * Set movie area.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @category accessing
	 */
	public void setMovieArea(Rectangle aRectangle) {
		movieArea = aRectangle;
		this.changed_($("movieArea"));

		if (movieExtent != null) {
			this.setMovieExtent(new Dimension(movieExtent.width, movieArea.height * movieExtent.width / movieArea.width));
		}
	}

	/**
	 * Answer movie extent.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension getMovieExtent() {
		if (movieExtent == null) {
			movieExtent = new Dimension(320, 240);
		}
		return movieExtent;
	}

	/**
	 * Set movie extent.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @category accessing
	 */
	public void setMovieExtent(Dimension aDimension) {
		movieExtent = aDimension;
		this.changed_($("movieExtent"));
	}

	/**
	 * Answer watch tick.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int getWatchTick() {
		return watchTick;
	}

	/**
	 * Set watch tick.
	 * 
	 * @param milliseconds int
	 * @category accessing
	 */
	public void setWatchTick(int milliseconds) {
		watchTick = milliseconds;
		this.changed_($("watchTick"));
	}

	/**
	 * Answer watch State.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean getWatchState() {
		return watchState;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunScreenRecorderViewAwt(this);
		} else {
			return new JunScreenRecorderViewSwing(this);
		}
	}

	/**
	 * Create an "Area" popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu _createAreaPopupMenu() {
		StPopupMenu aPopupMenu = new StPopupMenu();
		aPopupMenu.add(new StMenuItem($String("From user..."), new MenuPerformer(this, "areaFromYou")));
		aPopupMenu.addSeparator();
		aPopupMenu.add(new StMenuItem($String("Whole screen"), new MenuPerformer(this, "areaWholeScreen")));
		return aPopupMenu;
	}

	/**
	 * Create a "Size" popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu _createSizePopupMenu() {
		StPopupMenu aPopupMenu = new StPopupMenu();
		aPopupMenu.add(new StMenuItem("320@240 (half)", new MenuPerformer(this, "sizeHalf")));
		aPopupMenu.add(new StMenuItem("160@120 (quart)", new MenuPerformer(this, "sizeQuart")));
		aPopupMenu.add(new StMenuItem("640@480 (full)", new MenuPerformer(this, "sizeFull")));
		aPopupMenu.addSeparator();
		aPopupMenu.add(new StMenuItem("area (as is)", new MenuPerformer(this, "sizeAsIs")));
		aPopupMenu.add(new StMenuItem("area / 2", new MenuPerformer(this, "sizeDiv2")));
		aPopupMenu.add(new StMenuItem("area / 4", new MenuPerformer(this, "sizeDiv4")));
		aPopupMenu.add(new StMenuItem("area / 8", new MenuPerformer(this, "sizeDiv8")));
		return aPopupMenu;
	}

	/**
	 * Create a "Speed" popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu _createSpeedPopupMenu() {
		StPopupMenu aPopupMenu = new StPopupMenu();
		aPopupMenu.add(new StMenuItem("100 msec. (10 frames per sec.)", new MenuPerformer(this, "speed100msec")));
		aPopupMenu.add(new StMenuItem("200 msec. (5 frames per sec.)", new MenuPerformer(this, "speed200msec")));
		aPopupMenu.add(new StMenuItem("500 msec. (2 frames per sec.)", new MenuPerformer(this, "speed500msec")));
		aPopupMenu.add(new StMenuItem("1 sec. (1 frame per sec.)", new MenuPerformer(this, "speed1sec")));
		aPopupMenu.add(new StMenuItem("2 sec. (30 frames per min.)", new MenuPerformer(this, "speed2sec")));
		aPopupMenu.add(new StMenuItem("5 sec. (12 frames per min.)", new MenuPerformer(this, "speed5sec")));
		aPopupMenu.add(new StMenuItem("10 sec. (6 frames per min.)", new MenuPerformer(this, "speed10sec")));
		aPopupMenu.add(new StMenuItem("15 sec. (4 frames per min.)", new MenuPerformer(this, "speed15sec")));
		aPopupMenu.add(new StMenuItem("20 sec. (3 frames per min.)", new MenuPerformer(this, "speed20sec")));
		aPopupMenu.add(new StMenuItem("30 sec. (2 frames per min.)", new MenuPerformer(this, "speed30sec")));
		aPopupMenu.add(new StMenuItem("1 min. (1 frame per min.)", new MenuPerformer(this, "speed1min")));
		return aPopupMenu;
	}

	/**
	 * Return record/stop button label.
	 * 
	 * @return java.lang.String
	 * @category buttons
	 */
	public String getRecordButtonLabel() {
		return watchState == false ? $String("Record") : $String("Stop");
	}

	/**
	 * Return area button label.
	 * 
	 * @return java.lang.String
	 * @category buttons
	 */
	public String getAreaButtonLabel() {
		Rectangle rect = this.getMovieArea();
		return $String("Snap Area") + ": (" + rect.x + "@" + rect.y + " corner: " + (rect.x + rect.width) + "@" + (rect.y + rect.height) + ")";
	}

	/**
	 * Return size button label.
	 * 
	 * @return java.lang.String
	 * @category buttons
	 */
	public String getSizeButtonLabel() {
		Dimension size = this.getMovieExtent();
		return $String("Snap Size") + ": (" + size.width + "@" + size.height + ")";
	}

	/**
	 * Return speed button label.
	 * 
	 * @return java.lang.String
	 * @category buttons
	 */
	public String getSpeedButtonLabel() {
		String label = $String("Recording Speed") + ": ";
		if (watchTick < 1000) {
			return label + watchTick + " msec.";
		} else if (watchTick < 60000) {
			return label + (watchTick / 1000) + " sec.";
		} else {
			return label + (watchTick / 60000) + " min.";
		}
	}

	/**
	 * Set area to the whole screen.
	 * 
	 * @category accessing
	 */
	public void areaWholeScreen() {
		this.setMovieArea(new Rectangle(Toolkit.getDefaultToolkit().getScreenSize()));
	}

	/**
	 * Set area from user.
	 * 
	 * @category accessing
	 */
	public void areaFromYou() {
		Rectangle rect = SystemInterface._RectangleFromUser();

		if (rect != null) {
			this.setMovieArea(rect);
		}
	}

	/**
	 * Set size to "half" (320 x 240).
	 * 
	 * @category accessing
	 */
	public void sizeHalf() {
		this.setMovieExtent(new Dimension(320, 240));
	}

	/**
	 * Set size to "quart" (160 x 120).
	 * 
	 * @category accessing
	 */
	public void sizeQuart() {
		this.setMovieExtent(new Dimension(160, 120));
	}

	/**
	 * Set size to "full" (640 x 480).
	 * 
	 * @category accessing
	 */
	public void sizeFull() {
		this.setMovieExtent(new Dimension(640, 480));
	}

	/**
	 * Set size to area.
	 * 
	 * @category accessing
	 */
	public void sizeAsIs() {
		this.setMovieExtent(getMovieArea().getSize());
	}

	/**
	 * Set size to area / 2.
	 * 
	 * @category accessing
	 */
	public void sizeDiv2() {
		Dimension size = getMovieArea().getSize();
		this.setMovieExtent(new Dimension(size.width / 2, size.height / 2));
	}

	/**
	 * Set size to area / 4.
	 * 
	 * @category accessing
	 */
	public void sizeDiv4() {
		Dimension size = getMovieArea().getSize();
		this.setMovieExtent(new Dimension(size.width / 4, size.height / 4));
	}

	/**
	 * Set size to area / 8.
	 * 
	 * @category accessing
	 */
	public void sizeDiv8() {
		Dimension size = getMovieArea().getSize();
		this.setMovieExtent(new Dimension(size.width / 8, size.height / 8));
	}

	/**
	 * Set speed to 100 msec.
	 * 
	 * @category accessing
	 */
	public void speed100msec() {
		this.setWatchTick(100);
	}

	/**
	 * Set speed to 200 msec.
	 * 
	 * @category accessing
	 */
	public void speed200msec() {
		this.setWatchTick(200);
	}

	/**
	 * Set speed to 500 msec.
	 * 
	 * @category accessing
	 */
	public void speed500msec() {
		this.setWatchTick(500);
	}

	/**
	 * Set speed to 1 sec.
	 * 
	 * @category accessing
	 */
	public void speed1sec() {
		this.setWatchTick(1000);
	}

	/**
	 * Set speed to 2 sec.
	 * 
	 * @category accessing
	 */
	public void speed2sec() {
		this.setWatchTick(2000);
	}

	/**
	 * Set speed to 5 sec.
	 * 
	 * @category accessing
	 */
	public void speed5sec() {
		this.setWatchTick(5000);
	}

	/**
	 * Set speed to 10 sec.
	 * 
	 * @category accessing
	 */
	public void speed10sec() {
		this.setWatchTick(10000);
	}

	/**
	 * Set speed to 15 sec.
	 * 
	 * @category accessing
	 */
	public void speed15sec() {
		this.setWatchTick(15000);
	}

	/**
	 * Set speed to 20 sec.
	 * 
	 * @category accessing
	 */
	public void speed20sec() {
		this.setWatchTick(20000);
	}

	/**
	 * Set speed to 30 sec.
	 * 
	 * @category accessing
	 */
	public void speed30sec() {
		this.setWatchTick(30000);
	}

	/**
	 * Set speed to 1 min.
	 * 
	 * @category accessing
	 */
	public void speed1min() {
		this.setWatchTick(60000);
	}

	/**
	 * Runner.
	 * 
	 * @see java.lang.Runnable#run()
	 * @category processing
	 */
	public void run() {
		Thread thisThread = Thread.currentThread();
		ObjectOutputStream stream;

		try {
			imagesFile = File.createTempFile("images", ".tmp");
			stream = new ObjectOutputStream(new FileOutputStream(imagesFile));
		} catch (Exception e) {
			e.printStackTrace();
			return;
		}

		watchDone = false;
		imagesCount = 0;

		while (watchThread == thisThread) {
			int[] pixels;

			SystemInterface StPLInterface = SystemInterface.Current();
			synchronized (StPLInterface) {
				Rectangle rect = this.getMovieArea();
				pixels = StPLInterface.utilImageOfArea(rect.x, rect.y, rect.width, rect.height);
			}

			try {
				stream.writeObject(pixels);
				stream.reset();
			} catch (IOException e) {
				e.printStackTrace();
				watchDone = true;
				return;
			}

			++imagesCount;

			try {
				Thread.sleep(watchTick);
			} catch (InterruptedException e) {
			}
		}

		try {
			stream.writeObject(null);
			stream.close();
		} catch (IOException e) {
			e.printStackTrace();
		}

		watchDone = true;
	}

	/**
	 * Make movie file.
	 * @throws java.lang.Exception
	 * @category movie making
	 */
	public void makeMovieFile() throws Exception {
		(new JunProgress()).doWithStopButton_(new StBlockClosure() {
			public Object value_(Object obj) {
				final JunProgress progress = (JunProgress) obj;
				progress.message_($String("make movie..."));

				(new JunImagesToMovie(file, JunScreenRecorder.this.getMovieExtent())).do_(new StBlockClosure() {
					public Object value_(Object obj) {
						JunImagesToMovie imagesToMovie = (JunImagesToMovie) obj;

						ObjectInputStream stream = null;
						int[] pixels;
						int i = 0;
						try {
							stream = new ObjectInputStream(new FileInputStream(imagesFile));
							while ((pixels = (int[]) stream.readObject()) != null) {
								Dimension size = getMovieArea().getSize();
								imagesToMovie.addPixels_width_height_milliseconds_(pixels, size.width, size.height, watchTick);
								progress.value_((float) ++i / imagesCount);
							}
						} catch (IOException e) {
							e.printStackTrace();
						} catch (ClassNotFoundException e) {
							e.printStackTrace();
						} catch (RuntimeException e) {
							// stop button is pressed.
						} finally {
							if (stream != null) {
								try {
									stream.close();
								} catch (IOException e) {
								}
							}
						}

						return null;
					}
				});

				return null;
			}
		});

		imagesFile.delete();
		imagesFile = null;
	}

	/**
	 * Process record/stop button.
	 * 
	 * @category buttons
	 */
	public void recordButtonPressed() {
		if (watchState == false && watchThread == null) {
			watchState = true;
			this.changed_($("watchState"));

			JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, $String("Movie")), new String[] { "*.mov", "*.MOV" }) };
			file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "MOV"), new File(this.defaultBaseName() + ".mov"), fileTypes, fileTypes[0]);
			if (file != null) {
				watchThread = new Thread(this);
				watchThread.start();
			} else {
				watchState = false;
				this.changed_($("watchState"));
			}
		} else if (watchState && watchThread != null) {
			Thread thread = watchThread;
			watchThread = null;
			thread.interrupt();

			while (!watchDone) {
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
				}
			}

			try {
				makeMovieFile();
			} catch (Exception e) {
				e.printStackTrace();
			}

			watchState = false;
			this.changed_($("watchState"));
		}
	}

	/**
	 * Process snap button.
	 * 
	 * @category buttons
	 */
	public void snapButtonPressed() {
		StImage image = StImage._OfArea(this.getMovieArea());

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "JPEG"), new String[] { "*.jpg", "*.JPG" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "JPEG"), new File(this.defaultBaseName() + ".jpg"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			OutputStream stream = new FileOutputStream(file);
			JunImageStream imageStream = JunJpegImageStream.On_(stream);
			imageStream.nextPutImage_(image);
			imageStream.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Screen Recorder");
	}

}