package jp.co.sra.jun.goodies.movie.support;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;

import jp.co.sra.qt4jun.JunQTInterface;

import jp.co.sra.jun.goodies.animation.JunCartoonMovie;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.image.support.JunImageAdjuster;
import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunImagesToMovie class
 * 
 *  @author    Hoshi Takanori
 *  @created   2002/10/24 (by Hoshi Takanori)
 *  @updated   2004/02/10 (by Nobuto Matsubara)
 *  @updated   2006/12/06 (by m-asada)
 *  @updated   2007/11/07 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImagesToMovie.java,v 8.15 2008/02/20 06:31:50 nisinaka Exp $
 */
public class JunImagesToMovie extends JunAbstractObject {

	protected static final int TIME_SCALE = JunQTInterface.kVideoTimeScale;

	protected File file;
	protected Dimension extent;
	protected Color background;
	protected int movie;
	protected int track;
	protected int media;

	/**
	 * Create a new instance of JunImagesToMovie and evaluate the block.
	 * 
	 * @param aFile java.io.File
	 * @param aDimension java.awt.Dimension
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.goodies.movie.support.JunImagesToMovie
	 * @category Instance creation
	 */
	public static JunImagesToMovie File_extent_do_(File aFile, Dimension aDimension, StBlockClosure aBlock) {
		JunImagesToMovie imagesToMovie = new JunImagesToMovie(aFile, aDimension);
		imagesToMovie.do_(aBlock);
		return imagesToMovie;
	}

	/**
	 * Create a new instance of <code>JunImagesToMovie</code> and initialize it.
	 *
	 * @param aFile java.io.File
	 * @param aDimension java.awt.Dimension
	 * @category Instance creation
	 */
	public JunImagesToMovie(File aFile, Dimension aDimension) {
		super();
		this.file_(aFile);
		this.extent_(aDimension);
	}

	/**
	 * Create a new instance of <code>JunImagesToMovie</code> and initialize it.
	 *
	 * @param filenameString java.lang.String
	 * @param aDimension java.awt.Dimension
	 * @category Instance creation
	 */
	public JunImagesToMovie(String filenameString, Dimension aDimension) {
		this(new File(filenameString), aDimension);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		file = null;
		extent = null;
		background = null;
		movie = Integer.MIN_VALUE;
		track = Integer.MIN_VALUE;
		media = Integer.MIN_VALUE;
	}

	/**
	 * Answer background color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color background() {
		if (background == null) {
			background = Color.white;
		}
		return background;
	}

	/**
	 * Set background color.
	 * 
	 * @param color java.awt.Color
	 * @category accessing
	 */
	public void background_(Color color) {
		background = color;
	}

	/**
	 * Answer the receiver's extent.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension extent() {
		return extent;
	}

	/**
	 * Set the receiver's extent.
	 * 
	 * @param anExtent java.awt.Dimension
	 * @category accessing
	 */
	public void extent_(Dimension anExtent) {
		extent = anExtent;
	}

	/**
	 * Answer my current file
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File file() {
		return file;
	}

	/**
	 * Set my new file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void file_(File aFile) {
		file = aFile;
	}

	/**
	 * Answer my current filename.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String filename() {
		return (file == null) ? null : file.getAbsolutePath();
	}

	/**
	 * Add an image.
	 *
	 * @param image jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adding
	 */
	public StImage add_(StImage image) {
		return this.add_milliseconds_(image, 33);
	}

	/**
	 * Add an image with duration.
	 *
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param durationTime int
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adding
	 */
	public StImage add_duration_(StImage anImage, int durationTime) {
		if (extent == null) {
			return null;
		}
		if (media <= 0) {
			return null;
		}
		StImage theImage = this.adjustedImage_(anImage);
		this.addPixels_width_height_duration_(theImage.getPixels(), theImage.width(), theImage.height(), durationTime);
		return theImage;
	}

	/**
	 * Add an image with milliseconds.
	 *
	 * @param image jp.co.sra.smalltalk.StImage
	 * @param milliseconds int
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adding
	 */
	public StImage add_milliseconds_(StImage image, int milliseconds) {
		return this.add_duration_(image, milliseconds * TIME_SCALE / 1000);
	}

	/**
	 * Add an image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adding
	 */
	public StImage addImage_(StImage anImage) {
		return this.add_(anImage);
	}

	/**
	 * Add an image with the specified frame rate.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aNumber int
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adding
	 */
	public StImage addImage_framesPerSecond_(StImage anImage, int aNumber) {
		return this.addImage_keepTime_(anImage, 1000 / Math.min(Math.max(1, aNumber), 1000));
	}

	/**
	 * Add an image with the specified keep time.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param millisecondClockValue int
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adding
	 */
	public StImage addImage_keepTime_(StImage anImage, int millisecondClockValue) {
		return this.add_milliseconds_(anImage, millisecondClockValue);
	}

	/**
	 * Add pixels as an image with duration.
	 *
	 * @param pixels int[]
	 * @param width int
	 * @param height int
	 * @param duration int
	 * @category adding
	 */
	public void addPixels_width_height_duration_(int[] pixels, int width, int height, int duration) {
		JunQTInterface interface_ = JunQTInterface.Interface();
		interface_.qtAddMediaSampleImage(media, pixels, width, height, extent.width, extent.height, interface_.codecNormalQuality, interface_.kAnimationCodecType, duration);
	}

	/**
	 * Add pixels as an image with milliseconds.
	 *
	 * @param pixels int[]
	 * @param width int
	 * @param height int
	 * @param milliseconds int
	 * @category adding
	 */
	public void addPixels_width_height_milliseconds_(int[] pixels, int width, int height, int milliseconds) {
		this.addPixels_width_height_duration_(pixels, width, height, milliseconds * TIME_SCALE / 1000);
	}

	/**
	 * Answer an adjusted image.
	 *
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	public StImage adjustedImage_(StImage anImage) {
		if (extent == null) {
			return anImage;
		}

		StImage targetImage = anImage;
		StImage adjustedImage = null;
		if (targetImage.bounds().getSize().equals(this.extent())) {
			adjustedImage = targetImage;
		} else {
			Rectangle aBox = new Rectangle(0, 0, this.extent().width, this.extent().height);
			targetImage = JunImageAdjuster.AdjustImage_extent_keepAspect_(targetImage, aBox.getSize(), true);
			adjustedImage = new StImage(aBox.getSize());
			JunImageProcessor.Fill_color_(adjustedImage, this.background());
			Rectangle destRectangle = targetImage.bounds();
			destRectangle.translate((aBox.width - targetImage.bounds().width) / 2, (aBox.height - targetImage.bounds().height) / 2);
			adjustedImage.copy_from_in_rule_(destRectangle, new Point(0, 0), targetImage, StImage.Over);
		}
		return adjustedImage;
	}

	/**
	 * Do the job.
	 *
	 * @param doBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void do_(StBlockClosure doBlock) {
		JunQTInterface interface_ = JunQTInterface.Interface();
		int err;
		int[] array = interface_.qtCreateMovieFile(this.filename(), interface_.sigMoviePlayer, interface_.smSystemScript, interface_.createMovieFileDeleteCurFile + interface_.createMovieFileDontCreateResFile);
		if (array != null && array[0] == interface_.noErr) {
			int resRefNum = array[1];
			movie = array[2];
			track = interface_.qtNewMovieTrack(movie, interface_.qtFixRatio(extent.width, 1), interface_.qtFixRatio(extent.height, 1), interface_.kNoVolume);

			if (interface_.qtGetMoviesError() == interface_.noErr) {
				media = interface_.qtNewTrackMedia(track, interface_.VideoMediaType, TIME_SCALE, 0, 0);

				if (interface_.qtGetMoviesError() == interface_.noErr) {
					err = interface_.qtBeginMediaEdits(media);
					if (err == interface_.noErr) {
						if (doBlock.numArgs() == 0) {
							doBlock.value();
						} else {
							doBlock.value_(this);
						}

						err = interface_.qtEndMediaEdits(media);
						if (err == interface_.noErr) {
							int duration = interface_.qtGetMediaDuration(media);
							err = interface_.qtInsertMediaIntoTrack(track, 0, 0, duration, interface_.fixed1);
						}
					}
				}
			}

			err = interface_.qtAddMovieResource(movie, resRefNum, interface_.movieInDataForkResID, null);
			err = interface_.qtCloseMovieFile(resRefNum);
			interface_.qtDisposeMovie(movie);
		}
	}

	/**
	 * Create a JunCartoonMovie and open it.
	 * 
	 * @param aDirectory java.io.File
	 * @param aString java.lang.String
	 * @param aNumber int
	 * @return jp.co.sra.jun.goodies.animation.JunCartoonMovie
	 * @throws java.io.IOException
	 * @category Utilities
	 */
	public static JunCartoonMovie ImageSequenceDirectory_jpegImageFileNamePattern_framesPerSecond_(File aDirectory, String aString, int aNumber) throws IOException {
		JunCartoonMovie cartoonMovie = new JunCartoonMovie(aDirectory, aString, aNumber);
		cartoonMovie.open();
		return cartoonMovie;
	}

	/**
	 * Create a movie file.
	 * 
	 * @param aDirectory java.io.File
	 * @param aString java.lang.String
	 * @param aNumber int
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category Utilities
	 */
	public static void ImageSequenceDirectory_jpegImageFileNamePattern_framesPerSecond_movieFile_(File aDirectory, final String aString, final int aNumber, File aFile) throws IOException {
		if (aDirectory.exists() == false) {
			throw new IOException(aDirectory.getPath() + " does not exist.");
		}
		if (aDirectory.isDirectory() == false) {
			throw new IOException(aDirectory.getPath() + " is not a directory.");
		}

		final File[] files = aDirectory.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return JunStringUtility.StringMatch_and_(name, aString);
			}
		});
		if (files == null || files.length == 0) {
			throw new IOException("No image in " + aDirectory.getPath());
		}

		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			StImage anImage = null;
			JunImageStream aStream = null;
			try {
				aStream = JunJpegImageStream.On_(new FileInputStream(files[0]));
				anImage = aStream.nextImage();
			} finally {
				if (aStream != null) {
					aStream.close();
				}
			}

			final JunImagesToMovie imagesToMovie = new JunImagesToMovie(aFile, new Dimension(anImage.width(), anImage.height()));
			imagesToMovie.addImage_framesPerSecond_(anImage, aNumber);
			imagesToMovie.do_(new StBlockClosure() {
				public Object value() {
					for (int i = 1; i < files.length; i++) {
						JunImageStream aStream = null;
						try {
							aStream = JunJpegImageStream.On_(new FileInputStream(files[i]));
							imagesToMovie.addImage_framesPerSecond_(aStream.nextImage(), aNumber);
						} catch (IOException e) {
							throw new RuntimeException(e);
						} finally {
							if (aStream != null) {
								try {
									aStream.close();
								} catch (IOException e) {
									throw new RuntimeException(e);
								}
							}
						}
					}

					return null;
				}
			});

		} catch (RuntimeException e) {
			if (e.getCause() instanceof IOException) {
				throw (IOException) e.getCause();
			} else {
				throw e;
			}
		} finally {
			cursor._restore();
		}
	}
}
