package jp.co.sra.jun.goodies.lisp;

import java.io.StringWriter;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StNumber;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.system.framework.JunAbstractObject;
import jp.co.sra.jun.system.support.JunSmallCompiler;

/**
 * JunLispScanner class
 * 
 *  @author    ASTI Beijing
 *  @created   1998/12/09 (by ASTI Beijing)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunLispScanner.java,v 8.11 2008/02/20 06:31:49 nisinaka Exp $
 */
public class JunLispScanner extends JunAbstractObject {

	protected static final char CR = '\r';
	protected static final char LF = '\n';

	protected StReadStream source;
	protected int mark;
	protected Object token;
	protected StSymbol tokenType;
	protected StBlockClosure failBlock;
	protected JunLispScannerTable typeTable;
	protected JunSmallCompiler compiler;

	/**
	 * Create a new instance of JunLispScanner.
	 * 
	 * @category Instance creation
	 */
	public JunLispScanner() {
		super();
		this.initScanner();
	}

	/**
	 * Shorten a string by ellipsis if too long.
	 * 
	 * @return java.lang.String
	 * @param aString java.lang.String
	 * @param count int
	 * @category Utilities
	 */
	public static String ContractTo(String aString, int count) {
		int length = aString.length();
		if (length <= count) {
			return aString;
		}

		int half = (count / 2) - 1;
		return aString.substring(0, half) + "..." + aString.substring(length - count + half + 3);
	}

	/**
	 * Get a binary token.
	 * 
	 * @category x
	 */
	public void xBinary() {
		failBlock.value_("Syntax error" + source.peek());
	}

	/**
	 * Get a token as a Smalltalk object.
	 * 
	 * @category x
	 */
	public void xBrace() {
		tokenType = $("object");
		token = this.objectFrom_(source);
	}

	/**
	 * Get a comment token.
	 * 
	 * @category x
	 */
	public void xComment() {
		char ch;
		while ((ch = this.nextChar()) != CR) {
			if (ch == '!') {
				break;
			}
		}
		this.nextToken();
	}

	/**
	 * Get a number token.
	 * 
	 * @category x
	 */
	public void xDigit() {
		tokenType = $("number");
		token = this.numberFrom_(source);
	}

	/**
	 * Get a string token.
	 * 
	 * @category x
	 */
	public void xDoubleQuote() {
		tokenType = $("string");
		token = this.stringFrom_(source);
	}

	/**
	 * Get a signed token.
	 * 
	 * @category x
	 */
	public void xSign() {
		char sign = nextChar();
		char ch = peekChar();
		if (Character.isDigit(ch)) {
			tokenType = $("number");
			token = this.numberFrom_(source);
			if (sign == '-') {
				Number number = (Number) token;

				// I'm so sad to do this. :-(
				if (number instanceof Integer) {
					token = new Integer(-number.intValue());
					return;
				} else if (number instanceof Float) {
					token = new Float(-number.floatValue());
					return;
				} else if (number instanceof Double) {
					token = new Double(-number.doubleValue());
					return;
				} else if (number instanceof Long) {
					token = new Long(-number.longValue());
					return;
				}
			}
		} else {
			this.unNextChar();
			tokenType = $("symbol");
			token = this.symbolFrom_(source);
		}
	}

	/**
	 * Get a symbol token.
	 * 
	 * @category x
	 */
	public void xSymbol() {
		tokenType = $("symbol");
		token = this.symbolFrom_(source);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initScanner() {
		typeTable = JunLispScannerTable.ScannerTable();
		failBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				String errorMessage = (String) anObject;
				String label = errorMessage + " near " + ContractTo(token.toString(), 10);
				String string = source.upToEnd();
				if (string.length() == 0) {
					string = "--> end of file";
				} else {
					string = "--> " + ContractTo(string, 30);
				}
				throw SmalltalkException.Error(label + "\n" + string);
			}
		};

		// There is no other compiler available.
		compiler = new JunSmallCompiler();
	}

	/**
	 * Get a next token with multiple characters.
	 * 
	 * @param type java.lang.String
	 * @throws jp.co.sra.smalltalk.SmalltalkException 
	 * @category scanning
	 */
	protected void multiChar_(String type) {
		try {
			this.perform_(type);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Get the next character from the source.
	 * 
	 * @return char
	 * @category private
	 */
	protected char nextChar() {
		char c = source.next();
		if (c == LF) {
			return CR;
		} else if (c == CR) {
			if (source.peek() == LF) {
				source.next();
			}
			return CR;
		} else if (c == 0) {
			return '!';
		}
		return c;
	}

	/**
	 * Get a next token.
	 * 
	 * @return java.lang.Object
	 * @category scanning
	 */
	protected Object nextToken() {
		mark = source.position();
		char ch = this.peekChar();
		tokenType = typeTable.at_((int) ch);
		while (tokenType == ($("xDelimiter"))) {
			this.nextChar();
			ch = this.peekChar();
			tokenType = typeTable.at_((int) ch);
		}

		String tokenTypeString = tokenType.toString();
		if (tokenTypeString.charAt(0) == 'x') {
			this.multiChar_(tokenTypeString);
		} else {
			this.singleChar_(tokenTypeString);
		}

		return token;
	}

	/**
	 * Read a number from the stream.
	 * 
	 * @return java.lang.Number
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading
	 */
	protected Number numberFrom_(StReadStream aStream) {
		return StNumber.ReadFrom_(aStream);
	}

	/**
	 * Read a Smalltalk object from the stream.
	 * 
	 * @return java.lang.Object
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading 
	 */
	protected Object objectFrom_(StReadStream aStream) {
		StringWriter buffer = new StringWriter();
		char c = aStream.next();
		while ((c = aStream.next()) != '}') {
			if (aStream.atEnd()) {
				return failBlock.value_("Syntax error unmatched ${");
			}
			buffer.write(c);
		}

		return compiler.evaluate_in_to_notifying_ifFail_(buffer.toString(), null, null, null, new StBlockClosure());
	}

	/**
	 * Initialize for the source stream to scan.
	 * 
	 * @param inputStream jp.co.sra.smalltalk.StReadStream
	 * @category initialize-release
	 */
	protected void on_(StReadStream inputStream) {
		source = inputStream;
		mark = source.position();
	}

	/**
	 * Peek the next character from the source stream.
	 * 
	 * @return char
	 * @category private
	 */
	protected char peekChar() {
		char c = source.peek();
		if (c == LF) {
			return CR;
		} else if (c == CR) {
			return CR;
		} else if (c == 0) {
			return '!';
		}
		return c;
	}

	/**
	 * Get a next token which is just a single character.
	 * 
	 * @param type java.lang.Object
	 * @category scanning
	 */
	protected void singleChar_(Object type) {
		this.nextChar();
		token = type;
	}

	/**
	 * Read a string from the stream.
	 * 
	 * @return java.lang.String
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading 
	 */
	protected String stringFrom_(StReadStream aStream) {
		StringWriter buffer = new StringWriter();
		char c = aStream.next();
		if (c == '"') {
			while ((c = aStream.peek()) != 0) {
				if (c == '"') {
					aStream.next();
					c = aStream.peek();
					if (c != '"') {
						return new String(buffer.toString());
					}
				}
				buffer.write(aStream.next());
			}
		}

		String string = aStream.upToEnd();
		if (string.length() > 100) {
			string = string.substring(0, 99);
		}

		return (String) failBlock.value_("Syntax error unmatched '\"'");
	}

	/**
	 * Read a symbol from the stream.
	 * 
	 * @return java.lang.Object
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading 
	 */
	protected Object symbolFrom_(StReadStream aStream) {
		StringWriter buffer = new StringWriter();
		char c = aStream.peek();
		StSymbol type;
		while ((c != 0) && (((type = typeTable.at_(c)) == $("xSymbol")) || (type == $("xDigit")) || type == $("xSign"))) {
			buffer.write(aStream.next());
			c = aStream.peek();
		}

		String contents = buffer.toString();
		if (contents.equals("nil")) {
			return JunLispNil.NullList();
		}

		return $(contents);
	}

	/**
	 * Push back a character to the source stream.
	 * 
	 * @category private
	 */
	protected void unNextChar() {
		source.skip_(-1);
	}

	/**
	 * Push back a token to the source stream.
	 * 
	 * @category scanning
	 */
	protected void unNextToken() {
		source.position_(mark);
	}
}
