package jp.co.sra.jun.goodies.lisp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.StringWriter;
import java.util.Stack;
import java.util.Vector;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StAssociation;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunLispInterpreter class
 * 
 *  @author    ASTI Beijing
 *  @created   1998/12/09 (by xiaolai)
 *  @updated   2003/05/07 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun525 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunLispInterpreter.java,v 8.11 2008/02/20 06:31:49 nisinaka Exp $
 */
public class JunLispInterpreter extends JunApplicationModel {

	protected JunLispTable lispTable;
	protected Stack bindStack;
	protected StBlockClosure failBlock;
	protected PrintStream textCollector;

	protected StValueHolder textValue;

	/**
	 * Create a new instance of <code>JunLispInterpreter</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunLispInterpreter() {
		super();
		this.initialize();
	}

	/**
	 * Evaluate the Object as a S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param sExpression java.lang.Object
	 * @category Public access 
	 */
	public static Object Evaluate_(Object sExpression) {
		JunLispInterpreter anInterpreter = new JunLispInterpreter();
		StBlockClosure anAction = new StBlockClosure() {
			public Object value_(Object anObject) {
				throw SmalltalkException.Error((String) anObject);
			}
		};
		return anInterpreter.evaluateTopLevel_ifFail_(sExpression, anAction);
	}

	/**
	 * Evaluate the Object as a S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param sExpression java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Public access 
	 */
	public static Object Evaluate_ifFail_(Object sExpression, StBlockClosure aBlock) {
		JunLispInterpreter anInterpreter = new JunLispInterpreter();
		return anInterpreter.evaluateTopLevel_ifFail_(sExpression, aBlock);
	}

	/**
	 * Evaluate the Object as a stream which contains a S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param aStream java.lang.Object
	 * @category Public access 
	 */
	public static Object EvaluateFrom_(Object aStream) {
		JunLispInterpreter anInterpreter = new JunLispInterpreter();
		StBlockClosure anAction = new StBlockClosure() {
			public Object value_(Object anObject) {
				throw SmalltalkException.Error((String) anObject);
			}
		};
		return anInterpreter.evaluateTopLevel_ifFail_((JunLispParser.Parse_(aStream)), anAction);
	}

	/**
	 * Evaluate the Object as a stream which contains a S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param aStream java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Public access 
	 */
	public static Object EvaluateFrom_ifFail_(Object aStream, StBlockClosure aBlock) {
		JunLispInterpreter anInterpreter = new JunLispInterpreter();
		return anInterpreter.evaluateTopLevel_ifFail_((JunLispParser.Parse_(aStream)), aBlock);
	}

	/**
	 * Initialize the class.
	 * 
	 * @category Class initialization
	 */
	public static void Initialize() {
		InitializeFont();
	}

	/**
	 * Initialize the font of the class.
	 * 
	 * @category Class initialization
	 */
	public static void InitializeFont() {
		/*
		 String[] defaultFontName = new String[] { "Courier*", "courier*", };
		 int defaultFontSize = 12;
		 Font deviceFont = new Font("Courier", Font.BOLD, defaultFontSize);
		 */
	}

	/**
	 * Convert the object as a printable string.
	 * 
	 * @return java.lang.String
	 * @param anObject java.lang.Object
	 * @throws SmalltalkException
	 * @category Printing
	 */
	public static String PrintString_(Object anObject) {
		try {
			StringWriter aWriter = new StringWriter();
			if (anObject instanceof JunLispCons) {
				((JunLispCons) anObject).printOn_level_(aWriter, 0);
			} else {
				JunLispCons.Cell().printOn_object_(aWriter, anObject);
			}
			return aWriter.toString();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Bind the value to the symbol.
	 * 
	 * @param symbol java.lang.Object
	 * @param value java.lang.Object
	 * @category shallow binding
	 */
	public void bind_value_(Object symbol, Object value) {
		lispTable.intern_(symbol);
		Object saveValue = lispTable.getprop_key_(symbol, $("apval"));
		bindStack.push(new StAssociation(symbol, saveValue));
		putprop_key_value_(symbol, $("apval"), value);
	}

	/**
	 * Put the bind mark.
	 * 
	 * @category shallow binding
	 */
	public void bindMark() {
		bindStack.push(new StAssociation(null, null));
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunLispInterpreterViewAwt(this);
		} else {
			return new JunLispInterpreterViewSwing(this);
		}
	}

	/**
	 * Menu message: do it.
	 *
	 * @return java.lang.Object
	 * @param anObject java.lang.Object
	 * @category menu messages
	 */
	public Object doIt(Object anObject) {
		JunLispInterpreterView aView = (JunLispInterpreterView) anObject;
		StReadStream stream = new StReadStream(aView.getSelectedText());
		Object result = JunLispNil.NullList();
		while (stream.atEnd() == false) {
			result = this.execute_(stream);
			this.skipSeparators_(stream);
		}
		return result;
	}

	/**
	 * Evaluate the Object as a S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param sExpression java.lang.Object
	 * @category evaluating 
	 */
	public Object evaluate_(Object sExpression) {
		if (sExpression instanceof JunLispList) {
			return listEvaluate_((JunLispList) sExpression);
		}

		if (sExpression instanceof StSymbol) {
			StSymbol aSymbol = (StSymbol) sExpression;
			if (aSymbol == $("t")) {
				return aSymbol;
			}

			lispTable.intern_(sExpression);
			Object apval = lispTable.getprop_key_(sExpression, $("apval"));
			if (apval == null) {
				return this.fatal_(this.printString_(sExpression) + " is unbound atom");
			}

			return apval;
		}

		return sExpression;
	}

	/**
	 * Evaluate an S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param sExpression java.lang.Object
	 * @category evaluating 
	 */
	public Object evaluateTopLevel_(Object sExpression) {
		failBlock = new StBlockClosure() {
			public Object value_(Object errorMessage) {
				textCollector.println(errorMessage);
				return JunLispNil.NullList();
			}
		};

		JunCursors cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();
			return this.evaluate_(sExpression);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Evaluate the Object as a S-Expression.
	 * 
	 * @return java.lang.Object
	 * @param sExpression java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category evaluating 
	 */
	public Object evaluateTopLevel_ifFail_(Object sExpression, StBlockClosure aBlock) {
		failBlock = aBlock;
		return this.evaluate_(sExpression);
	}

	/**
	 * Execute the lisp program in the stream.
	 * 
	 * @return java.lang.Object
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category executing
	 */
	public Object execute_(StReadStream aStream) {
		this.skipSeparators_(aStream);
		int start = aStream.position() + 1;
		Object list = JunLispParser.Parse_(aStream);
		int end = aStream.position();
		String string = aStream.contents().substring(start, end);
		textCollector.println("> " + string);
		Object result = evaluateTopLevel_(list);
		textCollector.println(this.printString_(result));
		return result;
	}

	/**
	 * this function evaluate  a lisp function call statement to an Object in fact.
	 * 
	 * @return java.lang.Object
	 * @param funcBody jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category func eval
	 */
	public Object exprEval_arguList_(JunLispList funcBody, JunLispList arguList) {
		Object result = new Integer(-1);
		JunLispList expression = (JunLispCons) funcBody;
		Object funcName = ((JunLispCons) expression).head();
		expression = (JunLispList) (((JunLispCons) expression).tail());
		expression = (JunLispList) (((JunLispCons) expression).tail());
		JunLispList lvarList = (JunLispList) (((JunLispCons) expression).head());
		expression = (JunLispList) (((JunLispCons) expression).tail());

		if (arguList.length() != lvarList.length()) {
			return fatal_("too few or many arguments " + printString_(arguList) + " vs " + printString_(lvarList) + " for " + funcName.toString());
		}

		bindMark();
		Object tempList = arguList;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object lvar = ((JunLispCons) lvarList).head();
			bind_value_(lvar, evaluate_(each));
			lvarList = (JunLispList) (((JunLispCons) lvarList).tail());
			tempList = ((JunLispCons) tempList).tail();
		}
		tempList = expression;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			result = evaluate_(each);
			tempList = ((JunLispCons) tempList).tail();
		}
		unbind();

		return result;
	}

	/**
	 * Handle fatal error.
	 * 
	 * @return java.lang.Object
	 * @param message String
	 * @category error handling
	 */
	public Object fatal_(String message) {
		while (bindStack.empty()) { //bindStack reverseDo:
			StAssociation assoc = (StAssociation) (bindStack.pop());
			if (assoc.key() != null) {
				if (assoc.value() != null) {
					putprop_key_value_(assoc.key(), $("apval"), assoc.value());
				} else {
					remprop_key_(assoc.key(), $("apval"));
				}
			}
		}
		return failBlock.value_("*** Error: " + message);
	}

	/**
	 * this function evaluate  a lisp function call statement to an Object in fact.
	 * 
	 * @return java.lang.Object
	 * @param funcBody jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category func eval
	 */
	public Object fexprEval_arguList_(JunLispList funcBody, JunLispList arguList) {
		Object result = new Integer(-1);
		JunLispList expression = (JunLispCons) funcBody;
		Object funcName = ((JunLispCons) expression).head();
		expression = (JunLispList) (((JunLispCons) expression).tail());
		expression = (JunLispList) (((JunLispCons) expression).tail());
		JunLispList lvarList = (JunLispList) (((JunLispCons) expression).head());
		expression = (JunLispList) (((JunLispCons) expression).tail());

		if (lvarList.length() != 1) {
			return fatal_("too few or many arguments " + printString_(lvarList) + " for " + printString_(funcName));
		}

		bindMark();
		Object lvar = ((JunLispCons) lvarList).head();
		bind_value_(lvar, arguList);
		Object tempList = expression;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			result = evaluate_(each);
			tempList = ((JunLispCons) tempList).tail();
		}
		unbind();

		return result;
	}

	/**
	 * this function pre-define how to evaluate a lisp "+" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions
	 */
	public Object fsubrAdd_(JunLispList arguList) {
		Object tempList = arguList;
		Object v = JunLispNil.NullList();
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object a = evaluate_(each);
			if (!(a instanceof Number)) { //(a respondsTo: #+)
				return fatal_("unexpected argument " + printString_(a) + " for + ");
			}

			if (v instanceof JunLispNil) {
				v = a;
			} else {
				//v := v + a
				if (v instanceof Integer) {
					if (a instanceof Integer) {
						v = new Integer(((Integer) v).intValue() + ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Integer) v).intValue() + ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Integer) v).intValue() + ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Integer) v).intValue() + ((Float) a).floatValue());
					}
				} else if (v instanceof Long) {
					if (a instanceof Integer) {
						v = new Long(((Long) v).longValue() + ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Long) v).longValue() + ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Long) v).longValue() + ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Long) v).longValue() + ((Float) a).floatValue());
					}
				} else if (v instanceof Double) {
					if (a instanceof Integer) {
						v = new Double(((Double) v).doubleValue() + ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Double(((Double) v).doubleValue() + ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Double) v).doubleValue() + ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Double(((Double) v).doubleValue() + ((Float) a).floatValue());
					}
				} else if (v instanceof Float) {
					if (a instanceof Integer) {
						v = new Float(((Float) v).floatValue() + ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Float(((Float) v).floatValue() + ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Float) v).floatValue() + ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Float) v).floatValue() + ((Float) a).floatValue());
					}
				}
			}

			tempList = ((JunLispCons) tempList).tail();
		} //end while

		return v;
	}

	/**
	 * this function pre-define how to evaluate a lisp cond statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions
	 */
	public Object fsubrCond_(JunLispList arguList) {
		Object tempList = arguList;
		while (tempList instanceof JunLispCons) {
			Object each = (JunLispList) (((JunLispCons) tempList).head());
			Object each_head;
			if (each instanceof JunLispNil) {
				each_head = JunLispNil.NullList();
			} else {
				each_head = ((JunLispCons) each).head();
			}

			if (!(evaluate_(each_head) instanceof JunLispNil)) {
				//(self evaluate: each head) ~= JunLispNil null 
				Object result = JunLispNil.NullList();
				if (each instanceof JunLispCons) {
					if (((JunLispCons) each).tail() instanceof JunLispCons) {
						Object lisplist = ((JunLispCons) each).tail();
						while (lisplist instanceof JunLispCons) {
							Object expr = ((JunLispCons) lisplist).head();
							result = evaluate_(expr);
							lisplist = ((JunLispCons) lisplist).tail();
						}
					}
				}
				return result;
			}
			tempList = ((JunLispCons) tempList).tail();
		}

		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp defun statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrDefun_(JunLispList arguList) {
		Object funcName = ((JunLispCons) arguList).head();
		if (!(funcName instanceof StSymbol)) {
			return fatal_("unexpected function name " + printString_(funcName) + " for defun");
		}

		Object funcType = ((JunLispCons) (((JunLispCons) arguList).tail())).head();
		if (funcType instanceof StSymbol) {
			if (funcType == $("lambda")) {
				putprop_key_value_(funcName, $("expr"), arguList);
				return funcName;
			}
			if (funcType == $("nlambda")) {
				putprop_key_value_(funcName, $("fexpr"), arguList);
				return funcName;
			}
		}
		Object value = JunLispCons.Head_tail_(funcName, JunLispCons.Head_tail_($("lambda"), ((JunLispCons) arguList).tail()));
		putprop_key_value_(funcName, $("expr"), value);
		return funcName;
	}

	/**
	 * this function pre-define how to evaluate a lisp "/" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrDiv_(JunLispList arguList) {
		Object tempList = arguList;
		Object v = JunLispNil.NullList();
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object a = evaluate_(each);
			if (!(a instanceof Number)) { //(a respondsTo: #/)
				return fatal_("unexpected argument " + printString_(a) + " for / ");
			}

			if (v instanceof JunLispNil) {
				v = a;
			} else {
				//v := v / a  
				if (v instanceof Integer) {
					if (a instanceof Integer) {
						v = new Integer(((Integer) v).intValue() / ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Integer) v).intValue() / ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Integer) v).intValue() / ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Integer) v).intValue() / ((Float) a).floatValue());
					}
				} else if (v instanceof Long) {
					if (a instanceof Integer) {
						v = new Long(((Long) v).longValue() / ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Long) v).longValue() / ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Long) v).longValue() / ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Long) v).longValue() / ((Float) a).floatValue());
					}
				} else if (v instanceof Double) {
					if (a instanceof Integer) {
						v = new Double(((Double) v).doubleValue() / ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Double(((Double) v).doubleValue() / ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Double) v).doubleValue() / ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Double(((Double) v).doubleValue() / ((Float) a).floatValue());
					}
				} else if (v instanceof Float) {
					if (a instanceof Integer) {
						v = new Float(((Float) v).floatValue() / ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Float(((Float) v).floatValue() / ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Float) v).floatValue() / ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Float) v).floatValue() / ((Float) a).floatValue());
					}
				}
			}

			tempList = ((JunLispCons) tempList).tail();
		} // end while

		return v;
	}

	/**
	 * this function pre-define how to evaluate a lisp do statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrDo_(JunLispList arguList) {
		JunLispList locals = (JunLispList) (((JunLispCons) arguList).head());
		JunLispList executions = (JunLispList) (((JunLispCons) arguList).tail());
		bindMark();
		Object tempList = locals;
		while (tempList instanceof JunLispCons) {
			Object lvar = ((JunLispCons) tempList).head();
			bind_value_(lvar, JunLispNil.NullList());
			tempList = ((JunLispCons) tempList).tail();
		}
		Object result = JunLispNil.NullList();
		tempList = executions;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			result = evaluate_(each);
			tempList = ((JunLispCons) tempList).tail();
		}
		unbind();

		return result;
	}

	/**
	 * this function evaluate  a  pre-defined lisp function call  statement to an Object in fact.
	 * 
	 * @return java.lang.Object
	 * @param funcBody jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrEval_arguList_(JunLispList funcBody, JunLispList arguList) {
		StSymbol messageSelector = (StSymbol) (((JunLispCons) (((JunLispCons) funcBody).tail())).head());

		//	^self perform: messageSelector with: arguList
		if (messageSelector == $("fsubrAdd:")) {
			return fsubrAdd_(arguList);
		}

		if (messageSelector == $("fsubrCond:")) {
			return fsubrCond_(arguList);
		}

		if (messageSelector == $("fsubrDefun:")) {
			return fsubrDefun_(arguList);
		}

		if (messageSelector == $("fsubrDiv:")) {
			return fsubrDiv_(arguList);
		}

		if (messageSelector == $("fsubrDo:")) {
			return fsubrDo_(arguList);
		}

		if (messageSelector == $("fsubrldiv:")) {
			return fsubrldiv_(arguList);
		}

		if (messageSelector == $("fsubrIf:")) {
			return fsubrIf_(arguList);
		}

		if (messageSelector == $("fsubrMlt:")) {
			return fsubrMlt_(arguList);
		}

		if (messageSelector == $("fsubrMod:")) {
			return fsubrMod_(arguList);
		}

		if (messageSelector == $("fsubrProgn:")) {
			return fsubrProgn_(arguList);
		}

		if (messageSelector == $("fsubrQuote:")) {
			return fsubrQuote_(arguList);
		}

		if (messageSelector == $("fsubrRead:")) {
			return fsubrRead_(arguList);
		}

		if (messageSelector == $("fsubrRepeat:")) {
			return fsubrRepeat_(arguList);
		}

		if (messageSelector == $("fsubrSend:")) {
			return fsubrSend_(arguList);
		}

		if (messageSelector == $("fsubrSetq:")) {
			return fsubrSetq_(arguList);
		}

		if (messageSelector == $("fsubrSub:")) {
			return fsubrSub_(arguList);
		}

		if (messageSelector == $("fsubrWhile:")) {
			return fsubrWhile_(arguList);
		}

		return fatal_("Unknown method " + messageSelector.toString());
	}

	/**
	 * this function pre-define how to evaluate a lisp if statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrIf_(JunLispList arguList) {
		Object predicate = ((JunLispCons) arguList).head();
		Object then = ((JunLispCons) (((JunLispCons) arguList).tail())).head();
		JunLispList list = (JunLispList) (((JunLispCons) (((JunLispCons) arguList).tail())).tail());
		JunLispList truePart = JunLispNil.NullList();
		JunLispList falsePart = JunLispNil.NullList();
		boolean bool = true;
		Object tempList = list;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			if (each instanceof StSymbol) {
				if (each == $("else")) {
					bool = false;
					tempList = ((JunLispCons) tempList).tail();
					continue;
				}
			}
			if (bool) {
				truePart = JunLispCons.Head_tail_(each, truePart);
			} else {
				falsePart = JunLispCons.Head_tail_(each, falsePart);
			}
			tempList = ((JunLispCons) tempList).tail();
		}

		if (!(then instanceof StSymbol)) {
			return fatal_("unexpected   format for if ");
		}
		if (then != $("then")) {
			return fatal_("unexpected   format for if ");
		}

		truePart = (JunLispList) (((JunLispCons) truePart).reverse());
		falsePart = (JunLispList) (((JunLispCons) falsePart).reverse());
		Object result = JunLispNil.NullList();
		if (evaluate_(predicate) instanceof JunLispNil) { //(self evaluate: predicate) = JunLispNil null
			tempList = falsePart;
			while (tempList instanceof JunLispCons) {
				Object each = ((JunLispCons) tempList).head();
				result = evaluate_(each);
				tempList = ((JunLispCons) tempList).tail();
			}
		} else {
			tempList = truePart;
			while (tempList instanceof JunLispCons) {
				Object each = ((JunLispCons) tempList).head();
				result = evaluate_(each);
				tempList = ((JunLispCons) tempList).tail();
			}
		}

		return result;
	}

	/**
	 * this function pre-define how to evaluate a lisp "//" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrldiv_(JunLispList arguList) {
		double dValue = (double) -1;
		Object tempList = arguList;
		Object v = JunLispNil.NullList();
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object a = evaluate_(each);
			if (!(a instanceof Number)) { //(a respondsTo: #//)
				return fatal_("unexpected argument " + printString_(a) + " for // ");
			}

			if (v instanceof JunLispNil) {
				v = a;
			} else { //v := v // a
				if (v instanceof Integer) {
					if (a instanceof Integer) {
						dValue = (double) ((Integer) v).intValue() / ((Integer) a).intValue();
					} else if (a instanceof Long) {
						dValue = (double) ((Integer) v).intValue() / ((Long) a).longValue();
					} else if (a instanceof Double) {
						dValue = (double) ((Integer) v).intValue() / ((Double) a).doubleValue();
					} else if (a instanceof Float) {
						dValue = (double) ((Integer) v).intValue() / ((Float) a).floatValue();
					}
				} else if (v instanceof Long) {
					if (a instanceof Integer) {
						dValue = (double) ((Long) v).longValue() / ((Integer) a).intValue();
					} else if (a instanceof Long) {
						dValue = (double) ((Long) v).longValue() / ((Long) a).longValue();
					} else if (a instanceof Double) {
						dValue = (double) ((Long) v).longValue() / ((Double) a).doubleValue();
					} else if (a instanceof Float) {
						dValue = (double) ((Long) v).longValue() / ((Float) a).floatValue();
					}
				} else if (v instanceof Double) {
					if (a instanceof Integer) {
						dValue = (double) ((Double) v).doubleValue() / ((Integer) a).intValue();
					} else if (a instanceof Long) {
						dValue = (double) ((Double) v).doubleValue() / ((Long) a).longValue();
					} else if (a instanceof Double) {
						dValue = (double) ((Double) v).doubleValue() / ((Double) a).doubleValue();
					} else if (a instanceof Float) {
						dValue = (double) ((Double) v).doubleValue() / ((Float) a).floatValue();
					}
				} else if (v instanceof Float) {
					if (a instanceof Integer) {
						dValue = (double) ((Float) v).floatValue() / ((Integer) a).intValue();
					} else if (a instanceof Long) {
						dValue = (double) ((Float) v).floatValue() / ((Long) a).longValue();
					} else if (a instanceof Double) {
						dValue = (double) ((Float) v).floatValue() / ((Double) a).doubleValue();
					} else if (a instanceof Float) {
						dValue = (double) ((Float) v).floatValue() / ((Float) a).floatValue();
					}
				}

				if (dValue >= 0) {
					v = new Integer((int) dValue);
				} else {
					if (dValue < (int) dValue) {
						v = new Integer((int) dValue - 1);
					} else {
						v = new Integer((int) dValue);
					}
				}
			}

			tempList = ((JunLispCons) tempList).tail();
		} //end while

		return v;
	}

	/**
	 * this function pre-define how to evaluate a lisp "" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrMlt_(JunLispList arguList) {
		Object v = JunLispNil.NullList();
		Object tempList = arguList;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object a = evaluate_(each);
			if (!(a instanceof Number)) { //(a respondsTo: #*)
				return fatal_("unexpected argument " + printString_(a) + " for * ");
			}

			if (v instanceof JunLispNil) {
				v = a;
			} else { //v := v * a
				if (v instanceof Integer) {
					if (a instanceof Integer) {
						v = new Integer(((Integer) v).intValue() * ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Integer) v).intValue() * ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Integer) v).intValue() * ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Integer) v).intValue() * ((Float) a).floatValue());
					}
				} else if (v instanceof Long) {
					if (a instanceof Integer) {
						v = new Long(((Long) v).longValue() * ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Long) v).longValue() * ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Long) v).longValue() * ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Long) v).longValue() * ((Float) a).floatValue());
					}
				} else if (v instanceof Double) {
					if (a instanceof Integer) {
						v = new Double(((Double) v).doubleValue() * ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Double(((Double) v).doubleValue() * ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Double) v).doubleValue() * ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Double(((Double) v).doubleValue() * ((Float) a).floatValue());
					}
				} else if (v instanceof Float) {
					if (a instanceof Integer) {
						v = new Float(((Float) v).floatValue() * ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Float(((Float) v).floatValue() * ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Float) v).floatValue() * ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Float) v).floatValue() * ((Float) a).floatValue());
					}
				}
			}

			tempList = ((JunLispCons) tempList).tail();
		} //end while

		return v;
	}

	/**
	 * this function pre-define how to evaluate a lisp "\\" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrMod_(JunLispList arguList) {
		Object v = JunLispNil.NullList();
		Object tempList = arguList;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object a = evaluate_(each);
			if (!(a instanceof Number)) { //(a respondsTo: #\\)
				return fatal_("unexpected argument " + printString_(a) + " for \\");
			}

			if (v instanceof JunLispNil) {
				v = a;
			} else { //[v := v \\ a]
				if (v instanceof Integer) {
					if (a instanceof Integer) {
						v = new Integer(((Integer) v).intValue() % ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Integer) v).intValue() % ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Integer) v).intValue() % ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Integer) v).intValue() % ((Float) a).floatValue());
					}
				} else if (v instanceof Long) {
					if (a instanceof Integer) {
						v = new Long(((Long) v).longValue() % ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Long) v).longValue() % ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Long) v).longValue() % ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Long) v).longValue() % ((Float) a).floatValue());
					}
				} else if (v instanceof Double) {
					if (a instanceof Integer) {
						v = new Double(((Double) v).doubleValue() % ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Double(((Double) v).doubleValue() % ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Double) v).doubleValue() % ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Double(((Double) v).doubleValue() % ((Float) a).floatValue());
					}
				} else if (v instanceof Float) {
					if (a instanceof Integer) {
						v = new Float(((Float) v).floatValue() % ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Float(((Float) v).floatValue() % ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Float) v).floatValue() % ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Float) v).floatValue() % ((Float) a).floatValue());
					}
				}
			}

			tempList = ((JunLispCons) tempList).tail();
		} // end while

		return v;
	}

	/**
	 * this function pre-define how to evaluate a lisp progn statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrProgn_(JunLispList arguList) {
		Object result = JunLispNil.NullList();
		Object tempList = arguList;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			result = evaluate_(each);
			tempList = ((JunLispCons) tempList).tail();
		}

		return result;
	}

	/**
	 * this function pre-define how to evaluate a lisp quote statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrQuote_(JunLispList arguList) {
		if (arguList instanceof JunLispCons) {
			return ((JunLispCons) arguList).head();
		} else {
			return ((JunLispNil) arguList).head();
		}
	}

	/**
	 * The fsubr "read".
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrRead_(JunLispList arguList) {
		String message = "          " + JunSystem.$String("Input an S expression.") + "          ";
		String defaultString = "";
		if (arguList instanceof JunLispCons) {
			JunLispCons cons = (JunLispCons) arguList;
			if (cons.length() >= 1) {
				Object anObject = this.evaluate_(cons.nth_(1));
				if (anObject instanceof String) {
					message = (String) anObject;
				} else {
					message = this.printString_(anObject);
				}
			}
			if (cons.length() >= 2) {
				Object anObject = this.evaluate_(cons.nth_(2));
				if (anObject instanceof String) {
					defaultString = (String) anObject;
				} else {
					defaultString = this.printString_(defaultString);
				}
			}
		}

		String answer = JunDialog.Request_(message, defaultString);
		if (answer.length() == 0) {
			return JunLispNil.NullList();
		}

		return JunLispParser.Parse_(new StReadStream(answer));
	}

	/**
	 * this function pre-define how to evaluate a lisp repeat statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrRepeat_(JunLispList arguList) {
		JunLispList reverse = (JunLispList) (((JunLispCons) arguList).reverse());
		Object predicate = ((JunLispCons) reverse).head();
		Object until = ((JunLispCons) (((JunLispCons) reverse).tail())).head();
		JunLispList executions = (JunLispList) (((JunLispCons) (((JunLispCons) (((JunLispCons) reverse).tail())).tail())).reverse());
		if (!(until instanceof StSymbol)) {
			return fatal_("unexpected format for repeat");
		}
		if (until != $("until")) {
			return fatal_("unexpected format for repeat");
		}

		Object result = JunLispNil.NullList();
		Object tempList = executions;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			result = evaluate_(each);
			tempList = ((JunLispCons) tempList).tail();
		}
		while (!(evaluate_(predicate) instanceof JunLispNil)) {
			tempList = executions;
			while (tempList instanceof JunLispCons) {
				Object each = ((JunLispCons) tempList).head();
				result = evaluate_(each);
				tempList = ((JunLispCons) tempList).tail();
			}
		}

		return result;
	}

	/**
	 * The fsubr "send".
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException 
	 * @category fsubr functions
	 */
	public Object fsubrSend_(JunLispList arguList) {
		JunLispList list = arguList;
		Object receiver = this.evaluate_(list.head());
		if (receiver instanceof StObject == false) {
			return this.fatal_("unexpected receiver " + receiver.getClass() + " for send");
		}

		list = (JunLispList) list.tail();
		Object selector = this.evaluate_(list.head());
		if (selector instanceof StSymbol == false) {
			return this.fatal_("unexpected selector " + selector + " for send");
		}

		list = (JunLispList) list.tail();
		Vector arguments = new Vector();
		while (list instanceof JunLispCons) {
			arguments.addElement(this.evaluate_(list.head()));
			list = (JunLispList) list.tail();
		}

		Object[] argumentArray = new Object[arguments.size()];
		arguments.copyInto(argumentArray);
		try {
			return ((StObject) receiver).perform_withArguments_(selector.toString(), argumentArray);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * this function pre-define how to evaluate a lisp setq statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrSetq_(JunLispList arguList) {
		Object list = arguList;
		Object a2 = JunLispNil.NullList();
		while (list instanceof JunLispCons) {
			Object a1 = ((JunLispCons) list).head();
			if (!(a1 instanceof StSymbol)) {
				return fatal_("unexpected variable " + printString_(a1) + " for setq");
			}

			list = ((JunLispCons) list).tail();
			a2 = evaluate_(((JunLispCons) list).head());
			putprop_key_value_(a1, $("apval"), a2);
			list = ((JunLispCons) list).tail();
		}

		return a2;
	}

	/**
	 * this function pre-define how to evaluate a lisp sub statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrSub_(JunLispList arguList) {
		Object v = JunLispNil.NullList();
		Object tempList = arguList;
		while (tempList instanceof JunLispCons) {
			Object each = ((JunLispCons) tempList).head();
			Object a = evaluate_(each);
			if (!(a instanceof Number)) { //(a respondsTo: #-)
				return fatal_("unexpected argument " + printString_(a) + " for - ");
			}

			if (v instanceof JunLispNil) {
				v = a;
			} else { //[v := v - a]
				if (v instanceof Integer) {
					if (a instanceof Integer) {
						v = new Integer(((Integer) v).intValue() - ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Integer) v).intValue() - ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Integer) v).intValue() - ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Integer) v).intValue() - ((Float) a).floatValue());
					}
				} else if (v instanceof Long) {
					if (a instanceof Integer) {
						v = new Long(((Long) v).longValue() - ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Long(((Long) v).longValue() - ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Long) v).longValue() - ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Long) v).longValue() - ((Float) a).floatValue());
					}
				} else if (v instanceof Double) {
					if (a instanceof Integer) {
						v = new Double(((Double) v).doubleValue() - ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Double(((Double) v).doubleValue() - ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Double) v).doubleValue() - ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Double(((Double) v).doubleValue() - ((Float) a).floatValue());
					}
				} else if (v instanceof Float) {
					if (a instanceof Integer) {
						v = new Float(((Float) v).floatValue() - ((Integer) a).intValue());
					} else if (a instanceof Long) {
						v = new Float(((Float) v).floatValue() - ((Long) a).longValue());
					} else if (a instanceof Double) {
						v = new Double(((Float) v).floatValue() - ((Double) a).doubleValue());
					} else if (a instanceof Float) {
						v = new Float(((Float) v).floatValue() - ((Float) a).floatValue());
					}
				}
			}

			tempList = ((JunLispCons) tempList).tail();
		} // end while

		return v;
	}

	/**
	 * this function pre-define how to evaluate a lisp while statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category fsubr functions 
	 */
	public Object fsubrWhile_(JunLispList arguList) {
		Object predicate = ((JunLispCons) arguList).head();
		Object doSymbol = ((JunLispCons) (((JunLispCons) arguList).tail())).head();
		JunLispList executions = (JunLispList) (((JunLispCons) (((JunLispCons) arguList).tail())).tail());
		if (!(doSymbol instanceof StSymbol)) {
			return fatal_("unexpected format for while");
		}
		if (doSymbol != $("do")) {
			return fatal_("unexpected format for while");
		}
		Object result = JunLispNil.NullList();
		while (!(evaluate_(predicate) instanceof JunLispNil)) {
			Object each;
			Object tempList = executions;
			while (tempList instanceof JunLispCons) {
				each = ((JunLispCons) tempList).head();
				result = evaluate_(each);
				tempList = ((JunLispCons) tempList).tail();
			}
		}
		return result;
	}

	/**
	 * Get the property.
	 * 
	 * @return java.lang.Object
	 * @param identifier java.lang.Object
	 * @param key java.lang.Object
	 * @category property access
	 */
	public Object getprop_key_(Object identifier, Object key) {
		lispTable.intern_(identifier);
		Object value = lispTable.getprop_key_(identifier, key);
		if (value == null) {
			return JunLispNil.NullList();
		}
		return value;
	}

	/**
	 * Initialize the lisp interpreter.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		lispTable = new JunLispTable();
		bindStack = new Stack();
		failBlock = new StBlockClosure() {
			public Object value_(Object errorMessage) {
				throw SmalltalkException.Error((String) errorMessage);
			}
		};
		this.initializeSubrFunctions();
		this.initializeFsubrFunctions();
		this.initializeExprFunctions();
		this.initializeFexprFunctions();
		textCollector = new PrintStream(new ByteArrayOutputStream());
	}

	/**
	 * this function pre-define some functions for lisp
	 * 
	 * @category initialize-release
	 */
	public void initializeExprFunctions() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("\n\n");
		buffer.append("% Expr Functions \n");
		buffer.append("(progn \n\n");
		buffer.append("	% ++ \n");
		buffer.append("	(defun ++ lambda (x) \n");
		buffer.append("		(+ x 1)) \n\n");
		buffer.append("	% -- \n");
		buffer.append("	(defun -- lambda (x) \n");
		buffer.append("		(- x 1)) \n\n");
		buffer.append("	% assoc \n");
		buffer.append("	(defun assoc lambda (x a) \n");
		buffer.append("		(cond  \n");
		buffer.append("			((null a) nil) \n");
		buffer.append("			((equal x (car (car a))) (car a)) \n");
		buffer.append("			(t (assoc x (cdr a))))) \n\n");
		buffer.append("	% copy \n");
		buffer.append("	(defun copy lambda (x) \n");
		buffer.append("		(cond \n");
		buffer.append("			((null x) nil) \n");
		buffer.append("			(t (cons (car x) (copy (cdr x)))))) \n\n");
		buffer.append("	% mapc \n");
		buffer.append("	(defun mapc lambda (f x) \n");
		buffer.append("		(cond \n");
		buffer.append("			((null x) nil) \n");
		buffer.append("			(t (progn \n");
		buffer.append("				(eval (cons f (cons `(car x) nil))) \n");
		buffer.append("				(mapc f (cdr x)))))) \n\n");
		buffer.append("	% mapcar \n");
		buffer.append("	(defun mapcar lambda (f x) \n");
		buffer.append("		(cond \n");
		buffer.append("			((null x) nil) \n");
		buffer.append("			(t (cons \n");
		buffer.append("				(eval (cons f (cons `(car x) nil))) \n");
		buffer.append("				(mapcar f (cdr x)))))) \n\n");
		buffer.append("	) % end ");

		evaluateTopLevel_(JunLispParser.Parse_(buffer.toString()));
	}

	/**
	 * this function pre-define some functions for lisp
	 * 
	 * @category initialize-release
	 */
	public void initializeFexprFunctions() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("\n\n");
		buffer.append("% Expr Functions \n");
		buffer.append("(progn  \n\n");
		buffer.append("	% and \n");
		buffer.append("	(defun and nlambda (x) \n");
		buffer.append("		(do	(list) \n");
		buffer.append("			(setq list x) \n");
		buffer.append("			(while \n");
		buffer.append("				(if (null list) then nil else (eval (car list))) \n");
		buffer.append("				do \n");
		buffer.append("				(setq list (cdr list))) \n");
		buffer.append("			(if (null list) then t else nil))) \n\n");
		buffer.append("	% list \n");
		buffer.append("	(defun list nlambda (x) \n");
		buffer.append("		(mapcar `eval x)) \n\n");
		buffer.append("	% or \n");
		buffer.append("	(defun or nlambda (x) \n");
		buffer.append("		(do	(list) \n");
		buffer.append("			(setq list x) \n");
		buffer.append("			(while \n");
		buffer.append("				(if (null list) then nil else (not (eval (car list)))) \n");
		buffer.append("				do \n");
		buffer.append("				(setq list (cdr list))) \n");
		buffer.append("			(if (null list) then nil else t))) \n\n");
		buffer.append("	) % end");
		
		evaluateTopLevel_(JunLispParser.Parse_(buffer.toString()));
	}

	/**
	 * Initialize the fsubr functions.
	 * 
	 * @category initialize-release
	 */
	public void initializeFsubrFunctions() {
		setFsubrFunc_(new Object[] { $("*"), $("fsubrMlt:") });
		setFsubrFunc_(new Object[] { $("+"), $("fsubrAdd:") });
		setFsubrFunc_(new Object[] { $("-"), $("fsubrSub:") });
		setFsubrFunc_(new Object[] { $("/"), $("fsubrDiv:") });
		setFsubrFunc_(new Object[] { $("//"), $("fsubrldiv:") });
		setFsubrFunc_(new Object[] { $("cond"), $("fsubrCond:") });
		setFsubrFunc_(new Object[] { $("defun"), $("fsubrDefun:") });
		setFsubrFunc_(new Object[] { $("do"), $("fsubrDo:") });
		setFsubrFunc_(new Object[] { $("if"), $("fsubrIf:") });
		setFsubrFunc_(new Object[] { $("progn"), $("fsubrProgn:") });
		setFsubrFunc_(new Object[] { $("quote"), $("fsubrQuote:") });
		setFsubrFunc_(new Object[] { $("read"), $("fsubrRead:") });
		setFsubrFunc_(new Object[] { $("repeat"), $("fsubrRepeat:") });
		setFsubrFunc_(new Object[] { $("send"), $("fsubrSend:") });
		setFsubrFunc_(new Object[] { $("setq"), $("fsubrSetq:") });
		setFsubrFunc_(new Object[] { $("while"), $("fsubrWhile:") });
		setFsubrFunc_(new Object[] { $("\\\\"), $("fsubrMod:") });
	}

	/**
	 * Initialize the subr functions.
	 * 
	 * @category initialize-release
	 */
	public void initializeSubrFunctions() {
		setSubrFunc_(new Object[] { $("<"), $("subrLt:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("<="), $("subrLe:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("="), $("subrEqual:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("=="), $("subrEq:"), new Integer(2) });
		setSubrFunc_(new Object[] { $(">"), $("subrGt:"), new Integer(2) });
		setSubrFunc_(new Object[] { $(">="), $("subrGe:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("append"), $("subrAppend:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("atom"), $("subrAtom:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("car"), $("subrCar:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("cdr"), $("subrCdr:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("cons"), $("subrCons:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("consp"), $("subrConsp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("dtpr"), $("subrConsp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("doublep"), $("subrDoublep:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("eq"), $("subrEq:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("equal"), $("subrEqual:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("eval"), $("subrEval:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("exprs"), $("subrExprs"), new Integer(0) });
		setSubrFunc_(new Object[] { $("fexprs"), $("subrFexprs"), new Integer(0) });
		setSubrFunc_(new Object[] { $("floatp"), $("subrFloatp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("fsubrs"), $("subrFsubrs"), new Integer(0) });
		setSubrFunc_(new Object[] { $("gc"), $("subrGc"), new Integer(0) });
		setSubrFunc_(new Object[] { $("gensym"), $("subrGensym"), new Integer(0) });
		setSubrFunc_(new Object[] { $("getprop"), $("subrGetprop:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("integerp"), $("subrIntegerp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("last"), $("subrLast:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("length"), $("subrLength:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("listp"), $("subrListp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("member"), $("subrMember:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("memq"), $("subrMemq:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("nconc"), $("subrNconc:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("neq"), $("subrNeq:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("nequal"), $("subrNequal:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("not"), $("subrNull:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("nth"), $("subrNth:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("null"), $("subrNull:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("numberp"), $("subrNumberp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("oblist"), $("subrOblist"), new Integer(0) });
		setSubrFunc_(new Object[] { $("pp"), $("subrPp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("princ"), $("subrPrinc:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("print"), $("subrPrint:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("putprop"), $("subrPutprop:"), new Integer(3) });
		setSubrFunc_(new Object[] { $("remprop"), $("subrRemprop:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("reverse"), $("subrReverse:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("rplaca"), $("subrRplaca:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("placd"), $("subrPlacd:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("stringp"), $("subrStringp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("subrs"), $("subrSubrs"), new Integer(0) });
		setSubrFunc_(new Object[] { $("symbolp"), $("subrSymbolp:"), new Integer(1) });
		setSubrFunc_(new Object[] { $("terpri"), $("subrTerpri"), new Integer(0) });
		setSubrFunc_(new Object[] { $("~="), $("subrNequal:"), new Integer(2) });
		setSubrFunc_(new Object[] { $("~~"), $("subrNeq:"), new Integer(2) });
	}

	/**
	 * Put the property.
	 * 
	 * @return java.lang.Object
	 * @param identifier ja.lang.Object
	 * @param key java.langObject
	 * @param value java.lang.Object
	 * @category property access 
	 */
	public Object putprop_key_value_(Object identifier, Object key, Object value) {
		lispTable.intern_(identifier);
		return lispTable.putprop_key_value_(identifier, key, value);
	}

	/**
	 * Rmoeve the property.
	 * 
	 * @return java.lang.Object
	 * @param identifier java.lang.Object
	 * @param key java.lang.Object
	 * @category property access 
	 */
	public Object remprop_key_(Object identifier, Object key) {
		lispTable.intern_(identifier);
		Object value = lispTable.remprop_key_(identifier, key);
		if (value == null) {
			return JunLispNil.NullList();
		}
		return value;
	}

	/**
	 * Set the fsbur function.
	 * 
	 * @param bodyArray java.lang.Object[]
	 * @category private
	 */
	protected void setFsubrFunc_(Object[] bodyArray) {
		Object a1 = bodyArray[0];
		putprop_key_value_($(a1.toString()), $("fsubr"), JunLispCons.List_(bodyArray));
	}

	/**
	 * Set the sbur function.
	 * 
	 * @param bodyArray java.lang.Object[]
	 * @category private
	 */
	protected void setSubrFunc_(Object[] bodyArray) {
		Object a1 = bodyArray[0];
		putprop_key_value_($(a1.toString()), $("subr"), JunLispCons.List_(bodyArray));
	}

	/**
	 * Skip reading the separators
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category private
	 */
	protected void skipSeparators_(StReadStream aStream) {
		while (!aStream.atEnd()) {
			char ch = aStream.peek();
			StSymbol symbol = (StSymbol) JunLispScannerTable.ScannerTable().at_(ch);
			if (symbol != $("xDelimiter")) {
				break;
			}
			aStream.next();
		}
	}

	/**
	 * this function pre-define how to evaluate a lisp append  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions
	 */
	public Object subrAppend_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a1) + " for append");
		}
		if (!(a2 instanceof JunLispList)) {
			return fatal_("unexpected argument " + printString_(a2) + " for append");
		}
		return ((JunLispCons) a1).append_((JunLispList) a2);
	}

	/**
	 * this function pre-define how to evaluate a lisp atom  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrAtom_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (!(a1 instanceof JunLispCons)) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp car  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrCar_(Object[] arguArray) {
		Object list = arguArray[0];
		if (list instanceof JunLispCons) {
			return ((JunLispCons) list).head();
		}
		if (list instanceof JunLispNil) {
			return JunLispNil.NullList();
		}
		return fatal_("unexpceted argument " + printString_(list) + " for car");
	}

	/**
	 * this function pre-define how to evaluate a lisp cdr  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrCdr_(Object[] arguArray) {
		Object list = arguArray[0];
		if (list instanceof JunLispCons) {
			return ((JunLispCons) list).tail();
		}
		if (list instanceof JunLispNil) {
			return JunLispNil.NullList();
		}
		return fatal_("unexpceted argument " + printString_(list) + " for car");
	}

	/**
	 * this function pre-define how to evaluate a lisp cons statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions
	 */
	public Object subrCons_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		return JunLispCons.Head_tail_(a1, a2);
	}

	/**
	 * this function pre-define how to evaluate a lisp consp  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrConsp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof JunLispCons) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp doublep  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrDoublep_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof Double) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp eq or "=="  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrEq_(Object[] arguArray) {
		Object bool;
		if (arguArray[0] == arguArray[1]) {
			bool = $("t");
		} else {
			bool = JunLispNil.NullList();
		}
		return bool;
	}

	/**
	 * this function pre-define how to evaluate a lisp equal or "="  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrEqual_(Object[] arguArray) {
		Object bool;
		if (arguArray[0].equals(arguArray[1])) {
			bool = $("t");
		} else {
			bool = JunLispNil.NullList();
		}
		return bool;
	}

	/**
	 * this function pre-define how to evaluate a lisp eval  statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrEval_(Object[] arguArray) {
		Object a1 = arguArray[0];
		return evaluate_(a1);
	}

	/**
	 * this function evaluate  a  pre-defined lisp function call  statement to an Object in fact.
	 * 
	 * @return java.lang.Object
	 * @param funcBody jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param arguList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException 
	 * @category subr functions
	 */
	public Object subrEval_arguList_(JunLispList funcBody, JunLispList arguList) {
		Object funcName = ((JunLispCons) funcBody).head();
		StSymbol messageSelector = (StSymbol) (((JunLispCons) (((JunLispCons) funcBody).tail())).head());
		int arguCount = ((Number) (((JunLispCons) (((JunLispCons) (((JunLispCons) funcBody).tail())).tail())).head())).intValue();
		Object[] arguBuffer = new Object[arguCount];
		JunLispList list = arguList;
		for (int nIndex = 0; nIndex < arguCount; nIndex++) {
			if (list.nullList()) {
				return fatal_("too few arguments for " + printString_(funcName));
			}

			arguBuffer[nIndex] = (evaluate_(((JunLispCons) list).head()));
			list = (JunLispList) (((JunLispCons) list).tail());
		}

		if (!list.nullList()) {
			return fatal_("too many arguments for " + printString_(funcName));
		}

		if (arguCount == 0) {
			try {
				return perform_(messageSelector.toString());
			} catch (Exception e) {
				throw new SmalltalkException(e);
			}
		} else {
			//^self perform: messageSelector with: arguBuffer asArray 
			if (messageSelector == $("subrAppend:")) {
				return subrAppend_(arguBuffer);
			}
			if (messageSelector == $("subrAtom:")) {
				return subrAtom_(arguBuffer);
			}
			if (messageSelector == $("subrCar:")) {
				return subrCar_(arguBuffer);
			}
			if (messageSelector == $("subrCdr:")) {
				return subrCdr_(arguBuffer);
			}
			if (messageSelector == $("subrCons:")) {
				return subrCons_(arguBuffer);
			}
			if (messageSelector == $("subrConsp:")) {
				return subrConsp_(arguBuffer);
			}
			if (messageSelector == $("subrDoublep:")) {
				return subrDoublep_(arguBuffer);
			}
			if (messageSelector == $("subrEq:")) {
				return subrEq_(arguBuffer);
			}
			if (messageSelector == $("subrEqual:")) {
				return subrEqual_(arguBuffer);
			}
			if (messageSelector == $("subrEval:")) {
				return subrEval_(arguBuffer);
			}
			if (messageSelector == $("subrFloatp:")) {
				return subrFloatp_(arguBuffer);
			}
			if (messageSelector == $("subrGe:")) {
				return subrGe_(arguBuffer);
			}
			if (messageSelector == $("subrGetprop:")) {
				return subrGetprop_(arguBuffer);
			}
			if (messageSelector == $("subrGt:")) {
				return subrGt_(arguBuffer);
			}
			if (messageSelector == $("subrIntegerp:")) {
				return subrIntegerp_(arguBuffer);
			}
			if (messageSelector == $("subrLast:")) {
				return subrLast_(arguBuffer);
			}
			if (messageSelector == $("subrLe:")) {
				return subrLe_(arguBuffer);
			}
			if (messageSelector == $("subrLength:")) {
				return subrLength_(arguBuffer);
			}
			if (messageSelector == $("subrListp:")) {
				return subrListp_(arguBuffer);
			}
			if (messageSelector == $("subrLt:")) {
				return subrLt_(arguBuffer);
			}
			if (messageSelector == $("subrMember:")) {
				return subrMember_(arguBuffer);
			}
			if (messageSelector == $("subrMemq:")) {
				return subrMemq_(arguBuffer);
			}
			if (messageSelector == $("subrNconc:")) {
				return subrNconc_(arguBuffer);
			}
			if (messageSelector == $("subrNeq:")) {
				return subrNeq_(arguBuffer);
			}
			if (messageSelector == $("subrNequal:")) {
				return subrNequal_(arguBuffer);
			}
			if (messageSelector == $("subrNth:")) {
				return subrNth_(arguBuffer);
			}
			if (messageSelector == $("subrNull:")) {
				return subrNull_(arguBuffer);
			}
			if (messageSelector == $("subrNumberp:")) {
				return subrNumberp_(arguBuffer);
			}
			if (messageSelector == $("subrPp:")) {
				return subrPp_(arguBuffer);
			}
			if (messageSelector == $("subrPrinc:")) {
				return subrPrinc_(arguBuffer);
			}
			if (messageSelector == $("subrPutprop:")) {
				return subrPutprop_(arguBuffer);
			}
			if (messageSelector == $("subrRemprop:")) {
				return subrRemprop_(arguBuffer);
			}
			if (messageSelector == $("subrReverse:")) {
				return subrReverse_(arguBuffer);
			}
			if (messageSelector == $("subrRplaca:")) {
				return subrRplaca_(arguBuffer);
			}
			if (messageSelector == $("subrRplacd:")) {
				return subrRplacd_(arguBuffer);
			}
			if (messageSelector == $("subrStringp:")) {
				return subrStringp_(arguBuffer);
			}
			if (messageSelector == $("subrSymbolp:")) {
				return subrSymbolp_(arguBuffer);
			}
		}

		return fatal_("Unknown method " + messageSelector.toString());
	}

	/**
	 * this function pre-define how to evaluate a lisp exprs statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrExprs() {
		JunLispList list = JunLispNil.NullList();
		JunLispList lispcons = subrOblist();
		JunLispList reverse = (JunLispList) (((JunLispCons) lispcons).reverse());
		Object element = reverse;
		while (element instanceof JunLispCons) {
			Object id;
			JunLispList propkey;
			id = ((JunLispCons) element).head();
			propkey = (JunLispList) getprop_key_(id, $("expr"));
			if (!(propkey.nullList())) {
				list = JunLispCons.Head_tail_(id, list);
			}
			element = ((JunLispCons) element).tail();
		}
		return list;
	}

	/**
	 * this function pre-define how to evaluate a lisp fexprs statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrFexprs() {
		JunLispList list = JunLispNil.NullList();
		JunLispList lispcons = subrOblist();
		JunLispList reverse = (JunLispList) (((JunLispCons) lispcons).reverse());
		Object element = reverse;
		while (element instanceof JunLispCons) {
			Object id;
			JunLispList propkey;
			id = ((JunLispCons) element).head();
			propkey = (JunLispList) getprop_key_(id, $("fexpr"));

			if (!(propkey.nullList())) {
				list = JunLispCons.Head_tail_(id, list);
			}
			element = ((JunLispCons) element).tail();
		}
		return list;
	}

	/**
	 * this function pre-define how to evaluate a lisp floatp statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrFloatp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof Float) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp fsubrs statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrFsubrs() {
		JunLispList list = JunLispNil.NullList();
		JunLispList lispcons = subrOblist();
		JunLispList reverse = (JunLispList) (((JunLispCons) lispcons).reverse());
		Object element = reverse;
		while (element instanceof JunLispCons) {
			Object id;
			JunLispList propkey;
			id = ((JunLispCons) element).head();
			propkey = (JunLispList) getprop_key_(id, $("fsubr"));

			if (!(propkey.nullList())) {
				list = JunLispCons.Head_tail_(id, list);
			}
			element = ((JunLispCons) element).tail();
		}
		return list;
	}

	/**
	 * this function pre-define how to evaluate a lisp gc statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrGc() {
		System.gc();
		// ObjectMemory globalCompactingGC.
		return $("t");
	}

	/**
	 * this function pre-define how to evaluate a lisp &gt;= statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrGe_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];

		if (a1 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a1) + " for >= ");
		}
		if (a2 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a2) + " for >= ");
		}
		if ((a1 instanceof Number) && (a2 instanceof Number)) {
			if (((Number) a1).longValue() >= ((Number) a2).longValue()) {
				return $("t");
			}
		}
		if ((a1 instanceof String) && (a2 instanceof String)) {
			if (((String) a1).compareTo((String) a2) >= 0) {
				return $("t");
			}
		}

		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp gensym statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrGensym() {
		try {
			Thread.sleep(1);
		} catch (InterruptedException e) {
		}

		long nMilliseconds = System.currentTimeMillis();
		String string = "#id" + nMilliseconds;
		return $(string);
	}

	/**
	 * this function pre-define how to evaluate a lisp getprop statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrGetprop_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof StSymbol)) {
			return fatal_("unexpected argument " + printString_(a1) + " for getprop");
		}
		if (!(a2 instanceof StSymbol)) {
			return fatal_("unexpected argument " + printString_(a2) + " for getprop");
		}
		return getprop_key_((StSymbol) a1, (StSymbol) a2);
	}

	/**
	 * this function pre-define how to evaluate a lisp "&gt;" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrGt_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (a1 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a1) + " for > ");
		}
		if (a2 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a2) + " for > ");
		}
		if ((a1 instanceof Number) && (a2 instanceof Number)) {
			if (((Number) a1).longValue() > ((Number) a2).longValue()) {
				return $("t");
			}
		}
		if ((a1 instanceof String) && (a2 instanceof String)) {
			if (((String) a1).compareTo((String) a2) > 0) {
				return $("t");
			}
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp integerp statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrIntegerp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof Integer) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp last statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrLast_(Object[] arguArray) {
		Object list = arguArray[0];
		if (list instanceof JunLispCons) {
			return ((JunLispCons) list).last();
		}
		return fatal_("unexpected argument " + printString_(list) + " for last");
	}

	/**
	 * this function pre-define how to evaluate a lisp "&lt;=" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrLe_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (a1 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a1) + " for <= ");
		}
		if (a2 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a2) + " for <= ");
		}
		if ((a1 instanceof Number) && (a2 instanceof Number)) {
			if (((Number) a1).longValue() <= ((Number) a2).longValue()) {
				return $("t");
			}
		}
		if ((a1 instanceof String) && (a2 instanceof String)) {
			if (((String) a1).compareTo((String) a2) <= 0) {
				return $("t");
			}
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp length statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrLength_(Object[] arguArray) {
		Object list = arguArray[0];
		if (list instanceof JunLispCons) {
			return new Integer(((JunLispCons) list).length());
		}
		return fatal_("unexpected argument " + printString_(list) + " for length");
	}

	/**
	 * this function pre-define how to evaluate a lisp listp statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrListp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof JunLispList) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp "&lt;" statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrLt_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (a1 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a1) + " for < ");
		}
		if (a2 instanceof JunLispList) {
			return fatal_("unexpected argument " + printString_(a2) + " for < ");
		}
		if ((a1 instanceof Number) && (a2 instanceof Number)) {
			if (((Number) a1).longValue() < ((Number) a2).longValue()) {
				return $("t");
			}
		}
		if ((a1 instanceof String) && (a2 instanceof String)) {
			if (((String) a1).compareTo((String) a2) < 0) {
				return $("t");
			}
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp member statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrMember_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a2 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a2) + " for member");
		}
		return ((JunLispCons) a2).member_(a1);
	}

	/**
	 * this function pre-define how to evaluate a lisp memq statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrMemq_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a2 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a2) + " for memq");
		}
		return ((JunLispCons) a2).memq_(a1);
	}

	/**
	 * this function pre-define how to evaluate a lisp nconc statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrNconc_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a1) + " for nconc");
		}
		((JunLispCons) a1).nconc_((JunLispList) a2);
		return a1;
	}

	/**
	 * this function pre-define how to evaluate a lisp "~~" or neq statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrNeq_(Object[] arguArray) {
		Object bool;
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (a1 != a2) {
			bool = $("t");
		} else {
			bool = JunLispNil.NullList();
		}
		return bool;
	}

	/**
	 * this function pre-define how to evaluate a lisp "~=" or nequal statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrNequal_(Object[] arguArray) {
		Object bool;
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!a1.equals(a2)) {
			bool = $("t");
		} else {
			bool = JunLispNil.NullList();
		}
		return bool;
	}

	/**
	 * this function pre-define how to evaluate a lisp nth statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrNth_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof Number)) {
			return fatal_("unexpected argument " + printString_(a1) + " for nth");
		}
		if (!(a2 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a2) + " for nth");
		}
		return ((JunLispCons) a2).nth_(((Number) a1).intValue());
	}

	/**
	 * this function pre-define how to evaluate a lisp null statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrNull_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof JunLispNil) { //isMemberOf:  final class
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp numberp statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrNumberp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof Number) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp oblist statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public JunLispList subrOblist() {
		JunLispList list = JunLispNil.NullList();
		Object[] collection = lispTable.identifiers();
		int nSize = collection.length;
		for (int nIndex = nSize - 1; nIndex >= 0; nIndex--) {
			Object each = collection[nIndex];
			list = JunLispCons.Head_tail_(each, list);
		}
		return list;
	}

	/**
	 * Subroutine call - pp
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions
	 */
	public Object subrPp_(Object[] arguArray) {
		JunLispCons a = (JunLispCons) arguArray[0];
		String pretty = a.ppString();
		textCollector.print(pretty);
		return a;
	}

	/**
	 * Subroutine call - princ
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions
	 */
	public Object subrPrinc_(Object[] arguArray) {
		Object a = arguArray[0];
		if (a instanceof String) {
			textCollector.print(a);
		} else {
			textCollector.print(this.printString_(a));
		}
		return a;
	}

	/**
	 * Subroutine call - print
	 * 
	 * @return java.lang.Object
	 * @param arguArray java.lang.Object[]
	 * @category subr functions
	 */
	public Object subrPrint_(Object[] arguArray) {
		Object a = this.subrPrinc_(arguArray);
		textCollector.println();
		return a;
	}

	/**
	 * this function pre-define how to evaluate a lisp putprop statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrPutprop_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		Object a3 = arguArray[2];
		if (!(a1 instanceof StSymbol)) {
			return fatal_("unexpected argument " + printString_(a1) + " for putprop");
		}
		if (!(a2 instanceof StSymbol)) {
			return fatal_("unexpected argument " + printString_(a2) + " for putprop");
		}
		return putprop_key_value_(a1, a2, a3);
	}

	/**
	 * this function pre-define how to evaluate a lisp read statement to an Object
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrRead() {
		String message = "                " + JunSystem.$String("Input an S expression.") + "                ";
		String answer = JunDialog.Request_(message, "");
		if (answer.length() == 0) {
			return JunLispNil.NullList();
		}
		return JunLispParser.Parse_(new StReadStream(answer));
	}

	/**
	 * this function pre-define how to evaluate a lisp remprop statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrRemprop_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof StSymbol)) {
			return fatal_("unexpected argument " + printString_(a1) + " for remprop");
		}
		if (!(a2 instanceof StSymbol)) {
			return fatal_("unexpected argument " + printString_(a2) + " for remprop");
		}
		return remprop_key_(a1, a2);
	}

	/**
	 * this function pre-define how to evaluate a lisp reverse statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrReverse_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (!(a1 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a1) + " for reverse");
		}
		return ((JunLispCons) a1).reverse();
	}

	/**
	 * this function pre-define how to evaluate a lisp rplaca statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrRplaca_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a2) + " for rplaca");
		}
		((JunLispCons) a1).rplaca_(a2);
		return a1;
	}

	/**
	 * this function pre-define how to evaluate a lisp rplacd statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrRplacd_(Object[] arguArray) {
		Object a1 = arguArray[0];
		Object a2 = arguArray[1];
		if (!(a1 instanceof JunLispCons)) {
			return fatal_("unexpected argument " + printString_(a2) + " for rplacd");
		}
		((JunLispCons) a1).rplacd_(a2);
		return a1;
	}

	/**
	 * this function pre-define how to evaluate a lisp stringp statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrStringp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof String) { //and: [(a1 isKindOf: Symbol) not])
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * this function pre-define how to evaluate a lisp subrs statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrSubrs() {
		JunLispList list = JunLispNil.NullList();
		JunLispList lispcons = subrOblist();
		JunLispList reverse = (JunLispList) (((JunLispCons) lispcons).reverse());
		Object element = reverse;

		while (element instanceof JunLispCons) {
			Object id;
			JunLispList propkey;
			id = ((JunLispCons) element).head();
			propkey = (JunLispList) getprop_key_(id, $("subr"));
			if (!(propkey.nullList())) {
				list = JunLispCons.Head_tail_(id, list);
			}
			element = ((JunLispCons) element).tail();
		}

		return list;
	}

	/**
	 * this function pre-define how to evaluate a lisp symbolp statement to an Object.
	 * 
	 * @return java.lang.Object
	 * @param arguArray  java.lang.Object[]
	 * @category subr functions 
	 */
	public Object subrSymbolp_(Object[] arguArray) {
		Object a1 = arguArray[0];
		if (a1 instanceof StSymbol) {
			return $("t");
		}
		return JunLispNil.NullList();
	}

	/**
	 * Subroutine call - terpri
	 * 
	 * @return java.lang.Object
	 * @category subr functions
	 */
	public Object subrTerpri() {
		textCollector.println();
		return $("t");
	}

	/**
	 * Answer my current text collector.
	 *
	 * @return java.io.PrintStream
	 * @category accessing
	 */
	public PrintStream textCollector() {
		return textCollector;
	}

	/**
	 * Set my text collector.
	 *
	 * @param newTextCollector java.io.PrintStream
	 * @category accessing
	 */
	public void textCollector_(PrintStream newTextCollector) {
		textCollector = newTextCollector;
	}

	/**
	 * Answer my text value.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category adaptor
	 */
	public StValueHolder textValue() {
		if (textValue == null) {
			textValue = new StValueHolder(new String());
		}
		return textValue;
	}

	/**
	 * Unbind
	 * 
	 * @category shallow binding
	 */
	public void unbind() {
		StAssociation assoc;

		//assoc := bindStack removeLast.
		assoc = (StAssociation) (bindStack.pop());
		while (assoc.key() != null) {
			if (assoc.value() != null) {
				putprop_key_value_(assoc.key(), $("apval"), assoc.value());
			} else {
				remprop_key_(assoc.key(), $("apval"));
			}
			assoc = (StAssociation) (bindStack.pop());
		}
	}

	/**
	 * Evaluate the JunLispList.
	 * 
	 * @return java.lang.Object
	 * @param sExpression jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category evaluating 
	 */
	protected Object listEvaluate_(JunLispList sExpression) {
		if (sExpression.nullList()) {
			return sExpression;
		}

		Object funcName = ((JunLispCons) sExpression).head();
		Object arguList = ((JunLispCons) sExpression).tail();
		Object funcBody;

		if (funcName instanceof JunLispCons) {
			funcBody = funcName;
			JunLispCons lispcons = (JunLispCons) funcName;
			if (lispcons.head() instanceof StSymbol) {
				StSymbol symbol = (StSymbol) lispcons.head();
				if (symbol == $("lambda")) {
					funcBody = JunLispCons.Head_tail_($("lambda"), (JunLispList) funcBody);
					return exprEval_arguList_((JunLispList) funcBody, (JunLispList) arguList);
				}
				if (symbol == $("nlambda")) {
					funcBody = JunLispCons.Head_tail_($("nlambda"), (JunLispList) funcBody);
					return fexprEval_arguList_((JunLispList) funcBody, (JunLispList) arguList);
				}
			}
			return this.fatal_("unexpected function " + this.printString_(funcBody));
		}
		if (!(funcName instanceof StSymbol)) {
			return this.fatal_("null function " + this.printString_(funcName));
		}
		funcBody = getprop_key_(funcName, $("fexpr"));
		if (!(funcBody instanceof JunLispNil)) {
			return fexprEval_arguList_((JunLispList) funcBody, (JunLispList) arguList);
		}
		funcBody = getprop_key_(funcName, $("expr"));
		if (!(funcBody instanceof JunLispNil)) {
			return exprEval_arguList_((JunLispList) funcBody, (JunLispList) arguList);
		}
		funcBody = getprop_key_(funcName, $("fsubr"));
		if (!(funcBody instanceof JunLispNil)) {
			return fsubrEval_arguList_((JunLispList) funcBody, (JunLispList) arguList);
		}
		funcBody = getprop_key_(funcName, $("subr"));
		if (!(funcBody instanceof JunLispNil)) {
			return subrEval_arguList_((JunLispList) funcBody, (JunLispList) arguList);
		}
		return this.fatal_("undefined function " + this.printString_(funcName));
	}

	/**
	 * Convert the object to the printable string.
	 * 
	 * @return java.lang.String
	 * @param anObject java.lang.Object
	 * @category printing 
	 */
	protected String printString_(Object anObject) {
		return PrintString_(anObject);
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Lisp");
	}

}
