package jp.co.sra.jun.goodies.image.streams;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import javax.imageio.IIOException;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;
import jp.co.sra.smalltalk.StImage;

/**
 * JunJpegImageStream class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2000/11/12 (by Mitsuhiro Asada)
 *  @updated   2005/08/15 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunJpegImageStream.java,v 8.11 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunJpegImageStream extends JunImageStream {
	protected float quality = Float.NaN;
	public static final String DefaultImageExtension = "jpg";

	/**
	 * Constructor with an input stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#On_(java.io.InputStream)
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		return On_(_CreateImageStream(), stream);
	}

	/**
	 * Constructor with an output stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#On_(java.io.OutputStream)
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		return On_(_CreateImageStream(), stream);
	}

	/**
	 * Answer an instance of jpeg image stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @category Instance creation
	 */
	protected static JunImageStream _CreateImageStream() {
		return new JunJpegImageStream();
	}

	/**
	 * Read the image from the input stream.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#nextImage()
	 * @category accessing
	 */
	public StImage nextImage() throws IOException {
		if (inStream == null) {
			throw new IIOException("the inStream is null.");
		}

		ImageReader imageReader = null;
		ImageInputStream imageStream = null;
		try {
			imageReader = (ImageReader) ImageIO.getImageReadersBySuffix(this.defaultImageExtension()).next();
			imageStream = ImageIO.createImageInputStream(inStream);
			imageReader.setInput(imageStream);
			imageObject = new StImage(imageReader.read(0));
		} finally {
			if (imageStream != null) {
				imageStream.flush();
				imageStream.close();
			}
			if (imageReader != null) {
				imageReader.reset();
				imageReader.dispose();
			}
		}
		return imageObject;
	}

	/**
	 * Write the image on the output stream.
	 *
	 * @param newImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#nextPutImage_(jp.co.sra.smalltalk.StImage)
	 * @category accessing
	 */
	public void nextPutImage_(StImage newImage) throws IOException {
		if (outStream == null) {
			throw new IIOException("the outStream is null.");
		}

		BufferedImage anImage = new BufferedImage(newImage.width(), newImage.height(), BufferedImage.TYPE_3BYTE_BGR);
		anImage.setRGB(0, 0, newImage.width(), newImage.height(), newImage.getPixels(), 0, newImage.width());

		ImageWriter imageWriter = null;
		ImageOutputStream imageStream = null;
		try {
			imageWriter = (ImageWriter) ImageIO.getImageWritersBySuffix(this.defaultImageExtension()).next();
			imageStream = ImageIO.createImageOutputStream(outStream);
			imageWriter.setOutput(imageStream);
			ImageWriteParam writeParam = imageWriter.getDefaultWriteParam();
			writeParam.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
			if (writeParam.canWriteCompressed() && writeParam.getCompressionQuality() != this.quality()) {
				writeParam.setCompressionQuality(this.quality());
				imageWriter.write(null, new IIOImage(anImage, null, null), writeParam);
			} else {
				imageWriter.write(anImage);
			}
		} finally {
			if (imageStream != null) {
				imageStream.flush();
				imageStream.close();
			}
			if (imageWriter != null) {
				imageWriter.reset();
				imageWriter.dispose();
			}
		}
	}

	/**
	 * Answer the receiver's compression quality value.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float quality() {
		if (Float.isNaN(quality)) {
			quality = this.defaultQuality();
		}
		return quality;
	}

	/**
	 * Set the receiver's compression quality value.
	 * 
	 * @param normalizedValue float
	 * @category accessing
	 */
	public void quality_(float normalizedValue) {
		quality = Math.min(Math.max(0f, normalizedValue), 1f);
	}

	/**
	 * Answer the receiver's image file extension.
	 *
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#defaultImageExtension()
	 * @category defaults
	 */
	public String defaultImageExtension() {
		return DefaultImageExtension;
	}

	/**
	 * Answer the default compression quality value.
	 * 
	 * @return float
	 * @category defaults
	 */
	public float defaultQuality() {
		return 0.75f;
	}
}
