package jp.co.sra.jun.goodies.image.streams;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Iterator;

import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;

import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.jun.system.support.JunTestExamples;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * JunImageStreamTestExamples class
 * 
 *  @author    m-asada
 *  @created   2005/08/15 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageStreamTestExamples.java,v 8.14 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunImageStreamTestExamples extends JunTestExamples {

	/**
	 * Example: Display the image format that an ImageIO class supports.  
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example0() {
		String[] formatNames = ImageIO.getReaderFormatNames();
		System.out.println("ReaderFormatNames:");
		for (int i = 0; i < formatNames.length; i++) {
			System.out.println(formatNames[i]);
		}

		formatNames = ImageIO.getWriterFormatNames();
		System.out.println("WriterFormatNames:");
		for (int i = 0; i < formatNames.length; i++) {
			System.out.println(formatNames[i]);
		}

		return true;
	}

	/**
	 * Example: JPEG image file read/write.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example1_JPEG() {
		StImage anImage = StImage._FromUser();
		File file = RequestFile(JunJpegImageStream.DefaultImageExtension);
		if (file == null) {
			return false;
		}

		JunImageStream outStream = null;
		try {
			outStream = JunJpegImageStream.On_(new FileOutputStream(file));
			outStream.nextPutImage_(anImage);
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (outStream != null) {
				try {
					outStream.flush();
					outStream.close();
				} catch (IOException ex) {
				}
				outStream = null;
			}
		}

		JunImageStream inStream = null;
		try {
			inStream = JunJpegImageStream.On_(new FileInputStream(file));
			StImage image = inStream.nextImage();
			image._display();
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (inStream != null) {
				try {
					inStream.close();
				} catch (IOException ex) {
				}
				inStream = null;
			}
		}

		return true;
	}

	/**
	 * Example: PNG image file read/write.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example2_PNG() {
		StImage anImage = StImage._FromUser();
		File file = RequestFile(JunPngImageStream.DefaultImageExtension);
		if (file == null) {
			return false;
		}

		JunImageStream outStream = null;
		try {
			outStream = JunPngImageStream.On_(new FileOutputStream(file));
			outStream.nextPutImage_(anImage);
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (outStream != null) {
				try {
					outStream.flush();
					outStream.close();
				} catch (IOException ex) {
				}
				outStream = null;
			}
		}

		JunImageStream inStream = null;
		try {
			inStream = JunPngImageStream.On_(new FileInputStream(file));
			StImage image = inStream.nextImage();
			image._display();
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (inStream != null) {
				try {
					inStream.close();
				} catch (IOException ex) {
				}
				inStream = null;
			}
		}

		return true;
	}

	/**
	 * Example: BMP image file read/write.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example3_BMP() {
		StImage anImage = StImage._FromUser();
		File file = RequestFile(JunBmpImageStream.DefaultImageExtension);
		if (file == null) {
			return false;
		}

		JunImageStream outStream = null;
		try {
			outStream = JunBmpImageStream.On_(new FileOutputStream(file));
			outStream.nextPutImage_(anImage);
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (outStream != null) {
				try {
					outStream.flush();
					outStream.close();
				} catch (IOException ex) {
				}
				outStream = null;
			}
		}

		JunImageStream inStream = null;
		try {
			inStream = JunBmpImageStream.On_(new FileInputStream(file));
			StImage image = inStream.nextImage();
			image._display();
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (inStream != null) {
				try {
					inStream.close();
				} catch (IOException ex) {
				}
				inStream = null;
			}
		}

		return true;
	}

	/**
	 * Example: GIF image file read.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example4_GIF() {
		File file = RequestFile(JunGifImageStream.DefaultImageExtension);
		if (file == null || file.exists() == false) {
			return false;
		}

		JunImageStream inStream = null;
		try {
			inStream = JunGifImageStream.On_(new FileInputStream(file));
			StImage image = inStream.nextImage();
			image._display();
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return false;
		} finally {
			if (inStream != null) {
				try {
					inStream.close();
				} catch (IOException ex) {
				}
				inStream = null;
			}
		}

		return true;
	}

	/**
	 * Example5: GIF image file write.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example5_GIF_Write() throws IOException {
		StImage image = StImage._FromUser();

		File file = new File("zzz.gif");
		JunImageStream stream = null;
		try {
			stream = JunGifImageStream.On_(new FileOutputStream(file));

			JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
			try {
				cursor._show();

				stream.nextPutImage_(image);
			} finally {
				cursor._restore();
			}
		} finally {
			if (stream != null) {
				stream.close();
			}
		}

		return true;
	}

	/**
	 * Example6: GIF image file write.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example6_GIF_Write() throws IOException {
		StImage image = RequestImage(JunGifImageStream.DefaultImageExtension);
		if (image == null) {
			return false;
		}

		File file = new File("zzzzz.gif");
		JunImageStream stream = null;
		try {
			stream = JunGifImageStream.On_(new FileOutputStream(file));

			JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
			try {
				cursor._show();

				stream.nextPutImage_(image);
			} finally {
				cursor._restore();
			}
		} finally {
			if (stream != null) {
				stream.close();
			}
		}

		return true;
	}

	/**
	 * Test the ImageIO for reading GIF animation.
	 * 
	 * @return boolean
	 * @throws IOException 
	 * @category Testing
	 */
	protected static boolean Test_ImageIO() throws IOException {
		File file = RequestFile(JunGifImageStream.DefaultImageExtension);
		if (file == null || file.exists() == false) {
			return false;
		}

		FileInputStream in = new FileInputStream(file);
		FileChannel fc = in.getChannel();
		ByteBuffer bb = ByteBuffer.allocate(1024);
		fc.read(bb);
		fc.position(0);

		ImageInputStream imageInputStream = null;
		try {
			imageInputStream = ImageIO.createImageInputStream(in);

			ImageReader imageReader = null;
			try {
				// Iterator i = ImageIO.getImageReadersBySuffix(JunGifImageStream.DefaultImageExtension);
				Iterator i = ImageIO.getImageReaders(imageInputStream);
				if (i.hasNext() == false) {
					return false;
				}

				imageReader = (ImageReader) i.next();
				imageReader.setInput(imageInputStream);

				int num = imageReader.getNumImages(true);
				System.out.println(num);
				IIOMetadata metadata = imageReader.getStreamMetadata();
				String formatName = metadata.getNativeMetadataFormatName();
				Node node = metadata.getAsTree(formatName);
				DisplayMetadata(node);

				metadata = imageReader.getImageMetadata(0);
				formatName = metadata.getNativeMetadataFormatName();
				node = metadata.getAsTree(formatName);
				DisplayMetadata(node);

				ImageReadParam param = imageReader.getDefaultReadParam();
				BufferedImage bufferedImage = imageReader.read(0, param);
				StImage image = new StImage(bufferedImage);
				image._display();

			} finally {
				if (imageReader != null) {
					imageReader.dispose();
				}
			}
		} finally {
			if (imageInputStream != null) {
				imageInputStream.close();
			}
		}

		return true;
	}

	/**
	 * Request a file with the specified extension.
	 * 
	 * @param fileExtension java.io.String
	 * @return java.io.File
	 * @category Utilities
	 */
	protected static File RequestFile(String fileExtension) {
		fileExtension = fileExtension.toLowerCase();
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, fileExtension), new String[] { "*." + fileExtension, "*." + fileExtension.toUpperCase() }) };
		return JunFileRequesterDialog.RequestFile($String("Select a <1p> file.", null, fileExtension), new File(JunSystem.DefaultBaseName() + "." + fileExtension), fileTypes, fileTypes[0]);
	}

	/**
	 * Request a file with the specified extension and read the image.
	 * 
	 * @param fileExtension java.lang.String
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Utilities
	 */
	protected static StImage RequestImage(String fileExtension) {
		File file = RequestFile(fileExtension);
		if (file == null || file.exists() == false) {
			return null;
		}

		StImage anImage = null;
		JunImageStream inStream = null;
		try {
			inStream = JunGifImageStream.On_(new FileInputStream(file));
			anImage = inStream.nextImage();
		} catch (IOException e) {
			System.out.println(e.getMessage());
			return null;
		} finally {
			if (inStream != null) {
				try {
					inStream.close();
				} catch (IOException ex) {
				}
				inStream = null;
			}
		}

		return anImage;
	}

	/**
	 * A utility method to display the metadata.
	 * 
	 * @param root org.w3c.dom.Node
	 * @category Utilities
	 */
	private static void DisplayMetadata(Node root) {
		DisplayMetadata(root, 0);
	}

	/**
	 * A utility method to display the metadata.
	 * 
	 * @param root org.w3c.dom.Node
	 * @category Utilities
	 */
	private static void DisplayMetadata(Node node, int level) {
		Indent(level); // emit open tag
		System.out.print("<" + node.getNodeName());
		NamedNodeMap map = node.getAttributes();
		if (map != null) { // print attribute values
			int length = map.getLength();
			for (int i = 0; i < length; i++) {
				Node attr = map.item(i);
				System.out.print(" " + attr.getNodeName() + "=\"" + attr.getNodeValue() + "\"");
			}
		}

		Node child = node.getFirstChild();
		if (child != null) {
			System.out.println(">"); // close current tag
			while (child != null) { // emit child tags recursively
				DisplayMetadata(child, level + 1);
				child = child.getNextSibling();
			}
			Indent(level); // emit close tag
			System.out.println("</" + node.getNodeName() + ">");
		} else {
			System.out.println("/>");
		}
	}

	/**
	 * A utility method to print the level of the indentation.
	 * 
	 * @param level int
	 * @category Utilities
	 */
	private static void Indent(int level) {
		for (int i = 0; i < level; i++) {
			System.out.print("  ");
		}
	}

	/**
	 * Execute all examples.
	 * 
	 * @param args an array of command-line arguments
	 * @category Main
	 */
	public static void main(String[] args) {
		new JunImageStreamTestExamples();
	}

}
