package jp.co.sra.jun.goodies.image.streams;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.imageio.IIOException;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;

import jp.co.sra.smalltalk.StImage;

/**
 * JunGifImageStream class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2000/11/22 (by Mitsuhiro Asada)
 *  @updated   2004/08/12 (by Mitsuhiro Asada)
 *  @updated   2005/08/15 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGifImageStream.java,v 8.13 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunGifImageStream extends JunImageStream {

	public static final String DefaultImageExtension = "gif";

	/**
	 * Constructor with an input stream.
	 *
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream 
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#On_(java.io.InputStream)
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		return On_(new JunGifImageStream(), stream);
	}

	/**
	 * Constructor with an output stream.
	 *
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream 
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#On_(java.io.OutputStream)
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		String[] formatNames = ImageIO.getWriterFormatNames();
		for (int i = 0; i < formatNames.length; i++) {
			if (DefaultImageExtension.equals(formatNames[i])) {
				return On_(new JunGifImageStream(), stream);
			}
		}
		return On_(new JunSraGifImageStream(), stream);
	}

	/**
	 * Read the image from the input stream.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#nextImage()
	 * @category accessing
	 */
	public StImage nextImage() throws IOException {
		if (inStream == null) {
			throw new IIOException("the inStream is null.");
		}

		ImageReader imageReader = null;
		ImageInputStream imageStream = null;
		try {
			imageReader = (ImageReader) ImageIO.getImageReadersBySuffix(this.defaultImageExtension()).next();
			imageStream = ImageIO.createImageInputStream(inStream);
			imageReader.setInput(imageStream);
			imageObject = new StImage(imageReader.read(0));
		} finally {
			if (imageStream != null) {
				imageStream.flush();
				imageStream.close();
			}
			if (imageReader != null) {
				imageReader.reset();
				imageReader.dispose();
			}
		}
		return imageObject;
	}

	/**
	 * Write the image on the output stream.
	 *
	 * @param newImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#nextPutImage_(jp.co.sra.smalltalk.StImage)
	 * @category accessing
	 */
	public void nextPutImage_(StImage newImage) throws IOException {
		if (outStream == null) {
			throw new IIOException("the outStream is null.");
		}

		ImageWriter imageWriter = null;
		ImageOutputStream imageStream = null;
		try {
			imageWriter = (ImageWriter) ImageIO.getImageWritersBySuffix(this.defaultImageExtension()).next();
			imageStream = ImageIO.createImageOutputStream(outStream);
			imageWriter.setOutput(imageStream);
			imageWriter.write(newImage.image());
		} finally {
			if (imageStream != null) {
				imageStream.flush();
				imageStream.close();
			}
			if (imageWriter != null) {
				imageWriter.reset();
				imageWriter.dispose();
			}
		}
	}

	/**
	 * Answer the receiver's image file extension.
	 *
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#defaultImageExtension()
	 * @category defaults
	 */
	public String defaultImageExtension() {
		return DefaultImageExtension;
	}

}
