package jp.co.sra.jun.goodies.image.framework;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;

import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StOpaqueImage;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunImageDisplayController class
 * 
 *  @author    m-asada
 *  @created   2006/04/06 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun632 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageDisplayController.java,v 8.11 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunImageDisplayController extends JunAbstractController {
	protected Jun2dPoint oldPoint;
	protected Image oldImage;
	protected Rectangle oldRectangle;
	protected Cursor _cursor;

	/**
	 * Create a new instance of <code>JunImageDisplayController</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunImageDisplayController() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StController#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		super.initialize();
		oldPoint = null;
		oldImage = null;
		oldRectangle = null;
	}

	/**
	 * Add the myself as a listener of the view.
	 * 
	 * @param newView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	protected void buildListener(StView newView) {
		this.getView()._canvasComponent().addMouseListener(this);
		this.getView()._canvasComponent().addMouseMotionListener(this);
	}

	/**
	 * Remove references to objects that may refer to the receiver. 
	 * Remove the listeners of the view here as well.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	public void release() {
		super.release();
		this.getView()._canvasComponent().addMouseListener(this);
		this.getView()._canvasComponent().removeMouseListener(this);
	}

	/**
	 * Display cursor point on the receiver's view.
	 * 
	 * @param evt java.awt.event.MouseEvent 
	 * @category manipulating
	 */
	public void displayCursorPoint_(MouseEvent evt) {
		if (this.getModel().image() == null) {
			return;
		}
		String displayString = null;
		Component canvas = this.getView()._canvasComponent();
		Graphics graphicsContext = canvas.getGraphics();
		try {
			Rectangle effectiveArea = this.getModel().image().bounds();
			effectiveArea.setSize(effectiveArea.width + 1, effectiveArea.height + 1);
			Jun2dPoint newPoint = this.normalizedCursorPoint_(evt);
			if (newPoint.equals(oldPoint) == false) {
				displayString = this.displayStringForPoint_(newPoint, evt);
				if (oldImage != null) {
					graphicsContext.drawImage(oldImage, oldRectangle.x, oldRectangle.y, oldRectangle.width, oldRectangle.height, canvas);
				}

				oldRectangle = this.displayRectangleForString_(displayString, evt);
				oldImage = new BufferedImage(oldRectangle.width, oldRectangle.height, BufferedImage.TYPE_3BYTE_BGR);
				Graphics aGraphics = oldImage.getGraphics();
				try {
					aGraphics.translate(-oldRectangle.x, -oldRectangle.y);
					canvas.paint(aGraphics);
				} finally {
					if (aGraphics != null) {
						aGraphics.dispose();
						aGraphics = null;
					}
				}

				if (effectiveArea.contains(evt.getPoint())) {
					graphicsContext.setColor(Color.white);
					graphicsContext.fillRect(oldRectangle.x, oldRectangle.y, oldRectangle.width, oldRectangle.height);
					graphicsContext.setColor(Color.black);
					graphicsContext.drawString(displayString, oldRectangle.x + 2, oldRectangle.y + graphicsContext.getFontMetrics().getAscent());
				}
			}
			oldPoint = newPoint;
			if (displayString != null && displayString.length() > 0) {
				System.out.println(displayString);
			}
		} catch (Exception e) {
			System.out.println(displayString);
			oldImage = null;
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
				graphicsContext = null;
			}
		}
	}

	/**
	 * Answer the display rectangle for the specified string on the receiver's view.
	 * 
	 * @param displayString java.lang.String
	 * @param evt java.awt.event.MouseEvent 
	 * @return java.awt.Rectangle
	 * @category manipulating
	 */
	public Rectangle displayRectangleForString_(String displayString, MouseEvent evt) {
		oldRectangle = null;
		Point cursorPoint = evt.getPoint();
		Rectangle viewportBounds = this.getView()._getViewportBounds();
		FontMetrics fontMetrics = this.getView()._canvasComponent().getFontMetrics(this.getView()._canvasComponent().getFont());
		Rectangle aRectangle = new Rectangle(0, 0, fontMetrics.stringWidth(displayString) + 4, fontMetrics.getHeight());
		if (cursorPoint.x <= viewportBounds.getCenterX() && cursorPoint.y <= viewportBounds.getCenterY()) {
			aRectangle.translate(cursorPoint.x - aRectangle.x, cursorPoint.y - aRectangle.y);
			aRectangle.translate(3, 3);
		}
		if (cursorPoint.x > viewportBounds.getCenterX() && cursorPoint.y > viewportBounds.getCenterY()) {
			aRectangle.translate(cursorPoint.x - (aRectangle.x + aRectangle.width), cursorPoint.y - (aRectangle.y + aRectangle.height));
			aRectangle.translate(-2, -2);
		}
		if (cursorPoint.x <= viewportBounds.getCenterX() && cursorPoint.y > viewportBounds.getCenterY()) {
			aRectangle.translate(cursorPoint.x - aRectangle.x, cursorPoint.y - (aRectangle.y + aRectangle.height));
			aRectangle.translate(3, -2);
		}
		if (cursorPoint.x > viewportBounds.getCenterX() && cursorPoint.y <= viewportBounds.getCenterY()) {
			aRectangle.translate(cursorPoint.x - (aRectangle.x + aRectangle.width), cursorPoint.y - aRectangle.y);
			aRectangle.translate(-2, 3);
		}
		return aRectangle;
	}

	/**
	 * Answer the display string from the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param evt java.awt.event.MouseEvent 
	 * @return java.lang.String
	 * @category manipulating
	 */
	public String displayStringForPoint_(Jun2dPoint aPoint, MouseEvent evt) {
		String displayString = null;
		if (evt.isShiftDown()) {
			displayString = String.valueOf(Math.round(aPoint.x() * this.getModel().image().bounds().width)) + " , " + String.valueOf(Math.round(aPoint.y() * this.getModel().image().bounds().height));
		} else {
			displayString = new Float(aPoint.x()).toString() + " , " + new Float(aPoint.y()).toString();
		}
		return displayString;
	}

	/**
	 * Answer the normalized cursor point.
	 * 
	 * @param evt java.awt.event.MouseEvent 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public Jun2dPoint normalizedCursorPoint_(MouseEvent evt) {
		StDisplayable anImage = this.getModel().image();
		if (anImage instanceof StOpaqueImage) {
			anImage = ((StOpaqueImage) anImage).figure();
		}
		Point aPoint = evt.getPoint();
		aPoint.x = Math.min(Math.max(aPoint.x, 0), anImage.bounds().width);
		aPoint.y = Math.min(Math.max(aPoint.y, 0), anImage.bounds().height);
		return new Jun2dPoint(aPoint.x / (double) anImage.bounds().width, aPoint.y / (double) anImage.bounds().height);
	}

	/**
	 * Answer the receiver's model as JunImageDisplayModel.
	 * 
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel
	 * @category model accessing
	 */
	public JunImageDisplayModel getModel() {
		return (JunImageDisplayModel) this.view().model();
	}

	/**
	 * Invoked when a mouse button has been pressed on the view.
	 * 
	 * @param evt java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent evt) {
		if (evt.isPopupTrigger() || evt.isMetaDown()) {
			super.mousePressed(evt);
			return;
		}

		_cursor = this.getView().toComponent().getCursor();
		this.getView().toComponent().setCursor(JunCursors.CrossCursor());

		oldPoint = null;
		oldImage = null;
		oldRectangle = null;
		this.displayCursorPoint_(evt);
	}

	/**
	 * Invoked when a mouse button has been released on the view.
	 * 
	 * @param evt java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent evt) {
		if (evt.isPopupTrigger() || evt.isMetaDown()) {
			super.mouseReleased(evt);
			return;
		}

		if (oldImage != null) {
			this.getView()._canvasComponent().getGraphics().drawImage(oldImage, oldRectangle.x, oldRectangle.y, oldRectangle.width, oldRectangle.height, this.getView()._canvasComponent());
		}
		oldPoint = null;
		oldImage = null;
		oldRectangle = null;

		this.getView().toComponent().setCursor(_cursor);
	}

	/**
	 * Invoked when a mouse is dragged on the view.
	 *
	 * @param evt java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent evt) {
		if (evt.isPopupTrigger() || evt.isMetaDown()) {
			super.mouseDragged(evt);
			return;
		}

		this.displayCursorPoint_(evt);
	}

	/**
	 * Answer the receiver's view as JunImageDisplayView.
	 * 
	 * @return jp.co.sra.jun.goodies.image.framework.JunImageDisplayView
	 * @category view accessing
	 */
	public JunImageDisplayView getView() {
		return (JunImageDisplayView) this.view();
	}
}
