package jp.co.sra.jun.goodies.gauge;

import jp.co.sra.smalltalk.*;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.jun.goodies.track.JunSimpleSliderModel;

/**
 * JunSimpleGaugeModel class
 * 
 *  @author    Hoshi Takanori
 *  @created   2003/08/28 (by Hoshi Takanori)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun433 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSimpleGaugeModel.java,v 8.10 2008/02/20 06:31:34 nisinaka Exp $
 */
public class JunSimpleGaugeModel extends JunApplicationModel implements StBlockValued {

	protected static final double DefaultFromValue = 0;
	protected static final double DefaultToValue = 1;
	protected static final double DefaultRoundValue = 0.01;

	protected JunSimpleSliderModel simpleSlider;
	protected double fromValue = DefaultFromValue;
	protected double toValue = DefaultToValue;
	protected double roundValue = DefaultRoundValue;
	protected StBlockClosure valueStringBlock;

	/**
	 * Create a new instance of JunSimpleGaugeModel and initialize it.
	 *
	 * @param aNumber double
	 */
	public JunSimpleGaugeModel(double aNumber) {
		super();
		this.value_(aNumber);
	}

	/**
	 * Create a new instance of JunSimpleGaugeModel and initialize it.
	 *
	 * @param aNumber double
	 * @param from double
	 * @param to double
	 * @param round double
	 */
	public JunSimpleGaugeModel(double aNumber, double from, double to, double round) {
		super();
		fromValue = from;
		toValue = to;
		roundValue = round;
		this.value_(aNumber);
	}

	/**
	 * Initialize the JunSimpleGaugeModel when created.
	 */
	protected void initialize() {
		super.initialize();

		simpleSlider = null;
		fromValue = DefaultFromValue;
		toValue = DefaultToValue;
		roundValue = DefaultRoundValue;
		valueStringBlock = null;
	}

	/**
	 * Answer the simple slider.
	 *
	 * @return jp.co.sra.jun.goodies.track.JunSimpleSliderModel
	 */
	public JunSimpleSliderModel simpleSlider() {
		if (simpleSlider == null) {
			final JunSimpleGaugeModel this_ = this;
			simpleSlider = new JunSimpleSliderModel();
			simpleSlider.compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					this_.changed_($("value"));
					return null;
				}
			});
		}
		return simpleSlider;
	}

	/**
	 * Answer the from value.
	 *
	 * @return double
	 */
	public double fromValue() {
		return fromValue;
	}

	/**
	 * Set the from value to aNumber.
	 *
	 * @param aNumber double
	 */
	public void fromValue_(double aNumber) {
		if (fromValue != aNumber) {
			fromValue = aNumber;
			this.changed_($("value"));
		}
	}

	/**
	 * Answer the to value.
	 *
	 * @return double
	 */
	public double toValue() {
		return toValue;
	}

	/**
	 * Set the to value to aNumber.
	 *
	 * @param aNumber double
	 */
	public void toValue_(double aNumber) {
		if (toValue != aNumber) {
			toValue = aNumber;
			this.changed_($("value"));
		}
	}

	/**
	 * Answer the round value.
	 *
	 * @return double
	 */
	public double roundValue() {
		return roundValue;
	}

	/**
	 * Set the round value to aNumber.
	 *
	 * @param aNumber double
	 */
	public void roundValue_(double aNumber) {
		if (roundValue != aNumber) {
			roundValue = aNumber;
			this.changed_($("value"));
		}
	}

	/**
	 * Answer the step value.
	 *
	 * @return double
	 */
	public double stepValue() {
		return (fromValue <= toValue) ? roundValue : -roundValue;
	}

	/**
	 * Answer the value as double.
	 *
	 * @return double
	 */
	public double doubleValue() {
		double theValue = this.simpleSlider().doubleValue() * (toValue - fromValue);
		return Math.round(theValue / roundValue) * roundValue + fromValue;
	}

	/**
	 * Answer the value.
	 *
	 * @return java.lang.Object
	 */
	public Object value() {
		return new Double(this.doubleValue());
	}

	/**
	 * Set the value to aNumber.
	 *
	 * @param aNumber double
	 */
	public void value_(double aNumber) {
		double theValue = (aNumber - fromValue) / (toValue - fromValue);
		theValue = Math.max(0, Math.min(theValue, 1));
		if (theValue != this.simpleSlider().doubleValue()) {
			this.simpleSlider().value_(theValue);
		}
	}

	/**
	 * Answer the value string.
	 *
	 * @return java.lang.String
	 */
	public String valueString() {
		return this.valueStringBlock().value_(this.value()).toString();
	}

	/**
	 * Answer the value string block.
	 *
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 */
	public StBlockClosure valueStringBlock() {
		if (valueStringBlock == null) {
			valueStringBlock = this.defaultValueStringBlock();
		}
		return valueStringBlock;
	}

	/**
	 * Set the value string block to aBlock.
	 *
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	public void valueStringBlock_(StBlockClosure aBlock) {
		valueStringBlock = aBlock;
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 *
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Answer a default value string block.
	 *
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 */
	protected StBlockClosure defaultValueStringBlock() {
		return new StBlockClosure() {
			public Object value_(Object newValue) {
				return formatValueString(((Double) newValue).doubleValue());
			}
		};
	}

	/**
	 * Answer a formatted value string.
	 *
	 * @param aNumber double
	 * @return java.lang.String
	 */
	protected String formatValueString(double aNumber) {
		if (Math.floor(roundValue) == roundValue) {
			return Long.toString(Math.round(aNumber));
		} else {
			if (aNumber >= 0.0d) {
				return Double.toString(aNumber + roundValue / 10).substring(0, Double.toString(roundValue).length());
			} else {
				return Double.toString(aNumber - roundValue / 10).substring(0, Double.toString(roundValue).length() + 1);
			}
		}
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunSimpleGaugeViewAwt(this);
		} else {
			return new JunSimpleGaugeViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Simple Gauge");
	}

}
