package jp.co.sra.jun.goodies.freehand;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.Window;
import java.awt.event.WindowEvent;
import java.util.Vector;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.goodies.colors.JunColorChoiceDialog;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.nib.JunNibChoiceDialog;
import jp.co.sra.jun.goodies.nib.JunNibChoiceWithColor;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;

/**
 * JunFreehandPage class
 * 
 *  @author    nisinaka
 *  @created   2002/07/05 (by nisinaka)
 *  @updated   2004/04/02 (by matsubara)
 *  @version   699 (with StPL8.9) based on Jun634 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFreehandPage.java,v 8.12 2008/02/20 06:31:34 nisinaka Exp $
 */
public class JunFreehandPage extends JunApplicationModel {

	protected JunNibChoiceWithColor nibChoice;
	protected Vector nibCollection;
	protected Vector locusCollection;
	protected Vector scalarCollection;
	protected int currentIndex;

	/**
	 * Create an instance of JunFreehandPage and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFreehandPage() {
		super();
	}

	/**
	 * Create an instance of JunFreehandPage and initialize it with the JunLispList.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunFreehandPage(JunLispList aList) {
		this();
		this.fromLispList_(aList);
	}

	/**
	 * Initialize the instance variables.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.flushPage();
	}

	/**
	 * Answer the current index.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int currentIndex() {
		return currentIndex;
	}

	/**
	 * Flush the page.
	 * 
	 * @category accessing
	 */
	public void flushPage() {
		nibCollection = new Vector();
		locusCollection = new Vector();
		scalarCollection = new Vector();
		currentIndex = -1;
	}

	/**
	 * Answer the locus collection.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector locusCollection() {
		return locusCollection;
	}

	/**
	 * Answer the scalar collection.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector scalarCollection() {
		return scalarCollection;
	}

	/**
	 * Answer the nib collection.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector nibCollection() {
		return nibCollection;
	}

	/**
	 * Answer the current nib.
	 * 
	 * @return jp.co.sra.jun.goodies.nib.JunNibChoiceWithColor
	 * @category accessing
	 */
	public JunNibChoiceWithColor nibChoice() {
		if (nibChoice == null) {
			double[] nibCollection = new double[] { 1.0, 2.0, 3.0, 4.0, 5.0 };
			nibChoice = new JunNibChoiceWithColor(nibCollection, 1, Color.black, $("circle"));
		}
		return nibChoice;
	}

	/**
	 * Answer the current nib color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color nibColor() {
		return this.nibChoice().nibColor();
	}

	/**
	 * Set the new nib color.
	 * 
	 * @param newNibColor java.awt.Color
	 * @category accessing
	 */
	public void nibColor_(Color newNibColor) {
		this.nibChoice().nibColor_(newNibColor);
	}

	/**
	 * Answer the current nib width.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int nibWidth() {
		return (int) this.nibChoice().nibWidth();
	}

	/**
	 * Set the new nib width.
	 * 
	 * @param newNibWidth int
	 * @category accessing
	 */
	public void nibWidth_(int newNibWidth) {
		this.nibChoice().nibWidth_(newNibWidth);
	}

	/**
	 * Answer true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.locusCollection().isEmpty();
	}

	/**
	 * Change the nib color.
	 * 
	 * @category menu messages
	 */
	public void changeNibColor() {
		Color newColor = JunColorChoiceDialog.Request_color_("Select a color.", this.nibColor());
		this.nibColor_(newColor);
	}

	/**
	 * Change the nib width.
	 * 
	 * @category menu messages
	 */
	public void changeNibWidth() {
		Object[] anArray = { this.nibChoice().nibCollection(), new Double(this.nibWidth()), this.nibColor(), $("circle") };
		Object nibValue = JunNibChoiceDialog.Request("Please select a nib", anArray);
		if (nibValue == null) {
			return;
		}
		this.nibWidth_(((Double) ((Object[]) nibValue)[1]).intValue());
	}

	/**
	 * Change nib and color.
	 *
	 * @category menu messages
	 */
	public void changeNibWidthAndColor() {
		JunNibChoiceWithColor model = this.nibChoice();
		Window window = null;
		if (model.builder() == null || (window = model.builder().window()) == null || !window.isShowing()) {
			model.open();
		}
		if ((window = model.builder().window()) != null) {
			window.setName($String("Nib Choice"));
		}
	}

	/**
	 * Menu message: clear the page.
	 * 
	 * @category menu messages
	 */
	public void clearPage() {
		if (JunDialog.Confirm_($String("Really clear all?"), false)) {
			this.flushPage();
			this.changed();
		}
	}

	/**
	 * Menu message: go backward.
	 * 
	 * @category menu messages
	 */
	public void goBackward() {
		currentIndex = Math.max(-1, Math.min(this.currentIndex() - 1, this.locusCollection().size() - 1));
		this.changed();
	}

	/**
	 * Menu message: go forward.
	 * 
	 * @category menu messages
	 */
	public void goForward() {
		currentIndex = Math.max(-1, Math.min(this.currentIndex() + 1, this.locusCollection().size() - 1));
		this.changed();
	}

	/**
	 * Menu message: open a viewfinder.
	 * 
	 * @category menu messages
	 */
	public void openViewfinder() {
		JunOpenGL3dObject anObject = this.threeDimensionalObject();
		if (anObject == null) {
			return;
		}

		JunOpenGLDisplayModel.Show_eyePoint_sightPoint_upVector_(anObject, new Jun3dPoint(0, 0, 1000), new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 1, 0));
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFreehandViewAwt(this);
		} else {
			return new JunFreehandViewSwing(this);
		}
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Freehand Page");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param evt java.awt.event.WindowEvent
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent evt) {
		this.nibChoice().closeRequest();
		super.noticeOfWindowClose(evt);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		this.nibColorFromLispList_(aList);
		this.nibWidthFromLispList_(aList);
		this.nibCollectionFromLispList_(aList);
		this.locusCollectionFromLispList_(aList);
		this.scalarCollectionFromLispList_(aList);
		this.currentIndexFromLispList_(aList);
	}

	/**
	 * Get my nib color from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void nibColorFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("nibColor"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.nibColor_((Color) list.tail());
	}

	/**
	 * Get my nib width from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void nibWidthFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("nibWidth"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.nibWidth_(((Number) list.tail()).intValue());
	}

	/**
	 * Get my nib collection from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void nibCollectionFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("nibCollection"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Vector aVector = new Vector(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			aVector.add(((JunLispCons) anArray[i]).asArray());
		}

		nibCollection = aVector;
	}

	/**
	 * Get my locus collection from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void locusCollectionFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("locusCollection"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Vector aVector = new Vector(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			aVector.add(((JunLispCons) anArray[i]).asArray());
		}

		locusCollection = aVector;
	}

	/**
	 * Get my scalar collection from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void scalarCollectionFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("scalarCollection"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Vector aVector = new Vector(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			aVector.add(((JunLispCons) anArray[i]).asArray());
		}

		scalarCollection = aVector;
	}

	/**
	 * Get my nib width from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void currentIndexFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("currentIndex"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		currentIndex = ((Number) list.tail()).intValue();
	}

	/**
	 * Convert the receiver to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		list.add_(this.nibColorToLispList());
		list.add_(this.nibWidthToLispList());
		if (!this.nibCollection().isEmpty()) {
			list.add_(this.nibCollectionToLispList());
		}
		if (!this.locusCollection().isEmpty()) {
			list.add_(this.locusCollectionToLispList());
		}
		if (!this.scalarCollection().isEmpty()) {
			list.add_(this.scalarCollectionToLispList());
		}
		list.add_(this.currentIndexToLispList());
		return list;
	}

	/**
	 * Convert the receiver's nib color to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons nibColorToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("nibColor"));
		list.tail_(this.nibColor());
		return list;
	}

	/**
	 * Convert the receiver's nib width to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons nibWidthToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("nibWidth"));
		list.tail_(new Integer(this.nibWidth()));
		return list;
	}

	/**
	 * Convert the receiver's nib collection to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons nibCollectionToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("nibCollection"));
		JunLispList[] nibList = new JunLispList[this.nibCollection().size()];
		for (int i = 0; i < nibList.length; i++) {
			nibList[i] = JunLispList.List_((Object[]) this.nibCollection().get(i));
		}
		list.tail_(JunLispList.List_(nibList));
		return list;
	}

	/**
	 * Convert the receiver's locus collection to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons locusCollectionToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("locusCollection"));
		JunLispList[] locusList = new JunLispList[this.locusCollection().size()];
		for (int i = 0; i < locusList.length; i++) {
			locusList[i] = JunLispList.List_((Jun3dPoint[]) this.locusCollection().get(i));
		}
		list.tail_(JunLispList.List_(locusList));
		return list;
	}

	/**
	 * Convert the receiver's scalar collection to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons scalarCollectionToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("scalaarCollection"));
		list.tail_(JunLispList.List_(this.scalarCollection()));
		return list;
	}

	/**
	 * Convert the receiver's current index to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons currentIndexToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("currentIndex"));
		list.tail_(new Integer(this.currentIndex()));
		return list;
	}

	/**
	 * Draw a mark on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category private
	 */
	protected void _drawMark(Graphics aGraphics, Point aPoint) {
		int nibWidth = this.nibWidth();
		int x = aPoint.x - (nibWidth / 2);
		int y = aPoint.y - (nibWidth / 2);
		int side = (nibWidth / 2 * 2) + 1;
		aGraphics.fillRect(x, y, side, side);
	}

	/**
	 * Draw a rectangle on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aRectangle jp.co.sra.smalltalk.StRectangle
	 * @category private
	 */
	protected void _drawDotbox(Graphics aGraphics, StRectangle aRectangle) {
		if (aRectangle.width() < 3) {
			aGraphics.fillRect(aRectangle.x(), aRectangle.y(), aRectangle.width(), aRectangle.height());
		} else {
			StRectangle r = aRectangle.expandedBy_(new StRectangle(0, 0, 1, 1));
			aGraphics.fillArc(r.x(), r.y(), r.width(), r.height(), 0, 360);
		}
	}

	/**
	 * Draw a polyline on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param points java.awt.Point[]
	 * @category private
	 */
	protected void _drawPolyline(Graphics aGraphics, Point[] points) {
		for (int i = 1; i < points.length; i++) {
			aGraphics.drawLine(points[i - 1].x, points[i - 1].y, points[i].x, points[i].y);
		}
	}

	/**
	 * Draw a polyline on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param pointCollection java.util.Vector
	 * @category private
	 */
	protected void _drawPolyline(Graphics aGraphics, Vector pointCollection) {
		Point[] points = new Point[pointCollection.size()];
		pointCollection.copyInto(points);
		this._drawPolyline(aGraphics, points);
	}

	/**
	 * Answer the stroke.
	 * 
	 * @param width float
	 * @return java.awt.Stroke
	 * @category private
	 */
	protected Stroke _stroke(float width) {
		return new BasicStroke(width, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND);
	}

	/**
	 * Initialize with the specified values.
	 * 
	 * @param newNibColor java.awt.Color
	 * @param newNibWidth int
	 * @param newNibCollection java.util.Vector
	 * @param newLocusCollection java.util.Vector
	 * @param newScalarCollection java.util.Vector
	 * @param newCurrentIndex int
	 * @category private
	 */
	protected void nibColor_nibWidth_nibCollection_locusCollection_scalarCollection_currentIndex_(Color newNibColor, int newNibWidth, Vector newNibCollection, Vector newLocusCollection, Vector newScalarCollection, int newCurrentIndex) {
		this.nibColor_(newNibColor);
		this.nibWidth_(newNibWidth);
		nibCollection = newNibCollection;
		locusCollection = newLocusCollection;
		scalarCollection = newScalarCollection;
		currentIndex = Math.min(newCurrentIndex, locusCollection.size());
	}

	/**
	 * Set the stroke information.
	 * 
	 * @param pointCollection java.util.Vector
	 * @param boundingBox java.awt.Rectangle
	 * @param timeCollection java.util.Vector
	 * @category private
	 */
	protected void pointCollection_boundingBox_timeCollection_(Vector pointCollection, Rectangle boundingBox, Vector timeCollection) {
		if (pointCollection.isEmpty()) {
			return;
		}
		if (timeCollection.isEmpty()) {
			return;
		}

		int denomiWidth = boundingBox.width / 2;
		int denomiHeight = boundingBox.height / 2;
		Point centerPoint = new Point(denomiWidth, denomiHeight);
		int numberOfPoints = pointCollection.size();
		Jun2dPoint[] normalizedPointCollection = new Jun2dPoint[numberOfPoints];
		if ((denomiWidth == 0) || (denomiHeight == 0)) {
			for (int i = 0; i < numberOfPoints; i++) {
				normalizedPointCollection[i] = new Jun2dPoint(0, -0);
			}
		} else {
			for (int i = 0; i < numberOfPoints; i++) {
				Point p = (Point) pointCollection.get(i);
				double x = (double) (p.x - centerPoint.x) / denomiWidth;
				double y = (double) (p.y - centerPoint.y) / denomiHeight;
				normalizedPointCollection[i] = new Jun2dPoint(x, -y);
			}
		}

		long startTime = ((Number) timeCollection.firstElement()).longValue();
		long denomiTime = ((Number) timeCollection.lastElement()).longValue() - startTime;
		double[] normalizedTimeCollection = null;
		if (denomiTime == 0) {
			normalizedTimeCollection = new double[] { 0 };
		} else {
			int numberOfTimes = timeCollection.size();
			normalizedTimeCollection = new double[numberOfTimes];
			for (int i = 0; i < numberOfTimes; i++) {
				double t = ((Number) timeCollection.get(i)).doubleValue();
				normalizedTimeCollection[i] = (t - startTime) / denomiTime;
			}
		}

		Jun3dPoint[] locus = new Jun3dPoint[numberOfPoints];
		for (int i = 0; i < numberOfPoints; i++) {
			locus[i] = new Jun3dPoint(normalizedPointCollection[i], normalizedTimeCollection[i]);
		}

		this.nibCollection().setSize(currentIndex + 1);
		this.locusCollection().setSize(currentIndex + 1);
		this.scalarCollection().setSize(currentIndex + 1);

		this.nibCollection().add(new Object[] { this.nibColor(), new Integer(this.nibWidth()) });
		this.locusCollection().add(locus);
		this.scalarCollection().add(new Jun3dPoint(denomiWidth, denomiHeight, denomiTime));

		currentIndex++;
	}

	/**
	 * Answer the 3D object of the current status.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject threeDimensionalObject() {
		if (this.currentIndex() < 0) {
			return null;
		}

		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		Jun3dPoint translationPoint = new Jun3dPoint(0, 0, 0);
		for (int i = 0; i <= this.currentIndex(); i++) {
			Jun3dPoint[] points = (Jun3dPoint[]) locusCollection.get(i);
			Jun3dPoint scalar = (Jun3dPoint) scalarCollection.get(i);
			Object[] nib = (Object[]) nibCollection.get(i);
			Color color = (Color) nib[0];
			int dot = ((Number) nib[1]).intValue();
			Jun3dPoint[] scaledPoints = new Jun3dPoint[points.length];
			for (int j = 0; j < scaledPoints.length; j++) {
				scaledPoints[j] = points[j].multipliedBy_(scalar);
			}

			JunOpenGL3dObject object = null;
			if (scaledPoints.length == 1) {
				JunOpenGL3dVertex vertex = new JunOpenGL3dVertex(scaledPoints[0]);
				vertex.size_(dot);
				object = vertex;
			} else {
				JunOpenGL3dPolyline polyline = new JunOpenGL3dPolyline(scaledPoints);
				polyline.lineWidth_(dot);
				object = polyline;
			}

			object.paint_(color);
			object = object.translatedBy_(translationPoint);
			compoundObject.add_(object);
			translationPoint = translationPoint.plus_(new Jun3dPoint(0, 0, scalar.z()));
		}

		Jun3dBoundingBox boundingBox = compoundObject.boundingBox();
		compoundObject = (JunOpenGL3dCompoundObject) compoundObject.scaledBy_(new Jun3dPoint(2 / boundingBox.width(), 2 / boundingBox.height(), 2 / boundingBox.depth()));
		return compoundObject;
	}

	/**
	 * Answer one of my views.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunFreehandView && ((JunFreehandView) dependents[i]).model() == this) {
				return (JunFreehandView) dependents[i];
			}
		}

		return null;
	}

}
