package jp.co.sra.jun.goodies.drawing.map;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.goodies.drawing.element.JunCompositeElement;
import jp.co.sra.jun.goodies.drawing.element.JunDrawingElement;
import jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual;
import jp.co.sra.jun.goodies.drawing.element.JunLabelElement;
import jp.co.sra.jun.goodies.drawing.element.JunLinkElement;
import jp.co.sra.jun.goodies.drawing.element.JunVertexesElement;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunDrawingMap class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawingMap.java,v 8.10 2008/02/20 06:31:24 nisinaka Exp $
 */
public class JunDrawingMap extends JunDrawingVisual {
	protected ArrayList componentElements;
	protected transient Rectangle preferredBounds;
	protected transient ArrayList selectedElements;

	public static final Color CONTROLL_AREA_BORDER_COLOR = Color.darkGray;
	public static final Color CONTROLL_AREA_DONTMOVE_BORDER_COLOR = Color.lightGray;
	public static final Color CONTROLL_AREA_BACKGROUND_COLOR = Color.white;

	public static DataFlavor DataFlavor = new DataFlavor(JunDrawingMap.class, "JunDrawingMap");

	/**
	 * Create a new instance of JunDrawingMap and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunDrawingMap() {
		super();
	}

	/**
	 * Create a new instance of JunDrawingMap and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunDrawingMap(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the JunDrawingVisual.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		componentElements = null;
		preferredBounds = null;
		selectedElements = null;
	}

	/**
	 * Get the receiver's component elements.
	 * 
	 * @return java.uti.ArrayList
	 * @category accessing
	 */
	public ArrayList componentElements() {
		if (componentElements == null) {
			componentElements = new ArrayList();
		}
		return componentElements;
	}

	/**
	 * Set the receiver's component elements.
	 * 
	 * @param anArray java.uti.ArrayList
	 * @category accessing
	 */
	public void componentElements_(ArrayList anArray) {
		componentElements = anArray;
		this.flushBounds();
	}

	/**
	 * Get an array of the receiver's component elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement[]
	 * @category accessing
	 */
	public JunDrawingElement[] _componentElements() {
		JunDrawingElement[] elements = new JunDrawingElement[this.componentElementSize()];
		this.componentElements().toArray(elements);
		return elements;
	}

	/**
	 * Answer the element at the specified position in this components.
	 * 
	 * @param anIndex int
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category accessing
	 */
	public JunDrawingElement componentElementAt_(int anIndex) {
		if (this.componentElements().isEmpty()) {
			return null;
		}
		if (anIndex < 0 || this.componentElementSize() <= anIndex) {
			return null;
		}
		return (JunDrawingElement) this.componentElements().get(anIndex);
	}

	/**
	 * Answer the index of the first occurrence of anElement within the receiver.
	 * If the receiver does not contain anElement, answer -1.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return int
	 * @see java.util.ArrayList#indexOf(java.lang.Object);
	 * @category accessing
	 */
	public int componentElementIndexOf_(JunDrawingElement anElement) {
		return this.componentElements().indexOf(anElement);
	}

	/**
	 * Answer how many elements the receiver contains.
	 * 
	 * @return int
	 * @see java.util.ArrayList#size();
	 * @category accessing
	 */
	public int componentElementSize() {
		return this.componentElements().size();
	}

	/**
	 * Answer the receiver's label elements.
	 * 
	 * @return java.uti.ArrayList
	 * @category accessing
	 */
	public ArrayList labelElements() {
		ArrayList anArray = new ArrayList();
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			if (elements[i].isLabel()) {
				anArray.add(elements[i]);
			}
		}
		return anArray;
	}

	/**
	 * Answer an array of the receiver's label elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunLabelElement[]
	 * @category accessing
	 */
	public JunLabelElement[] _labelElements() {
		ArrayList anArray = this.labelElements();
		return (JunLabelElement[]) anArray.toArray(new JunLabelElement[anArray.size()]);
	}

	/**
	 * Answer the receiver's link elements.
	 * 
	 * @return java.uti.ArrayList
	 * @category accessing
	 */
	public ArrayList linkElements() {
		ArrayList anArray = new ArrayList();
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			if (elements[i].isLink()) {
				anArray.add(elements[i]);
			}
		}
		return anArray;
	}

	/**
	 * Answer an array of the receiver's link elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunLinkElement[]
	 * @category accessing
	 */
	public JunLinkElement[] _linkElements() {
		ArrayList anArray = this.linkElements();
		return (JunLinkElement[]) anArray.toArray(new JunLinkElement[anArray.size()]);
	}

	/**
	 * Add the specified element to the end of this component elements.
	 * 
	 * @param newElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return boolean
	 * @category adding
	 */
	public boolean addElement_(JunDrawingElement newElement) {
		if (newElement == null || this.componentElements().contains(newElement)) {
			return false;
		}
		this.flushBounds();
		newElement.parent_(this);
		return this.componentElements().add(newElement);
	}

	/**
	 * Add the specified element to the position of this component elements.
	 * 
	 * @param newElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @param anIndex int
	 * @return boolean
	 * @category adding
	 */
	public void addElement_beforeIndex_(JunDrawingElement newElement, int anIndex) {
		if (newElement == null || this.componentElements().contains(newElement)) {
			return;
		}
		if (0 <= anIndex && anIndex < this.componentElementSize()) {
			this.componentElements().add(anIndex, newElement);
		} else {
			this.componentElements().add(newElement);
		}
		newElement.parent_(this);
		this.flushBounds();
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(Point aPoint) {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent_(java.awt.Dimension)
	 * @category bounds accessing
	 */
	public void extent_(Dimension aDimension) {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		if (preferredBounds == null) {
			preferredBounds = super.preferredBounds();
			JunDrawingElement[] elements = this._componentElements();
			for (int i = 0; i < elements.length; i++) {
				preferredBounds.add(elements[i].bounds());
				for (Iterator areaIterator = elements[i].controllPointAreas().values().iterator(); areaIterator.hasNext();) {
					preferredBounds.add((Rectangle) areaIterator.next());
				}
			}
			preferredBounds.setSize(preferredBounds.width + 1, preferredBounds.height + 1);
		}
		return preferredBounds;
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);
		ArrayList newComponents = new ArrayList();
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			newComponents.add(context.containsKey(elements[i]) ? context.get(elements[i]) : elements[i].copy(context));
		}
		this.componentElements_(newComponents);
		return this;
	}

	/**
	 * Display the receiver's element on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			elements[i].displayOn_(aGraphics);

			if (this.selectedElements().contains(elements[i])) {
				HashMap controllPointAreas = elements[i].controllPointAreas();
				for (Iterator iterator = controllPointAreas.keySet().iterator(); iterator.hasNext();) {
					StSymbol key = (StSymbol) iterator.next();
					Rectangle area = (Rectangle) controllPointAreas.get(key);
					aGraphics.setColor(CONTROLL_AREA_BACKGROUND_COLOR);
					aGraphics.fillRect(area.x, area.y, area.width, area.height);
					if (elements[i].isLink() && (key == JunVertexesElement.CONTROLL_POINT_BEGIN || key == JunVertexesElement.CONTROLL_POINT_END)) {
						aGraphics.setColor(CONTROLL_AREA_DONTMOVE_BORDER_COLOR);
					} else {
						aGraphics.setColor(CONTROLL_AREA_BORDER_COLOR);
					}
					aGraphics.drawRect(area.x, area.y, area.width, area.height);
				}
			}
		}
	}

	/**
	 * Evaluate aBlock with each of the receiver's elements as the argument.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object elementsDo_(StBlockClosure aBlock) {
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			aBlock.value_(elements[i]);
			if (elements[i].isComposite()) {
				((JunCompositeElement) elements[i]).elementsDo_(aBlock);
			}
		}
		return this;
	}

	/**
	 * Find a drawing element with the specified id number.
	 * 
	 * @param findId long
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#findElement_(long)
	 * @category finding
	 */
	public JunDrawingElement findElement_(long findId) {
		if (findId < 0 || this.componentElementSize() == 0) {
			return null;
		}
		for (Iterator iterator = this.componentElements().iterator(); iterator.hasNext();) {
			JunDrawingElement findElement = ((JunDrawingElement) iterator.next()).findElement_(findId);
			if (findElement != null) {
				return findElement;
			}
		}
		return null;
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @category flushing
	 */
	protected void flushBounds() {
		preferredBounds = null;
	}

	/**
	 * Flush the receiver's selected elements.
	 * 
	 * @category flushing
	 */
	protected void flushSelectedElements() {
		selectedElements = null;
	}

	/**
	 * Remove the specified element from receiver's component elements.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return boolean
	 * @category removing
	 */
	public boolean removeElement_(JunDrawingElement anElement) {
		if (this.componentElements().contains(anElement) == false) {
			return false;
		}
		this.removeSelectedElement_(anElement);
		boolean result = this.componentElements().remove(anElement);
		anElement.parent_(null);
		this.flushBounds();
		return result;
	}

	/**
	 * Remove all selected elements from receiver's component elements.
	 * 
	 * @category removing
	 */
	public void removeSelectedElements() {
		if (this.selectedElements().isEmpty()) {
			return;
		}

		JunDrawingElement[] selectedElements = this._selectedElements();
		for (int i = 0; i < selectedElements.length; i++) {
			this.removeElement_(selectedElements[i]);
			if (selectedElements[i].hasLabel()) {
				JunLinkElement linkElement = (JunLinkElement) selectedElements[i];
				JunLabelElement labelElement = linkElement.labelElement();
				if (this.selectedElements().contains(labelElement) == false) {
					linkElement.labelElement_(null);
					this.removeElement_(labelElement);
				}
			} else if (selectedElements[i].isLabel()) {
				((JunLabelElement) selectedElements[i]).baseElement().labelElement_(null);
			}

		}

		JunLinkElement[] linkElements = this._linkElements();
		for (int i = 0; i < linkElements.length; i++) {
			if (linkElements[i].checkElement() == false) {
				this.removeElement_(linkElements[i]);
			}
		}
		JunLabelElement[] labelElements = this._labelElements();
		for (int i = 0; i < labelElements.length; i++) {
			if (labelElements[i].checkElement() == false) {
				this.removeElement_(labelElements[i]);
			}
		}
	}

	/**
	 * Answer the drawing element including a specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category selecting
	 */
	public JunDrawingElement which_(Point aPoint) {
		for (ListIterator iterator = this.componentElements().listIterator(this.componentElementSize()); iterator.hasPrevious();) {
			JunDrawingElement element = (JunDrawingElement) iterator.previous();
			if (element.containsPoint_(aPoint) || (this.selectedElements().contains(element) && element.containsPointInControllArea_(aPoint))) {
				return element;
			}
		}
		return null;
	}

	/**
	 * Get the receiver's selected elements.
	 * 
	 * @return java.uti.ArrayList
	 * @category selecting
	 */
	public ArrayList selectedElements() {
		if (selectedElements == null) {
			selectedElements = new ArrayList();
		}
		return selectedElements;
	}

	/**
	 * Get an array of the receiver's selected elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement[]
	 * @category accessing
	 */
	public JunDrawingElement[] _selectedElements() {
		return (JunDrawingElement[]) this.selectedElements().toArray(new JunDrawingElement[this.selectedElements().size()]);
	}

	/**
	 * Added the selected element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return boolean
	 * @category selecting
	 */
	public boolean addSelectedElement_(JunDrawingElement anElement) {
		if (this.selectedElements().contains(anElement)) {
			return false;
		}
		return this.selectedElements().add(anElement);
	}

	/**
	 * Remove the selected element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.graph.element.JunDrawingElement
	 * @return boolean
	 * @category selecting
	 */
	public boolean removeSelectedElement_(JunDrawingElement anElement) {
		if (this.selectedElements().contains(anElement) == false) {
			return false;
		}
		return this.selectedElements().remove(anElement);
	}

	/**
	 * Answer the current selected element
	 * 
	 * @return jp.co.sra.jun.goodies.graph.element.JunDrawingElement
	 * @category selecting
	 */
	public JunDrawingElement currentElement() {
		if (this.selectedElements().isEmpty()) {
			return null;
		}
		return (JunDrawingElement) this.selectedElements().get(this.selectedElements().size() - 1);
	}

	/**
	 * Clear the selected elements.
	 * 
	 * @category selecting
	 */
	public void clearSelectedElements() {
		selectedElements = new ArrayList();
	}

	/**
	 * Select all elements.
	 * 
	 * @category selecting
	 */
	public void selectAll() {
		this.flushSelectedElements();
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			this.addSelectedElement_(elements[i]);
		}
	}

	/**
	 * Answer true if receiver is composite element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isComposite()
	 * @category testing
	 */
	public boolean isComposite() {
		return true;
	}

	/**
	 * Answer true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.componentElements().isEmpty();
	}

	/**
	 * Answer true if receiver is map, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isMap()
	 * @category testing
	 */
	public boolean isMap() {
		return true;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunDrawingMap.DataFlavor };
	}

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		JunLispCons componentsList = this.componentElementsToLispList();
		if (componentsList != null) {
			list.add_(componentsList);

		}
		return list;
	}

	/**
	 * Convert the receiver's component elements to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons componentElementsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("componentElements"));
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			list.add_(elements[i].toLispList());
		}
		return list;
	}

	/**
	 * Get the component element from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.componentElementsFromLispList_(aList);
	}

	/**
	 * Get the receiver's component elements from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void componentElementsFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("componentElements"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] objects = ((JunLispList) list.tail()).asArray();
		for (int i = 0; i < objects.length; i++) {
			JunDrawingElement anElement = JunDrawingElement.FromLispList_((JunLispList) objects[i]);
			this.addElement_(anElement);
		}
	}
}
