package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.HashMap;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunRectangularElement class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunRectangularElement.java,v 8.10 2008/02/20 06:31:23 nisinaka Exp $
 */
public abstract class JunRectangularElement extends JunNodeElement {
	protected Point elementLocation;
	protected Dimension elementExtent;

	/**
	 * Create a new instance of JunRectangularElement and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunRectangularElement() {
		super();
	}

	/**
	 * Create a new instance of JunRectangularElement and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunRectangularElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the JunDrawingVisual.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		elementLocation = null;
		elementExtent = null;
	}

	/**
	 * Answer the receiver's location point.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location()
	 * @category bounds accessing
	 */
	public Point location() {
		if (elementLocation == null) {
			elementLocation = this.defaultLocation();
		}
		return elementLocation;
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(Point aPoint) {
		elementLocation = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		this.flushBounds();
	}

	/**
	 * Answer the receiver's extent size.
	 * 
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent()
	 * @category bounds accessing
	 */
	public Dimension extent() {
		if (elementExtent == null) {
			elementExtent = this.defaultExtent();
		}
		return elementExtent;
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent_(java.awt.Dimension)
	 * @category bounds accessing
	 */
	public void extent_(Dimension aDimension) {
		elementExtent = new Dimension(Math.max(aDimension.width, this.defaultMinimumExtent().width), Math.max(aDimension.height, this.defaultMinimumExtent().height));
		this.flushBounds();
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return new Rectangle(this.location(), this.extent());
	}

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.util.HashMap
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#controllPoints()
	 * @category bounds accessing
	 */
	public HashMap controllPoints() {
		Point locationPoint = this.location();
		Point extentPoint = this.extentPoint();

		HashMap pointMap = new HashMap();
		pointMap.put(JunDrawingElement.CONTROLL_POINT_TOP_LEFT, new Point(locationPoint.x, locationPoint.y));
		pointMap.put(JunDrawingElement.CONTROLL_POINT_TOP_RIGHT, new Point(extentPoint.x, locationPoint.y));
		pointMap.put(JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT, new Point(locationPoint.x, extentPoint.y));
		pointMap.put(JunDrawingElement.CONTROLL_POINT_BOTTOM_RIGHT, new Point(extentPoint.x, extentPoint.y));
		return pointMap;
	}

	/**
	 * Replaces the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#controllPointAt_put_(jp.co.sra.smalltalk.StSymbol, java.awt.Point)
	 * @category bounds accessing
	 */
	public void controllPointAt_put_(StSymbol aSymbol, Point aPoint) {
		Rectangle bounds = this.bounds();
		Point controllPoint = this.controllPointAt_(aSymbol);
		boolean changeVertical = (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_RIGHT);
		boolean changeHorizontal = (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT);
		int deltaWidth = Math.max(aPoint.x, 0) - controllPoint.x;
		int deltaHeight = Math.max(aPoint.y, 0) - controllPoint.y;
		int newX = (changeHorizontal) ? (Math.max(bounds.x + deltaWidth, 0)) : (bounds.x);
		int newY = (changeVertical) ? (Math.max(bounds.y + deltaHeight, 0)) : (bounds.y);
		int newWidth = (changeHorizontal) ? (bounds.width + (bounds.x - newX)) : (Math.max(bounds.width + deltaWidth, this.defaultMinimumExtent().width));
		int newHeight = (changeVertical) ? (bounds.height + (bounds.y - newY)) : (Math.max(bounds.height + deltaHeight, this.defaultMinimumExtent().height));
		if (newWidth < this.defaultMinimumExtent().width) {
			newX = newX + (newWidth - this.defaultMinimumExtent().width);
			newWidth = this.defaultMinimumExtent().width;
		}
		if (newHeight < this.defaultMinimumExtent().height) {
			newY = newY + (newHeight - this.defaultMinimumExtent().height);
			newHeight = this.defaultMinimumExtent().height;
		}
		this.bounds_(new Rectangle(newX, newY, newWidth, newHeight));
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);

		this.location_(new Point(elementLocation));
		this.extent_(new Dimension(elementExtent));

		return this;
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @category flushing
	 */
	protected void flushBounds() {
	}

	/**
	 * Answer true if receiver is rectangular element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isRectangular()
	 * @category testing
	 */
	public boolean isRectangular() {
		return true;
	}

	/**
	 * Convert the element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.locationToLispList());
		list.add_(this.extentToLispList());
		return list;
	}

	/**
	 * Convert the location to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons locationToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("location"));
		list.tail_(new Jun2dPoint(this.location()));
		return list;
	}

	/**
	 * Convert the extent to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons extentToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("extent"));
		list.tail_(new Jun2dPoint(this.extent().width, this.extent().height));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.locationFromLispList_(aList);
		this.extentFromLispList_(aList);
	}

	/**
	 * Get the location from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void locationFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("location"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.location_(((Jun2dPoint) list.tail())._toPoint());
	}

	/**
	 * Get the extent from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void extentFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("extent"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		Point aPoint = ((Jun2dPoint) list.tail())._toPoint();
		this.extent_(new Dimension(aPoint.x, aPoint.y));
	}
}
