package jp.co.sra.jun.goodies.drawing.element;

import java.awt.BasicStroke;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.datatransfer.DataFlavor;
import java.awt.geom.GeneralPath;
import java.util.HashMap;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunFreehandElement class
 * 
 *  @author    m-asada
 *  @created   2005/10/13 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFreehandElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunFreehandElement extends JunVertexesElement {
	protected Point elementLocation;
	protected Dimension elementExtent;

	protected transient Dimension minimumExtent;

	public static DataFlavor DataFlavor = new DataFlavor(JunFreehandElement.class, "JunFreehandElement");

	/**
	 * Create a new instance of <code>JunFreehandElement</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFreehandElement() {
		super();
	}

	/**
	 * Create a new instance of <code>JunFreehandElement</code> and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunFreehandElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the JunDrawingVisual.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		elementLocation = null;
		elementExtent = null;
		minimumExtent = null;
	}

	/**
	 * Answer the receiver's location point.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location()
	 * @category bounds accessing
	 */
	public Point location() {
		if (elementLocation == null) {
			elementLocation = this.points().isEmpty() ? this.defaultLocation() : this.pointBounds().getLocation();
		}
		return elementLocation;
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(Point aPoint) {
		elementLocation = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		this.flushBounds();
	}

	/**
	 * Answer the receiver's extent size.
	 * 
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent()
	 * @category bounds accessing
	 */
	public Dimension extent() {
		if (elementExtent == null) {
			elementExtent = this.points().isEmpty() ? this.defaultExtent() : this.pointBounds().getSize();
		}
		return elementExtent;
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent_(java.awt.Dimension)
	 * @category bounds accessing
	 */
	public void extent_(Dimension aDimension) {
		elementExtent = new Dimension(Math.max(aDimension.width, this.defaultMinimumExtent().width), Math.max(aDimension.height, this.defaultMinimumExtent().height));
		this.flushBounds();
	}

	/**
	 * Set the receiver's rectangle.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#bounds_(java.awt.Rectangle)
	 * @category bounds accessing
	 */
	public void bounds_(Rectangle aRectangle) {
		Rectangle preferredBounds = this.preferredBounds();
		int newX = aRectangle.x - (preferredBounds.x - this.location().x);
		int newY = aRectangle.y - (preferredBounds.y - this.location().y);
		int newWidth = aRectangle.width - (preferredBounds.width - this.extent().width);
		int newHeight = aRectangle.height - (preferredBounds.height - this.extent().height);
		if (this.defaultMinimumExtent().width >= newWidth) {
			newX = this.location().x;
			newWidth = this.extent().width;
		}
		if (this.defaultMinimumExtent().height >= newHeight) {
			newY = this.location().y;
			newHeight = this.extent().height;
		}
		super.bounds_(new Rectangle(newX, newY, newWidth, newHeight));
	}

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.util.HashMap
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#controllPoints()
	 * @category bounds accessing
	 */
	public HashMap controllPoints() {
		Point locationPoint = this.location();
		Point extentPoint = this.extentPoint();

		HashMap pointMap = new HashMap();
		pointMap.put(JunDrawingElement.CONTROLL_POINT_TOP_LEFT, new Point(locationPoint.x, locationPoint.y));
		pointMap.put(JunDrawingElement.CONTROLL_POINT_TOP_RIGHT, new Point(extentPoint.x, locationPoint.y));
		pointMap.put(JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT, new Point(locationPoint.x, extentPoint.y));
		pointMap.put(JunDrawingElement.CONTROLL_POINT_BOTTOM_RIGHT, new Point(extentPoint.x, extentPoint.y));
		return pointMap;
	}

	/**
	 * Replaces the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#controllPointAt_put_(jp.co.sra.smalltalk.StSymbol, java.awt.Point)
	 * @category bounds accessing
	 */
	public void controllPointAt_put_(StSymbol aSymbol, Point aPoint) {
		Rectangle bounds = this.bounds();
		Point controllPoint = this.controllPointAt_(aSymbol);
		boolean changeVertical = (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_RIGHT);
		boolean changeHorizontal = (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT);
		int deltaWidth = Math.max(aPoint.x, 0) - controllPoint.x;
		int deltaHeight = Math.max(aPoint.y, 0) - controllPoint.y;
		int newX = (changeHorizontal) ? (Math.max(bounds.x + deltaWidth, 0)) : (bounds.x);
		int newY = (changeVertical) ? (Math.max(bounds.y + deltaHeight, 0)) : (bounds.y);
		int newWidth = (changeHorizontal) ? (bounds.width + (bounds.x - newX)) : (Math.max(bounds.width + deltaWidth, this.defaultMinimumExtent().width));
		int newHeight = (changeVertical) ? (bounds.height + (bounds.y - newY)) : (Math.max(bounds.height + deltaHeight, this.defaultMinimumExtent().height));
		if (newWidth < this.defaultMinimumExtent().width) {
			newX = newX + (newWidth - this.defaultMinimumExtent().width);
			newWidth = this.defaultMinimumExtent().width;
		}
		if (newHeight < this.defaultMinimumExtent().height) {
			newY = newY + (newHeight - this.defaultMinimumExtent().height);
			newHeight = this.defaultMinimumExtent().height;
		}
		this.bounds_(new Rectangle(newX, newY, newWidth, newHeight));
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);

		this.location_(new Point(elementLocation));
		this.extent_(new Dimension(elementExtent));
		minimumExtent = null;

		return this;
	}

	/**
	 * Get the default minimum extent.
	 *
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#defaultMinimumExtent()
	 * @category defaults
	 */
	public Dimension defaultMinimumExtent() {
		if (minimumExtent == null) {
			Rectangle preferredBounds = this.preferredBounds();
			Rectangle sourceBounds = new Rectangle(this.location(), this.extent());
			int deltaWidth = preferredBounds.width - sourceBounds.width;
			int deltaHeight = preferredBounds.height - sourceBounds.height;

			Rectangle pointBounds = this.pointBounds();
			int x, y;
			if (pointBounds.width > pointBounds.height) {
				double scale = 20.0d / pointBounds.width;
				x = Math.max(1, Math.min(20, pointBounds.width));
				y = Math.max(1, (int) (pointBounds.height * scale));
			} else {
				double scale = 20.0d / pointBounds.height;
				x = Math.max(1, (int) (pointBounds.width * scale));
				y = Math.max(1, Math.min(20, pointBounds.height));
			}
			minimumExtent = new Dimension(x + deltaWidth, y + deltaHeight);
		}
		return minimumExtent;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		Graphics2D graphicsContext = (Graphics2D) aGraphics.create();
		try {
			graphicsContext.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			super.displayOn_(graphicsContext);
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
				graphicsContext = null;
			}
		}
	}

	/**
	 * Answer true if receiver is rectangular element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isRectangular()
	 * @category testing
	 */
	public boolean isRectangular() {
		return true;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunFreehandElement.DataFlavor };
	}

	/**
	 * Answer the receiver's shapes.
	 * 
	 * @return java.awt.geom.Shape[]
	 * @see jp.co.sra.jun.goodies.drawing.element.JunVertexesElement#shapes()
	 * @category private
	 */
	protected Shape[] shapes() {
		if (shapes == null) {
			Rectangle pointBounds = this.pointBounds();
			Point baseLocation = this.location();
			Dimension baseSize = this.extent();
			int deltaX = pointBounds.x - baseLocation.x;
			int deltaY = pointBounds.y - baseLocation.y;

			Point[] srcPoints = this._points();
			GeneralPath path = new GeneralPath();
			for (int i = 0; i < srcPoints.length; i++) {
				int pointX = (pointBounds.width == 1) ? srcPoints[i].x - deltaX : (baseLocation.x + Math.max(Math.round((srcPoints[i].x - deltaX - baseLocation.x) * baseSize.width / (float) pointBounds.width), 0));
				int pointY = (pointBounds.height == 1) ? srcPoints[i].y - deltaY : (baseLocation.y + Math.max(Math.round((srcPoints[i].y - deltaY - baseLocation.y) * baseSize.height / (float) pointBounds.height), 0));
				if (i == 0) {
					path.moveTo(pointX, pointY);
				} else {
					path.lineTo(pointX, pointY);
				}
			}
			shapes = new Shape[] { path };
		}
		return shapes;
	}

	/**
	 * Answer the receiver's stroke
	 * 
	 * @return java.awt.BasicStroke
	 * @see jp.co.sra.jun.goodies.drawing.element.JunVertexesElement#stroke()
	 * @category private
	 */
	protected BasicStroke stroke() {
		if (stroke == null) {
			stroke = new BasicStroke(this.lineWidth(), BasicStroke.CAP_BUTT, BasicStroke.JOIN_ROUND);
		}
		return stroke;
	}

	/**
	 * Convert the element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.locationToLispList());
		list.add_(this.extentToLispList());
		return list;
	}

	/**
	 * Convert the location to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons locationToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("location"));
		list.tail_(new Jun2dPoint(this.location()));
		return list;
	}

	/**
	 * Convert the extent to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons extentToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("extent"));
		list.tail_(new Jun2dPoint(this.extent().width, this.extent().height));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.locationFromLispList_(aList);
		this.extentFromLispList_(aList);
	}

	/**
	 * Get the location from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void locationFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("location"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.location_(((Jun2dPoint) list.tail())._toPoint());
	}

	/**
	 * Get the extent from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void extentFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("extent"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		Point aPoint = ((Jun2dPoint) list.tail())._toPoint();
		this.extent_(new Dimension(aPoint.x, aPoint.y));
	}
}
