package jp.co.sra.jun.goodies.display;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Point;
import java.awt.PopupMenu;
import java.awt.ScrollPane;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StViewCanvas;
import jp.co.sra.smalltalk.WindowSpecLayout;

import jp.co.sra.jun.system.framework.JunAbstractViewPanel;

/**
 * JunDisplayViewAwt class
 * 
 *  @author    nisinaka
 *  @created   1998/11/16 (by nisinaka)
 *  @updated   1999/11/11 (by MATSUDA Ryouichi)
 *  @updated   2002/01/09 (by nisinaka)
 *  @updated   2002/11/21 (by nisinaka)
 *  @updated   2003/03/19 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDisplayViewAwt.java,v 8.16 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunDisplayViewAwt extends JunAbstractViewPanel implements JunDisplayView {

	protected StViewCanvas canvas;
	protected ScrollPane scrollPane;

	/**
	 * Create a new instance of JunDisplayViewAwt and initialize it.
	 * 
	 * @param aDisplayModel jp.co.sra.jun.goodies.display.JunDisplayModel
	 * @category Instance creation
	 */
	public JunDisplayViewAwt(JunDisplayModel aDisplayModel) {
		super(aDisplayModel);
	}

	/**
	 * Create a new instance of JunDisplayViewAwt and initialize it.
	 *
	 * @param aDisplayModel jp.co.sra.jun.goodies.display.JunDisplayModel
	 * @param aDimension java.awt.Dimension
	 * @category Instance creation
	 */
	public JunDisplayViewAwt(JunDisplayModel aDisplayModel, Dimension aDimension) {
		super(aDisplayModel);
		this.setSize(aDimension);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		canvas = null;
		scrollPane = null;
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new WindowSpecLayout());
		this.add(this.scrollPane(), WindowSpecLayout.Position(0f, 0, 0f, 0, 1f, 0, 1f, 0));
	}

	/**
	 * Answer my visual object.
	 * 
	 * @return jp.co.sra.smalltalk.StDisplayable
	 * @category accessing
	 */
	public StDisplayable visual() {
		return this.getDisplayModel().visual();
	}

	/**
	 * Gets the preferred size of this component.
	 *
	 * @return A dimension object indicating this component's preferred size.
	 * @see #getMinimumSize
	 * @see java.awt.LayoutManager
	 * @see java.awt.Container#getPreferredSize()
	 * @category bounds accessing
	 */
	public Dimension getPreferredSize() {
		Point location = this.scrollPane().getLocation();
		Dimension size = this.scrollPane().getPreferredSize();
		return new Dimension(size.width + location.x * 2, size.height + location.y * 2);
	}

	/**
	 * Answer my viewport size.
	 * 
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.display.JunDisplayView#_getViewportSize()
	 * @category bounds accessing
	 */
	public Dimension _getViewportSize() {
		return this.scrollPane().getViewportSize();
	}

	/**
	 * Answer my model as JunDisplayModel.
	 * 
	 * @return jp.co.sra.jun.goodies.display.JunDisplayModel
	 * @see jp.co.sra.jun.goodies.display.JunDisplayView#getDisplayModel()
	 * @category model accessing
	 */
	public JunDisplayModel getDisplayModel() {
		return (JunDisplayModel) this.model();
	}

	/**
	 * Answer a default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewPanel#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunDisplayController();
	}

	/**
	 * Answer my canvas.
	 *
	 * @return jp.co.sra.smalltalk.StViewCanvas
	 * @category user interface
	 */
	public StViewCanvas canvas() {
		if (canvas == null) {
			canvas = new StViewCanvas() {
				public Dimension getPreferredSize() {
					if (getDisplayModel()._isScrollable()) {
						return visual().bounds().getSize();
					}
					return super.getPreferredSize();
				}

				public void paint(Graphics g) {
					super.paint(g);
					g.setColor(Color.white);
					Dimension visualSize = visual().bounds().getSize();
					int x = (this.getWidth() - visualSize.width) / 2;
					int y = (this.getHeight() - visualSize.height) / 2;
					visual().displayOn_at_(g, new Point(x, y));
				}
			};
		}

		return canvas;
	}

	/**
	 * Answer my canvas as Component.
	 * 
	 * @return java.awt.Component
	 * @see jp.co.sra.jun.goodies.display.JunDisplayView#canvasComponent()
	 * @category user interface
	 */
	public Component canvasComponent() {
		return this.canvas();
	}

	/**
	 * Answer my scroll pane, which already contains the canvas.
	 *
	 * @return java.awt.ScrollPane
	 * @category user interface
	 */
	protected ScrollPane scrollPane() {
		if (scrollPane == null) {
			scrollPane = new ScrollPane() {
				public Dimension getPreferredSize() {
					Dimension size = visual().bounds().getSize();
					Insets insets = this.getInsets();
					return new Dimension(size.width + insets.left + insets.right, size.height + insets.top + insets.bottom);
				}

				public void setBounds(int x, int y, int width, int height) {
					super.setBounds(x, y, width, height);

					if (getDisplayModel()._isScrollable() == false) {
						Insets insets = this.getInsets();
						canvas().setSize(width - insets.left - insets.right, height - insets.top - insets.bottom);
					}
				}
			};
			scrollPane.add(this.canvas());
		}

		return scrollPane;
	}

	/**
	 * Do the layout on the scroll pane.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#addNotify()
	 * @category actions
	 */
	public void addNotify() {
		super.addNotify();

		this.scrollPane().doLayout();
	}

	/**
	 * Show the popup menu at the specified point on the view.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.smalltalk.StView#_showPopupMenu(int, int)
	 * @category popup menu
	 */
	public void _showPopupMenu(int x, int y) {
		PopupMenu popupMenu = this.popupMenuView();
		if (popupMenu != null) {
			popupMenu.show(this.canvas(), x, y);
		}
	}

	/**
	 * Update the receiver according to the change notification from the model.
	 * 
	 * @param e jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent e) {
		if (this.topComponent().isShowing() == false) {
			return;
		}

		StSymbol aspect = e.getAspect();
		if (aspect == $("redisplay")) {
			this.canvas().repaint();
		} else if (aspect == $("visual")) {
			this.scrollPane().doLayout();
			this.canvas().repaint();
		} else {
			super.update_(e);
		}
	}

}
