package jp.co.sra.jun.goodies.colors;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.StViewJPanel;
import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunColorBarViewSwing class
 * 
 *  @author    nisinaka
 *  @created   2004/06/11 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunColorBarViewSwing.java,v 8.10 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunColorBarViewSwing extends JunAbstractViewJPanel implements JunColorBarView {

	protected StRectangle firstMarkerRectangle;
	protected StRectangle lastMarkerRectangle;
	protected StRectangle sliderRectangle;
	protected StRectangle savedSliderRectangle;

	/**
	 * Create a new StView with a bordered panel which contains JunColorBarViewSwing.
	 * 
	 * @param aColorBarModel jp.co.sra.jun.goodies.colors.JunColorBarModel
	 * @return jp.co.sra.smalltalk.StView 
	 * @category Instance creation
	 */
	public static StView OnBorderedPanel_(JunColorBarModel aColorBarModel) {
		return new StViewJPanel(aColorBarModel) {
			protected void buildComponent() {
				JunColorBarViewSwing colorBarView = new JunColorBarViewSwing((JunColorBarModel) this.model());

				this.setLayout(new BorderLayout());
				this.add(colorBarView._onBorderedPanel(), BorderLayout.CENTER);
			}
		};
	}

	/**
	 * Create a new instance of JunColorBarViewSwing and initialize it.
	 * 
	 * @param aColorBarModel jp.co.sra.jun.goodies.colors.JunColorBarModel
	 * @category Instance creation
	 */
	public JunColorBarViewSwing(JunColorBarModel aColorBarModel) {
		super(aColorBarModel);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		this.flushMarkerRectangles();
		this.flushSliderRectangles();
	}

	/**
	 * Flush the markers rectangles.
	 * 
	 * @category initialize-release
	 */
	protected void flushMarkerRectangles() {
		firstMarkerRectangle = null;
		lastMarkerRectangle = null;
	}

	/**
	 * Flush the markers rectangles.
	 * 
	 * @category initialize-release
	 */
	protected void flushSliderRectangles() {
		sliderRectangle = null;
		savedSliderRectangle = null;
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setPreferredSize(new Dimension(250, 40));
	}

	/**
	 * Answer my model as JunColorBarModel.
	 * 
	 * @return jp.co.sra.jun.goodies.colors.JunColorBarModel
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#getColorBarModel()
	 * @category model accessing
	 */
	public JunColorBarModel getColorBarModel() {
		return (JunColorBarModel) this.model();
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewCanvas#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunColorBarController();
	}

	/**
	 * Answer my current first marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#firstMarkerRectangle()
	 * @category accessing
	 */
	public StRectangle firstMarkerRectangle() {
		if (firstMarkerRectangle == null) {
			StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
			int x = box.originX() + Math.round(box.width() * (float) this.getColorBarModel().interval()[0]);
			StRectangle marker = this.markerRectangle().align_with_(this.markerRectangle().bottomRight(), new Point(x, box.top()));
			firstMarkerRectangle = marker;
		}
		return firstMarkerRectangle;
	}

	/**
	 * Answer my current first marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#lastMarkerRectangle()
	 * @category accessing
	 */
	public StRectangle lastMarkerRectangle() {
		if (lastMarkerRectangle == null) {
			StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
			int x = box.originX() + Math.round(box.width() * (float) this.getColorBarModel().interval()[1]);
			StRectangle marker = this.markerRectangle().align_with_(this.markerRectangle().bottomLeft(), new Point(x, box.top()));
			lastMarkerRectangle = marker;
		}
		return lastMarkerRectangle;
	}

	/**
	 * Answer my current slider rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#sliderRectangle()
	 * @category accessing
	 */
	public StRectangle sliderRectangle() {
		if (sliderRectangle == null) {
			sliderRectangle = this.sliderRectangleForValue_(this.getColorBarModel()._doubleValue());
		}
		return sliderRectangle;
	}

	/**
	 * Answer my current another interval rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#anotherIntervalRectangle()
	 * @category accessing
	 */
	public StRectangle anotherIntervalRectangle() {
		return StRectangle.Origin_corner_(this.firstMarkerRectangle().topRight(), this.lastMarkerRectangle().bottomLeft());
	}

	/**
	 * Answer my current margin.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	protected StRectangle margin() {
		return new StRectangle(8, 8, 8, 8);
	}

	/**
	 * Answer my current marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	protected StRectangle markerRectangle() {
		return new StRectangle(0, 0, 4, 8);
	}

	/**
	 * Answer my current bar area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category area accessing
	 */
	public StRectangle barArea() {
		return new StRectangle(this.getSize());
	}

	/**
	 * Answer the slider rectangle for the value.
	 * 
	 * @param normalizedValue double
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	protected StRectangle sliderRectangleForValue_(double normalizedValue) {
		StRectangle margin = this.margin();
		StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
		int x = box.originX() + Math.round(box.width() * (float) normalizedValue);
		StRectangle slider = StRectangle.Origin_extent_(new Point(x, box.bottom()), new Point(0, margin.bottom()));
		slider = slider.expandedBy_(new StRectangle(margin.left() - 2, 0, margin.right() - 2, 1));
		return slider;
	}

	/**
	 * Answer my current range area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#rangeArea()
	 * @category area accessing
	 */
	public StRectangle rangeArea() {
		return this.barArea().insetBy_(this.margin());
	}

	/**
	 * Answer my current spectrum area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category area accessing
	 */
	public StRectangle spectrumArea() {
		int x1 = this.firstMarkerRectangle().right();
		int x2 = this.lastMarkerRectangle().left() + 1;
		StRectangle area = this.rangeArea();
		return new StRectangle(x1, area.top() + 1, x2, area.bottom() - 1);
	}

	/**
	 * Answer my current ultra first area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category area accessing
	 */
	public StRectangle ultraFirstArea() {
		int right = this.firstMarkerRectangle().right() + 1;
		StRectangle area = this.rangeArea();
		return new StRectangle(area.left(), area.top(), right, area.bottom());
	}

	/**
	 * Answer my current ultra last area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category area accessing
	 */
	public StRectangle ultraLastArea() {
		int left = this.lastMarkerRectangle().left();
		StRectangle area = this.rangeArea();
		return new StRectangle(left, area.top(), area.right(), area.bottom());
	}

	/**
	 * Set the bounds of the view.
	 * Also needs to flush the slider rectangles which need to redraw.
	 *
	 * @param x The new <i>x</i>-coordinate of this component.
	 * @param y The new <i>y</i>-coordinate of this component.
	 * @param width The new <code>width</code> of this component.
	 * @param height The new <code>height</code> of this component.
	 * @see java.awt.Component#setBounds(int, int, int, int)
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);
		this.flushMarkerRectangles();
		this.flushSliderRectangles();
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		if (this.isShowing() == false) {
			return;
		}

		this.displayBackgroundOn_(aGraphics);
		this.displayUltraFirstOn_(aGraphics);
		this.displayUltraLastOn_(aGraphics);
		this.displayIntervalOn_(aGraphics);
		this.displaySpectrumOn_(aGraphics);
		this.displaySliderOn_withColor_(aGraphics, Gray20);
	}

	/**
	 * Display the receiver's background on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayBackgroundOn_(Graphics aGraphics) {
		StRectangle box = this.barArea();
		aGraphics.setColor(Gray70);
		aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
		aGraphics.setColor(Gray80);
		aGraphics.drawLine(box.left(), box.top(), box.right() - 1, box.top());
		aGraphics.drawLine(box.left(), box.top(), box.left(), box.bottom() - 1);
		aGraphics.setColor(Gray20);
		aGraphics.drawLine(box.left(), box.bottom() - 1, box.right() - 1, box.bottom() - 1);
		aGraphics.drawLine(box.right() - 1, box.top(), box.right() - 1, box.bottom() - 1);

		box = this.rangeArea();
		aGraphics.setColor(Gray80);
		aGraphics.drawLine(box.left() + 1, box.top() + 1, box.left() + 1, box.bottom() - 1);
		aGraphics.drawLine(box.left() + 1, box.top() + 1, box.right(), box.top() + 1);
		aGraphics.drawLine(box.left(), box.bottom(), box.right(), box.bottom());
		aGraphics.drawLine(box.right(), box.top(), box.right(), box.bottom());
		aGraphics.setColor(Gray20);
		aGraphics.drawLine(box.left(), box.top(), box.left(), box.bottom() - 1);
		aGraphics.drawLine(box.left(), box.top(), box.right() - 1, box.top());
		aGraphics.drawLine(box.left(), box.bottom() - 1, box.right() - 1, box.bottom() - 1);
		aGraphics.drawLine(box.right() - 1, box.top(), box.right() - 1, box.bottom() - 1);
	}

	/**
	 * Display the receiver's ultra first area on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayUltraFirstOn_(Graphics aGraphics) {
		StRectangle box = this.ultraFirstArea().insetBy_(1);
		aGraphics.setColor(this.getColorBarModel().ultraFirstColor());
		aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
	}

	/**
	 * Display the receiver's ultra last area on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayUltraLastOn_(Graphics aGraphics) {
		StRectangle box = this.ultraLastArea().insetBy_(1);
		aGraphics.setColor(this.getColorBarModel().ultraLastColor());
		aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
	}

	/**
	 * Display the receiver's interval area on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayIntervalOn_(Graphics aGraphics) {
		StRectangle box = this.firstMarkerRectangle();
		aGraphics.setColor(Color.blue);
		aGraphics.drawLine(box.right(), box.bottom(), box.right(), box.top());
		aGraphics.drawLine(box.right(), box.top(), box.left(), box.top());
		aGraphics.drawLine(box.left(), box.top(), box.right(), box.bottom());

		box = this.lastMarkerRectangle();
		aGraphics.setColor(Color.red);
		aGraphics.drawLine(box.left(), box.bottom(), box.left(), box.top());
		aGraphics.drawLine(box.left(), box.top(), box.right(), box.top());
		aGraphics.drawLine(box.right(), box.top(), box.left(), box.bottom());
	}

	/**
	 * Display the receiver's spectrum area on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displaySpectrumOn_(Graphics aGraphics) {
		StRectangle clipBox = this.rangeArea().insetBy_(new StRectangle(1, 0, 1, 0));
		StRectangle box = this.spectrumArea();
		int range = box.width() - 1;
		for (int x = 0; x <= range; x++) {
			Point fromPoint = new Point(x + box.left(), box.top());
			Point toPoint = new Point(x + box.left() + 1, box.bottom());
			StRectangle lineBox = StRectangle.Origin_corner_(fromPoint, toPoint);
			lineBox = lineBox.intersect_(clipBox);
			double normalizedValue = (range == 0) ? 0 : (double) x / range;
			aGraphics.setColor(this.getColorBarModel().getColor_(normalizedValue));
			aGraphics.fillRect(lineBox.x(), lineBox.y(), lineBox.width(), lineBox.height());
		}
	}

	/**
	 * Display the receiver's slider area on the graphics with the specified color.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aColor java.awt.Color
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#displaySliderOn_withColor_(Graphics, Color)
	 * @category displaying
	 */
	public void displaySliderOn_withColor_(Graphics aGraphics, Color aColor) {
		StRectangle box = this.sliderRectangle();
		aGraphics.setColor(aColor);
		aGraphics.fillPolygon(new int[] { box.left(), box.right(), box.topCenter().x }, new int[] { box.bottom(), box.bottom(), box.top() - 1 }, 3);
		savedSliderRectangle = box;
	}

	/**
	 * Answer true if the view is for a vertical bar, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#isVertical()
	 * @category testing
	 */
	public boolean isVertical() {
		return false;
	}

	/**
	 * Action for the update notification.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("value")) {
			sliderRectangle = null;
			StRectangle clipBox = this.sliderRectangle();
			if (clipBox.equals(savedSliderRectangle) == false) {
				if (savedSliderRectangle != null) {
					clipBox = clipBox.merge_(savedSliderRectangle);
				}
				Graphics g = this.getGraphics();
				if (g != null) {
					try {
						g.clipRect(clipBox.x(), clipBox.y(), clipBox.width(), clipBox.height());
						this.displayOn_(g);
					} finally {
						g.dispose();
					}
				}
			}
		} else if (aSymbol == $("interval")) {
			this.flushMarkerRectangles();
			this.repaint();
		} else if (aSymbol == $("color")) {
			this.repaint();
		} else {
			super.update_(evt);
		}
	}

	/**
	 * Set the slider at the point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#setSliderAt(java.awt.Point)
	 * @category setting
	 */
	public void setSliderAt(Point aPoint) {
		this.getColorBarModel().value_(this.valueAtPoint(aPoint));

		Graphics g = this.getGraphics();
		if (g != null) {
			try {
				this.displaySliderOn_withColor_(g, JunColorBarView.Gray40);
			} finally {
				g.dispose();
			}
		}
	}

	/**
	 * Set the first marker at the point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#setFirstMarkerAt(java.awt.Point)
	 * @category setting
	 */
	public void setFirstMarkerAt(Point aPoint) {
		this.getColorBarModel().firstMarker_(this.valueAtPoint(aPoint));
	}

	/**
	 * Set the last marker at the point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#setLastMarkerAt(java.awt.Point)
	 * @category setting
	 */
	public void setLastMarkerAt(Point aPoint) {
		this.getColorBarModel().lastMarker_(this.valueAtPoint(aPoint));
	}

	/**
	 * Set the interval at the point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#setIntervalAt(java.awt.Point)
	 * @category setting
	 */
	public void setIntervalAt(Point aPoint) {
		double value = this.valueAtPoint(aPoint);
		double amount = this.getColorBarModel().lastMarker() - this.getColorBarModel().firstMarker();
		double first = value - amount / 2;
		double last = first + amount;
		if (first < 0) {
			first = 0;
			last = amount;
		}
		if (last > 1) {
			first = 1 - amount;
			last = 1;
		}
		this.getColorBarModel().interval_(new double[] { first, last });
	}

	/**
	 * Convert the point to a value for the model.
	 * 
	 * @param aPoint java.awt.Point
	 * @return double
	 * @category private
	 */
	protected double valueAtPoint(Point aPoint) {
		StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
		double value = (aPoint.x - box.originX()) / (double) box.width();
		return Math.max(0, Math.min(value, 1));
	}

}
