package jp.co.sra.jun.goodies.colors;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.StViewPanel;

/**
 * JunColorBarVerticalViewAwt class
 * 
 *  @author    nisinaka
 *  @created   2004/06/11 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun452 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunColorBarVerticalViewAwt.java,v 8.10 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunColorBarVerticalViewAwt extends JunColorBarViewAwt {

	/**
	 * Create a new StView with a bordered panel which contains JunColorBarViewAwt.
	 * 
	 * @param aColorBarModel jp.co.sra.jun.goodies.colors.JunColorBarModel
	 * @return jp.co.sra.smalltalk.StView 
	 * @category Instance creation
	 */
	public static StView OnBorderedPanel_(JunColorBarModel aColorBarModel) {
		return new StViewPanel(aColorBarModel) {
			protected void buildComponent() {
				JunColorBarViewAwt colorBarView = new JunColorBarVerticalViewAwt((JunColorBarModel) this.model());

				this.setLayout(new BorderLayout());
				this.add(colorBarView._onBorderedPanel(), BorderLayout.CENTER);
			}
		};
	}

	/**
	 * Create a new instance of JunColorBarViewAwt and initialize it.
	 * 
	 * @param aColorBarModel jp.co.sra.jun.goodies.colors.JunColorBarModel
	 * @category Instance creation
	 */
	public JunColorBarVerticalViewAwt(JunColorBarModel aColorBarModel) {
		super(aColorBarModel);
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setSize(40, 200);
	}

	/**
	 * Answer my current first marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#firstMarkerRectangle()
	 * @category accessing
	 */
	public StRectangle firstMarkerRectangle() {
		if (firstMarkerRectangle == null) {
			StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
			int y = box.originY() + Math.round(box.height() * (float) this.getColorBarModel().interval()[0]);
			StRectangle marker = this.markerRectangle().align_with_(this.markerRectangle().bottomRight(), new Point(box.left(), y));
			firstMarkerRectangle = marker;
		}
		return firstMarkerRectangle;
	}

	/**
	 * Answer my current first marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#lastMarkerRectangle()
	 * @category accessing
	 */
	public StRectangle lastMarkerRectangle() {
		if (lastMarkerRectangle == null) {
			StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
			int y = box.originY() + Math.round(box.height() * (float) this.getColorBarModel().interval()[1]);
			StRectangle marker = this.markerRectangle().align_with_(this.markerRectangle().topRight(), new Point(box.left(), y));
			lastMarkerRectangle = marker;
		}
		return lastMarkerRectangle;
	}

	/**
	 * Answer my current another interval rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#anotherIntervalRectangle()
	 * @category accessing
	 */
	public StRectangle anotherIntervalRectangle() {
		return StRectangle.Origin_corner_(this.firstMarkerRectangle().bottomLeft(), this.lastMarkerRectangle().topRight());
	}

	/**
	 * Answer my current marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#markerRectangle()
	 * @category accessing
	 */
	protected StRectangle markerRectangle() {
		return new StRectangle(0, 0, 8, 4);
	}

	/**
	 * Answer the slider rectangle for the value.
	 * 
	 * @param normalizedValue double
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#sliderRectangleForValue_(double)
	 * @category accessing
	 */
	protected StRectangle sliderRectangleForValue_(double normalizedValue) {
		StRectangle margin = this.margin();
		StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
		int y = box.originY() + Math.round(box.height() * (float) normalizedValue);
		StRectangle slider = StRectangle.Origin_extent_(new Point(box.right(), y), new Point(margin.right(), 0));
		slider = slider.expandedBy_(new StRectangle(0, margin.top() - 2, 1, margin.bottom() - 2));
		return slider;
	}

	/**
	 * Answer my current spectrum area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#spectrumArea()
	 * @category area accessing
	 */
	public StRectangle spectrumArea() {
		int y1 = this.firstMarkerRectangle().bottom();
		int y2 = this.lastMarkerRectangle().top() + 1;
		StRectangle area = this.rangeArea();
		return new StRectangle(area.left() + 1, y1, area.right() - 1, y2);
	}

	/**
	 * Answer my current ultra first area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#ultraFirstArea()
	 * @category area accessing
	 */
	public StRectangle ultraFirstArea() {
		int bottom = this.firstMarkerRectangle().bottom() + 1;
		StRectangle area = this.rangeArea();
		return new StRectangle(area.left(), area.top(), area.right(), bottom);
	}

	/**
	 * Answer my current ultra last area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#ultraLastArea()
	 * @category area accessing
	 */
	public StRectangle ultraLastArea() {
		int top = this.lastMarkerRectangle().top();
		StRectangle area = this.rangeArea();
		return new StRectangle(area.left(), top, area.right(), area.bottom());
	}

	/**
	 * Display the receiver's interval area on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#displayIntervalOn_(java.awt.Graphics)
	 * @category displaying
	 */
	protected void displayIntervalOn_(Graphics aGraphics) {
		StRectangle box = this.firstMarkerRectangle();
		aGraphics.setColor(Color.blue);
		aGraphics.drawLine(box.right(), box.bottom(), box.left(), box.bottom());
		aGraphics.drawLine(box.left(), box.bottom(), box.left(), box.top());
		aGraphics.drawLine(box.left(), box.top(), box.right(), box.bottom());

		box = this.lastMarkerRectangle();
		aGraphics.setColor(Color.red);
		aGraphics.drawLine(box.right(), box.top(), box.left(), box.top());
		aGraphics.drawLine(box.left(), box.top(), box.left(), box.bottom());
		aGraphics.drawLine(box.left(), box.bottom(), box.right(), box.top());
	}

	/**
	 * Display the receiver's spectrum area on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#displaySpectrumOn_(java.awt.Graphics)
	 * @category displaying
	 */
	protected void displaySpectrumOn_(Graphics aGraphics) {
		StRectangle clipBox = this.rangeArea().insetBy_(new StRectangle(0, 1, 0, 1));
		StRectangle box = this.spectrumArea();
		int range = box.height() - 1;
		for (int y = 0; y <= range; y++) {
			Point fromPoint = new Point(box.left(), y + box.top());
			Point toPoint = new Point(box.right(), y + box.top() + 1);
			StRectangle lineBox = StRectangle.Origin_corner_(fromPoint, toPoint);
			lineBox = lineBox.intersect_(clipBox);
			double normalizedValue = (range == 0) ? 0 : (double) y / range;
			aGraphics.setColor(this.getColorBarModel().getColor_(normalizedValue));
			aGraphics.fillRect(lineBox.x(), lineBox.y(), lineBox.width(), lineBox.height());
		}
	}

	/**
	 * Display the receiver's slider area on the graphics with the specified color.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aColor java.awt.Color
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#displaySliderOn_withColor_(Graphics, Color)
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#displaySliderOn_withColor_(java.awt.Graphics, java.awt.Color)
	 * @category displaying
	 */
	public void displaySliderOn_withColor_(Graphics aGraphics, Color aColor) {
		StRectangle box = this.sliderRectangle();
		aGraphics.setColor(aColor);
		aGraphics.fillPolygon(new int[] { box.right(), box.right(), box.left()}, new int[] { box.top(), box.bottom(), box.leftCenter().y }, 3);
		savedSliderRectangle = box;
	}

	/**
	 * Answer true if the view is for a vertical bar, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarView#isVertical()
	 * @category testing
	 */
	public boolean isVertical() {
		return true;
	}

	/**
	 * Convert the point to a value for the model.
	 * 
	 * @param aPoint java.awt.Point
	 * @return double
	 * @see jp.co.sra.jun.goodies.colors.JunColorBarViewAwt#valueAtPoint(java.awt.Point)
	 * @category private
	 */
	protected double valueAtPoint(Point aPoint) {
		StRectangle box = this.rangeArea().insetBy_(new StRectangle(0, 0, 1, 1));
		double value = (aPoint.y - box.originY()) / (double) box.height();
		return Math.max(0, Math.min(value, 1));
	}

}
