package jp.co.sra.jun.goodies.catalog;

import java.awt.Dimension;
import java.awt.Point;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.goodies.image.support.JunImageAdjuster;
import jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer;
import jp.co.sra.jun.goodies.utilities.JunImageUtility;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunImageCatalog class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/05/21 (by Mitsuhiro Asada)
 *  @updated   2006/04/11 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun640 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageCatalog.java,v 8.12 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunImageCatalog extends JunVisualCatalog {
	protected int[] visualClip;

	/**
	 * Create a new instance of <code>JunImageCatalog</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunImageCatalog() {
		super();
	}

	/**
	 * Create a new instance of <code>JunImageCatalog</code> and set a directory.
	 * 
	 * @param directoryFilename java.io.File
	 * @category Instance creation
	 */
	public JunImageCatalog(File directoryFilename) {
		super(directoryFilename);
	}

	/**
	 * Create a new instance of <code>JunImageCatalog</code> and set a directory.
	 * 
	 * @param fromDirectoryFilename java.io.File
	 * @param toDirectoryFilename java.io.File
	 * @category Instance creation
	 */
	public JunImageCatalog(File fromDirectoryFilename, File toDirectoryFilename) {
		super(fromDirectoryFilename, toDirectoryFilename);
	}

	/**
	 * Create a new instance of <code>JunImageCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#Dive_(java.io.File)
	 * @category Utilities
	 */
	public static JunVisualCatalog Dive_(File directoryFilename) {
		return _Dive_catalog_(directoryFilename, new JunImageCatalog());
	}

	/**
	 * Create a new instance of <code>JunImageCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#Dive_block_(java.io.File, jp.co.sra.smalltalk.StBlockClosure)
	 * @category Utilities
	 */
	public static JunVisualCatalog Dive_block_(File directoryFilename, StBlockClosure aBlock) {
		return _Dive_block_catalog_(directoryFilename, aBlock, new JunImageCatalog());
	}

	/**
	 * Initialize this when created.
	 * 
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		visualClip = null;
	}

	/**
	 * Answer the visual clip.
	 * 
	 * @return int[]
	 * @category accessing
	 */
	public int[] visualClip() {
		return visualClip;
	}

	/**
	 * Set the visual clip.
	 * 
	 * @param anArray int[]
	 * @category accessing
	 */
	public void visualClip_(int[] anArray) {
		visualClip = anArray;
	}

	/**
	 * Set the visual clip.
	 * 
	 * @param anArray java.awt.Point
	 * @category accessing
	 */
	public void visualClip_(Point anArray) {
		visualClip = new int[] { anArray.x, anArray.y };
	}

	/**
	 * Answer the default title string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#defaultTitleString()
	 * @category defaults
	 */
	public String defaultTitleString() {
		return "Image Catalog";
	}

	/**
	 * Answer the default visual catalog directory name.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#defaultVisualCatalogDirectoryName()
	 * @category defaults
	 */
	public String defaultVisualCatalogDirectoryName() {
		return "ImageCatalog";
	}

	/**
	 * Answer the default visual clip directory name.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#defaultVisualClipDirectoryName()
	 * @category defaults
	 */
	public String defaultVisualClipDirectoryName() {
		return "images";
	}

	/**
	 * Answer the visual clip files.
	 * 
	 * @return java.io.File[]
	 * @category file accessing
	 */
	public File[] visualClipFiles() {
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] visualClipFiles = new File[originalVisualFiles.length];
		for (int i = 0; i < originalVisualFiles.length; i++) {
			String aString = originalVisualFiles[i].getName();
			File aFilename = new File(this.visualClipDirectory(), aString);
			visualClipFiles[i] = aFilename;
		}
		return visualClipFiles;
	}

	/**
	 * Make visual clip files.
	 * 
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#makeVisualClipFiles()
	 * @category making
	 */
	protected void makeVisualClipFiles() {
		progressValue.value_(0.0f);
		int anIndex = 1;
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] visualClipFiles = this.visualClipFiles();

		for (int i = 0; i < originalVisualFiles.length; i++) {
			String aString = $String("Image:") + " " + visualClipFiles[i].getName();
			progressMessage.value_(JunStringUtility._ContractString_to_(aString, this.defaultContractLength()));

			if (originalVisualFiles[i].equals(visualClipFiles[i]) == false) {
				StImage anImage = null;
				JunMoviePlayer aMoviePlayer = null;
				try {
					aMoviePlayer = new JunMoviePlayer(originalVisualFiles[i]);
					anImage = aMoviePlayer.asImage();
				} finally {
					if (aMoviePlayer != null) {
						aMoviePlayer.release();
						aMoviePlayer = null;
					}
				}

				JunImageAdjuster anAdjuster = new JunImageAdjuster();
				Dimension imageExtent = this.imageClipExtent_(anImage);
				anImage = anAdjuster.adjust_extent_(anImage, imageExtent);

				try {
					JunImageUtility.WriteImage_to_(anImage, visualClipFiles[i]);
				} catch (IOException e) {
					System.out.println(e.getMessage());
					e.printStackTrace();
				}
			}

			progressValue.value_(((float) anIndex) / this.numberOfVisuals());
			anIndex = anIndex + 1;
		}
	}

	/**
	 * Answer true if the specified extension string is for an image, otherwise false.
	 * 
	 * @param extensionString java.lang.String
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#incluesExtensions_(java.lang.String)
	 * @category testing
	 */
	public boolean incluesExtensions_(String extensionString) {
		String extension = extensionString.toLowerCase();
		return Arrays.asList(JunSystem.DefaultImageExtensions()).contains(extension);
	}

	/**
	 * Answer the image clip extent.
	 * 
	 * @return java.awt.Dimension
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	protected Dimension imageClipExtent_(StImage anImage) {
		Dimension imageExtent = null;
		if (this.visualClip() == null) {
			imageExtent = anImage.bounds().getSize();
		} else {
			imageExtent = this.proportionalExtentOfImage_within_(anImage, new Dimension(this.visualClip()[0], this.visualClip()[1]));
		}
		return imageExtent;
	}
}
