package jp.co.sra.jun.goodies.calendar;

import java.awt.Point;
import java.awt.Toolkit;
import java.awt.datatransfer.StringSelection;
import java.util.Calendar;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunCalendarDateModel class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2003/10/06 (by Nobuto Matsubara)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun472 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCalendarDateModel.java,v 8.12 2008/02/20 06:31:12 nisinaka Exp $
 */
public class JunCalendarDateModel extends JunCalendarModel {

	protected Calendar calendarDate;
	protected Calendar originalDate;
	protected int[][] calendarMatrix;
	protected StBlockClosure validateBlock;
	protected String[] monthNames;
	protected String[] weekdayNames;
	protected StDisplayable displayObject;
	protected StPopupMenu _popupMenu;

	/**
	 * Create a new instance of <code>JunCalendarDateModel</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunCalendarDateModel() {
		this.initialize();
	}

	/**
	 * Create instance and initialize date.
	 * 
	 * @param aDate Calendar
	 * @category Utilities
	 */
	public static JunCalendarDateModel Date_(Calendar aDate) {
		JunCalendarDateModel calendarDateModel = new JunCalendarDateModel();
		calendarDateModel.date_(aDate);
		return calendarDateModel;
	}

	/**
	 * Create instance and initialize with year, month and day.
	 * 
	 * @param year int
	 * @param month int
	 * @param day int
	 * @category Utilities
	 */
	public static JunCalendarDateModel Year_month_day_(int year, int month, int day) {
		Calendar aDate = Calendar.getInstance();
		aDate.set(year, month, 1);
		aDate.set(aDate.get(Calendar.YEAR), aDate.get(Calendar.MONTH) - 1, Math.max(1, Math.min(day, aDate.getActualMaximum(Calendar.DATE))));
		return JunCalendarDateModel.Date_(aDate);
	}

	/**
	 * initialize.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		//	super.initialize();
		this.initialize_(Calendar.getInstance());
	}

	/**
	 * initialize.
	 * 
	 * @category initialize-release
	 */
	public void initialize_(Calendar aDate) {
		this.date_(aDate);
		originalDate = aDate;
		monthNames = null;
		weekdayNames = null;
		_popupMenu = null;
	}

	/**
	 * Set point to calendar matrix.
	 * @return int
	 * @param aPoint Point
	 * @category accessing
	 */
	public int atPoint_(Point aPoint) {
		int[][] array = this.calendarMatrix();
		return (int) array[aPoint.y][aPoint.x];
	}

	/**
	 * Return current date.
	 * @return java.util.Calendar
	 * @category accessing
	 */
	public Calendar date() {
		return calendarDate;
	}

	/**
	 * set a date.
	 * @category accessing
	 */
	public void date_(Calendar aDate) {
		Calendar oldDate = null, newDate;
		if (this.date() != null) {
			oldDate = (Calendar) this.date().clone();
		}
		newDate = (Calendar) this.validateBlock().value_value_(oldDate, aDate);
		if (newDate == null) {
			return;
		}
		calendarDate = newDate;
		this.makeCalendarMatrix();
		if (oldDate == null) {
			this.changed_($("date"));
			return;
		}
		if (!(oldDate.get(Calendar.YEAR) == calendarDate.get(Calendar.YEAR))) {
			this.changed_($("year"));
			return;
		}
		if (!(oldDate.get(Calendar.MONTH) == calendarDate.get(Calendar.MONTH))) {
			this.changed_($("month"));
			return;
		}
		if (!(oldDate.get(Calendar.DATE) == calendarDate.get(Calendar.DATE))) {
			this.changed_($("day"));
			return;
		}
	}

	/**
	 * Answer current index of day of month.
	 * @return int
	 * @category accessing
	 */
	public int day() {
		return this.date().get(Calendar.DAY_OF_MONTH);
	}

	/**
	 * Set the index of day.
	 * @param day int
	 * @category accessing
	 */
	public void day_(int day) {
		Calendar newDate;
		if (!(1 <= day && day <= this.date().getActualMaximum(Calendar.DATE))) {
			return;
		}
		newDate = Calendar.getInstance();
		newDate.set(this.year(), this.month() - 1, day);
		this.date_(newDate);
	}

	/**
	 * Answer current index of month.
	 * @return int
	 * @category accessing
	 */
	public int month() {
		return this.date().get(Calendar.MONTH) + 1;
	}

	/**
	 * Set month.
	 * @param month int
	 * @category accessing
	 */
	public void month_(int month) {
		Calendar newDate = Calendar.getInstance();
		newDate.set(this.year(), Math.max(0, Math.min(month - 1, 11)), 1);
		newDate.set(newDate.get(Calendar.YEAR), newDate.get(Calendar.MONTH), Math.min(this.day(), newDate.getActualMaximum(Calendar.DATE)));
		this.date_(newDate);
	}

	/**
	 * Answer month names.
	 * @return String[]
	 * @category accessing
	 */
	public String[] monthNames() {
		if (monthNames == null) {
			String[] anArray = new String[12];
			anArray[0] = "1";
			anArray[1] = "2";
			anArray[2] = "3";
			anArray[3] = "4";
			anArray[4] = "5";
			anArray[5] = "6";
			anArray[6] = "7";
			anArray[7] = "8";
			anArray[8] = "9";
			anArray[9] = "10";
			anArray[10] = "11";
			anArray[11] = "12";
			monthNames = anArray;
		}
		return monthNames;
	}

	/**
	 * Answer original date.
	 * @return java.util.Calendar
	 * @category accessing
	 */
	public Calendar original() {
		return originalDate;
	}

	/**
	 * Return valudate block.
	 * @category accessing
	 */
	public StBlockClosure validateBlock() {
		if (validateBlock == null) {
			validateBlock = new StBlockClosure() {
				public Object value_value_(Object oldDate, Object newDate) {
					return newDate;
				}
			};
		}
		return validateBlock;
	}

	/**
	 * Set a blockclosure.
	 * @param aBlock StBlockClosure
	 * @category accessing
	 */
	public void validateBlock_(StBlockClosure aBlock) {
		validateBlock = aBlock;
	}

	/**
	 * Return weekday names.
	 * @return String[]
	 * @category accessing
	 */
	public String[] weekdayNames() {
		if (weekdayNames == null) {
			String[] anArray = new String[7];
			anArray[0] = JunSystem.$String("Su.", "Su");
			anArray[1] = JunSystem.$String("Mo.", "Mo");
			anArray[2] = JunSystem.$String("Tu.", "Tu");
			anArray[3] = JunSystem.$String("We.", "We");
			anArray[4] = JunSystem.$String("Th.", "Th");
			anArray[5] = JunSystem.$String("Fr.", "Fr");
			anArray[6] = JunSystem.$String("Sa.", "Sa");
			weekdayNames = anArray;
		}
		return weekdayNames;
	}

	/**
	 * Answer the current year.
	 * @return int
	 * @category accessing
	 */
	public int year() {
		return this.date().get(Calendar.YEAR);
	}

	/**
	 * Set year.
	 * @param year int
	 * @category accessing
	 */
	public void year_(int year) {
		Calendar newDate = Calendar.getInstance();
		newDate.set(Math.max(year, 100), this.date().get(Calendar.MONTH), 1);
		newDate.set(newDate.get(Calendar.YEAR), newDate.get(Calendar.MONTH), Math.min(this.date().get(Calendar.DATE), newDate.getActualMaximum(Calendar.DATE)));
		this.date_(newDate);
	}

	/**
	 * Action month down.
	 * @category actions
	 */
	public void monthDown() {
		int aYear, aMonth;
		Calendar newDate = Calendar.getInstance();
		if (this.month() == 1) {
			aYear = this.year() - 1;
			aMonth = 12;
		} else {
			aYear = this.year();
			aMonth = this.month() - 1;
		}
		newDate.set(aYear, aMonth - 1, 1);
		newDate.set(newDate.get(Calendar.YEAR), newDate.get(Calendar.MONTH), Math.min(this.day(), newDate.getActualMaximum(Calendar.DATE)));
		this.date_(newDate);
		this.changed_($("month"));
	}

	/**
	 * Action month up.
	 * @category actions
	 */
	public void monthUp() {
		int aYear, aMonth;
		Calendar newDate = Calendar.getInstance();
		if (this.month() == 12) {
			aYear = this.year() + 1;
			aMonth = 1;
		} else {
			aYear = this.year();
			aMonth = this.month() + 1;
		}
		newDate.set(aYear, aMonth - 1, 1);
		newDate.set(newDate.get(Calendar.YEAR), newDate.get(Calendar.MONTH), Math.min(this.day(), newDate.getActualMaximum(Calendar.DATE)));
		this.date_(newDate);
		this.changed_($("month"));
	}

	/**
	 * Action year down.
	 * @category actions
	 */
	public void yearDown() {
		this.year_(this.year() - 1);
		this.changed_($("year"));
	}

	/**
	 * Action year up.
	 * @category actions
	 */
	public void yearUp() {
		this.year_(this.year() + 1);
		this.changed_($("year"));
	}

	/**
	 * Return a calendarDate view.
	 * @return jp.co.sra.smalltalk.StView
	 * @category accessing
	 */
	public StView calendarDateView() {
		StView aView = this.defaultView();
		return aView;
	}

	/**
	 * Return a default view.
	 * @return jp.co.sra.smalltalk.StView
	 * @category defaults
	 */
	public StView defaultView() {
		JunCalendarDateView aView;
		if (GetDefaultViewMode() == VIEW_AWT) {
			aView = new JunCalendarDateViewAwt(this);
		} else {
			aView = new JunCalendarDateViewSwing(this);
		}
		return aView;
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Copy"), new MenuPerformer(this, "copyDate")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Today's date", "Today"), new MenuPerformer(this, "todayDate")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Original date"), new MenuPerformer(this, "originalDate")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Month down"), new MenuPerformer(this, "monthDown")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Month up"), new MenuPerformer(this, "monthUp")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Year down"), new MenuPerformer(this, "yearDown")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Year up"), new MenuPerformer(this, "yearUp")));
		}
		return _popupMenu;
	}

	/**
	 * Copy selected date.
	 * @category menu messages
	 */
	public void copyDate() {
		StringBuffer buf = new StringBuffer();
		buf.append(this.monthNames()[this.month() - 1]);
		buf.append(' ');
		buf.append(this.day());
		buf.append(", ");
		buf.append(this.year());
		Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(buf.toString()), null);
	}

	/**
	 * Set current date to original date.
	 * @category menu messages
	 */
	public void originalDate() {
		this.date_(this.original());
	}

	/**
	 * Set current date to today.
	 * @category menu messages
	 */
	public void todayDate() {
		this.date_(Calendar.getInstance());
	}

	/**
	 * Return calendarMatrix.
	 * @return int[][]
	 * @category accessing
	 */
	public int[][] calendarMatrix() {
		if (calendarMatrix == null) {
			this.makeCalendarMatrix();
		}
		return calendarMatrix;
	}

	/**
	 * make calendar matrix.
	 * @category private
	 */
	private void makeCalendarMatrix() {
		Calendar date;
		int weekday, x, y;
		Integer[][] array;

		date = Calendar.getInstance();
		date.set(this.year(), this.month() - 1, 1);
		weekday = date.get(Calendar.DAY_OF_WEEK);
		array = new Integer[6][7];
		x = weekday - 1;
		y = 0;
		for (int i = 0; i < date.getActualMaximum(Calendar.DATE); ++i) {
			array[y][x] = new Integer(i + 1);
			x = x + 1;
			if (x > 6) {
				x = 0;
				y = y + 1;
			}
		}
		int rowCount = 0;
		for (int i = 0; i < array.length; i++) {
			int nullCount = 0;
			for (int j = 0; j < array[i].length; j++) {
				if (array[i][j] == null) {
					nullCount++;
				}
			}
			if (nullCount < 7) {
				rowCount++;
			}
		}
		int[][] intArray = new int[rowCount][7];
		for (int i = 0; i < rowCount; ++i) {
			for (int j = 0; j < 7; ++j) {
				if (array[i][j] != null) {
					intArray[i][j] = array[i][j].intValue();
				}
			}
		}
		calendarMatrix = intArray;
	}

	/**
	 * Return Window title.
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("YYMMDD", "Date");
	}

	/**
	 * Return current date info.
	 *
	 * @return java.lang.String
	 * @category printing
	 */
	public String toString() {
		StringBuffer buf = new StringBuffer();
		buf.append(this.month());
		buf.append(" ");
		buf.append(this.day());
		buf.append(",");
		buf.append(this.year());
		return buf.toString();
	}

	/**
	 * Return current date info.
	 * 
	 * @return java.lang.String
	 * @category printing
	 */
	public String printCalendarString() {
		StringBuffer buf = new StringBuffer();
		this.printCalendarOn_(buf);
		return buf.toString();
	}

	/**
	 * Print calendar date on aBuffer.
	 * 
	 * @param aBuffer java.lang.StringBuffer
	 * @category printing
	 */
	public void printCalendarOn_(StringBuffer aBuffer) {
		Calendar date = this.date();
		aBuffer.append(date.get(Calendar.YEAR));
		aBuffer.append("	");
		aBuffer.append(this.monthNames()[date.get(Calendar.MONTH)]);
		aBuffer.append("	");
		aBuffer.append("\n");
		for (int index = 0; index < this.weekdayNames().length; ++index) {
			aBuffer.append(this.weekdayNames()[index]);
			if (index < this.weekdayNames().length) {
				aBuffer.append("	");
			}
		}
		aBuffer.append("\n");
		for (int index = 0; index < this.calendarMatrix().length; ++index) {
			for (int index2 = 0; index2 < this.calendarMatrix()[index].length; ++index2) {
				if (this.calendarMatrix()[index][index2] != 0) {
					aBuffer.append(this.calendarMatrix()[index][index2]);
				}
				if (index < this.calendarMatrix().length) {
					aBuffer.append("	");
				}
			}
			aBuffer.append("\n");
		}
	}

	/**
	 * Set new date.
	 * 
	 * @param aDate java.util.Calendar
	 * @category accessing
	 */
	public void setDate_(Calendar aDate) {
		calendarDate = aDate;
	}
}
