package jp.co.sra.jun.geometry.surfaces;

import java.util.ArrayList;
import java.util.List;

import jp.co.sra.jun.geometry.abstracts.JunSurface;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.boundaries.JunBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;

/**
 * JunCompoundSurface class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/07/05 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun610 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCompoundSurface.java,v 8.5 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunCompoundSurface extends JunSurface {
	protected List components;
	protected JunBoundingBox boundingBox;

	/**
	 * Create a new instance of <code>JunCompoundSurface</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunCompoundSurface() {
		super();
	}

	/**
	 * Create a new instance of <code>JunCompoundSurface</code> and initialize it.
	 * 
	 * @param anArrayOfJunSurface jp.co.sra.jun.geometry.abstracts.JunSurface[]
	 * @category Instance creation
	 */
	public JunCompoundSurface(JunSurface[] anArrayOfJunSurface) {
		super();
		this.setComponents_(anArrayOfJunSurface);
	}

	/**
	 * Create a new instance of <code>JunCompoundSurface</code> and initialize it.
	 * 
	 * @param anArrayOfJunSurface java.util.List
	 * @category Instance creation
	 */
	public JunCompoundSurface(List anArrayOfJunSurface) {
		super();
		this.setComponents_(anArrayOfJunSurface);
	}

	/**
	 * Answer the geometry objects as <code>List</code>.
	 * 
	 * @return java.util.List
	 * @category accessing
	 */
	protected List _components() {
		if (components == null) {
			components = new ArrayList();
		}
		return components;
	}

	/**
	 * Answer the receiver's components.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface[]
	 * @category accessing
	 */
	public JunSurface[] components() {
		return (JunSurface[]) this._components().toArray(new JunSurface[this._components().size()]);
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.JunBoundingBox
	 * @category bounds accessing
	 */
	public JunBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = this.preferredBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunCompoundSurface another = (JunCompoundSurface) anObject;
		JunSurface[] myComponents = this.components();
		JunSurface[] anotherComponents = another.components();
		if (myComponents.length != anotherComponents.length) {
			return false;
		}

		for (int i = 0; i < myComponents.length; i++) {
			if (myComponents[i].equal_(anotherComponents[i]) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunCompoundSurface another = (JunCompoundSurface) anObject;
		JunSurface[] myComponents = this.components();
		JunSurface[] anotherComponents = another.components();
		if (myComponents.length != anotherComponents.length) {
			return false;
		}

		for (int i = 0; i < myComponents.length; i++) {
			if (myComponents[i].equals(anotherComponents[i]) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer the distance from specified Point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#distanceFromPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category functions
	 */
	public double distanceFromPoint_(Jun3dPoint aJun3dPoint) {
		return this.nearestPointFromPoint_(aJun3dPoint).distance_(aJun3dPoint);
	}
	
	/**
	 * Answer the intersecting point with the specified line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLine_(Jun3dLine aJun3dLine) { 
		JunSurface[] components = this.components();
		for (int i = 0; i <components.length; i++) {
			try {
			Jun3dPoint intersectingPoint = (Jun3dPoint) components[i].perform_with_("intersectingPointWithLine_", aJun3dLine);
			if (intersectingPoint != null) {
				return intersectingPoint;
			}
			} catch (Exception e) {
				System.out.println(e.getMessage());
				e.printStackTrace();
			}
		}
		return null;
	}

	/**
	 * Answer the intersecting point with the specified line segment.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLineSegment_(Jun3dLine aJun3dLine) { 
		JunSurface[] components = this.components();
		for (int i = 0; i <components.length; i++) {
			try {
			Jun3dPoint intersectingPoint = (Jun3dPoint) components[i].perform_with_("intersectingPointWithLineSegment_", aJun3dLine);
			if (intersectingPoint != null) {
				return intersectingPoint;
			}
			} catch (Exception e) {
				System.out.println(e.getMessage());
				e.printStackTrace();
			}
		}
		return null;
	}

	/**
	 * Answer the distance from specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions
	 */
	public Jun3dPoint nearestPointFromPoint_(Jun3dPoint aJun3dPoint) {
		JunSurface[] components = this.components();
		Jun3dPoint thePoint = null;
		double minDistance;
		try {
			thePoint = (Jun3dPoint) components[0].perform_with_("nearestPointFromPoint_", aJun3dPoint);
			minDistance = aJun3dPoint.distance_(thePoint);
			for (int i = 1; i < components.length; i++) {
				Jun3dPoint point = (Jun3dPoint) components[i].perform_with_("nearestPointFromPoint_", aJun3dPoint);
				double distance = aJun3dPoint.distance_(point);
				if (minDistance > distance) {
					thePoint = point;
					minDistance = distance;
				}
			}
		} catch (Exception e) {
			System.out.println(e.getMessage());
			e.printStackTrace();
		}

		return thePoint;
	}

	/**
	 * Answer <code>true</code> if the receiver is a compound surface, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isCompoundSurface()
	 * @category testing
	 */
	public boolean isCompoundSurface() {
		return true;
	}

	/**
	 * Answer the receiver's preferred bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.JunBoundingBox
	 * @category private
	 */
	protected JunBoundingBox preferredBoundingBox() {
		JunSurface[] components = this.components();
		JunBoundingBox boundingBox = null;
		try {
			if (components[0].is2d()) {
				Jun2dBoundingBox bounds = ((Jun2dBoundingBox) components[0].perform_("boundingBox"));
				for (int i = 1; i < components.length; i++) {
					bounds.merge_(((Jun2dBoundingBox) components[i].perform_("boundingBox")));
				}
				boundingBox = bounds;
			} else {
				Jun3dBoundingBox bounds = ((Jun3dBoundingBox) components[0].perform_("boundingBox"));
				for (int i = 1; i < components.length; i++) {
					bounds.merge_(((Jun3dBoundingBox) components[i].perform_("boundingBox")));
				}
				boundingBox = bounds;
			}
		} catch (Exception e) {
			System.out.println(e.getMessage());
			e.printStackTrace();
		}

		return boundingBox;
	}

	/**
	 * Set the receiver's components.
	 * 
	 * @param anArrayOfJunSurface java.util.List
	 * @category private
	 */
	protected void setComponents_(List anArrayOfJunSurface) {
		components = anArrayOfJunSurface;
		boundingBox = null;
	}

	/**
	 * Set the receiver's components.
	 * 
	 * @param anArrayOfJunSurface jp.co.sra.jun.geometry.abstracts.JunSurface[]
	 * @category private
	 */
	protected void setComponents_(JunSurface[] anArrayOfJunSurface) {
		List aList = new ArrayList();
		for (int i = 0; i < anArrayOfJunSurface.length; i++) {
			aList.add(anArrayOfJunSurface[i]);
		}
		this.setComponents_(aList);
	}
}
