package jp.co.sra.jun.geometry.surfaces;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun2dCircle class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/21 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun696 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dCircle.java,v 8.18 2008/02/20 06:30:58 nisinaka Exp $
 */
public class Jun2dCircle extends JunCircle {

	protected double x0;
	protected double y0;

	/**
	 * Create a new instance of Jun2dCircle and initialize it.
	 * 
	 * @category Instance creation
	 */
	private Jun2dCircle() {
		super();
	}

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public Jun2dCircle(double radiusValue) {
		this(new Jun2dPoint(0, 0), radiusValue);
	}

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public Jun2dCircle(Jun2dPoint centerPoint, double radiusValue) {
		super();
		this.setCenter_(centerPoint);
		this.setRadius_(radiusValue);
	}

	/**
	 * Answer the receiver's center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint center() {
		return new Jun2dPoint(x0, y0);
	}

	/**
	 * Answer my center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#_center()
	 * @category accessing
	 */
	protected JunPoint _center() {
		return this.center();
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (super.equal_(anObject) == false) {
			return false;
		}

		Jun2dCircle aCircle = (Jun2dCircle) anObject;
		return this.isEqualNumber_to_(x0, aCircle.x0) && this.isEqualNumber_to_(y0, aCircle.y0);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (super.equals(anObject) == false) {
			return false;
		}

		Jun2dCircle aCircle = (Jun2dCircle) anObject;
		return this.x0() == aCircle.x0() && this.y0() == aCircle.y0();
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject aCircle = JunOpenGL3dObject.CircleBy_radius_(10, this.radius()).translatedBy_(Jun3dPoint.Coerce_(this.center()));
		aCircle.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_(Jun2dCircle.this.defaultColor());
				return null;
			}
		});
		return aCircle;
	}

	/**
	 * Convert to a JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#asPlane()
	 * @category converting
	 */
	public JunPlane asPlane() {
		Jun3dPoint centerPoint = this.center().as3dPoint();
		Jun3dLine aLine = centerPoint.to_(new Jun3dPoint(this.center(), 1));
		Jun3dPoint aPoint = centerPoint.plus_(new Jun3dPoint(this.radius(), 0, 0));
		return new JunPlane(aPoint, aPoint.transform_(Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(120), aLine)), aPoint.transform_(Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(240), aLine)));
	}

	/**
	 * Convert to a JunTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category converting
	 */
	public Jun2dTriangle asTriangle() {
		Jun2dPoint centerPoint = this.center();
		Jun2dPoint aPoint = centerPoint.plus_(new Jun2dPoint(this.radius(), 0));
		return new Jun2dTriangle(aPoint, aPoint.transform_(Jun2dTransformation.Rotate_around_(JunAngle.FromDeg_(120), centerPoint)), aPoint.transform_(Jun2dTransformation.Rotate_around_(JunAngle.FromDeg_(240), centerPoint)));
	}

	/**
	 * Answer the receiver's track points with the specified division number.
	 * 
	 * @param divisionNumber int
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category functions
	 */
	public Jun2dPoint[] trackPointsBy_(int divisionNumber) {
		Jun3dPoint[] a3dPoints = new Jun3dCircle(new Jun3dPoint(this.center().x(), this.center().y(), 0), this.radius(), new Jun3dPoint(0, 0, 1)).trackPointsBy_(divisionNumber);
		Jun2dPoint[] a2dPoints = new Jun2dPoint[a3dPoints.length];
		for (int i = 0; i < a3dPoints.length; i++) {
			a2dPoints[i] = new Jun2dPoint(a3dPoints[i].x(), a3dPoints[i].y());
		}
		return a2dPoints;
	}

	/**
	 * Answer the parameter x0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double x0() {
		return x0;
	}

	/**
	 * Answer the parameter y0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double y0() {
		return y0;
	}

	/**
	 * Answer true if the receiver is a 2d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer the new <code>Jun2dCircle</code> which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category transforming
	 */
	public Jun2dCircle rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun2dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new <code>Jun2dCircle</code> which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category transforming
	 */
	public Jun2dCircle scaledBy_(double aNumber) {
		return this.transform_(Jun2dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new <code>Jun2dCircle</code> which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category transforming
	 */
	public Jun2dCircle scaledBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply a transformation 'aTransformation' to the receiver.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category transforming
	 */
	public Jun2dCircle transform_(Jun2dTransformation aTransformation) {
		return new Jun2dCircle(this.center().transform_(aTransformation), new Jun2dPoint(0, 0).to_(new Jun2dPoint(this.radius(), 0)).transform_(aTransformation).length());
	}

	/**
	 * Answer the new <code>Jun2dCircle</code> which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category transforming
	 */
	public Jun2dCircle translatedBy_(double aNumber) {
		return this.transform_(Jun2dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the new <code>Jun2dCircle</code> which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category transforming
	 */
	public Jun2dCircle translatedBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Translate_(aPoint));
	}

	/**
	 * Set the receiver's center point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setCenter_(JunPoint aPoint) {
		Jun2dPoint centerPoint = Jun2dPoint.Coerce_(aPoint);
		this.setX0_(centerPoint.x());
		this.setY0_(centerPoint.y());
	}

	/**
	 * Set the parameter x0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setX0_(double aNumber) {
		x0 = aNumber;
	}

	/**
	 * Set the parameter y0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setY0_(double aNumber) {
		y0 = aNumber;
	}
}
