package jp.co.sra.jun.geometry.pluralities;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.boundaries.JunBoundingBox;
import jp.co.sra.jun.geometry.boundaries.JunBoundingObject;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.goodies.image.support.JunImageAdjuster;
import jp.co.sra.jun.goodies.image.support.JunImageProcessor;

/**
 * Jun2dBoundingBoxes class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2006/04/24 (by Mitsuhiro Asada)
 *  @updated   2007/05/29 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dBoundingBoxes.java,v 8.15 2008/02/20 06:30:57 nisinaka Exp $
 */
public class Jun2dBoundingBoxes extends JunBoundingBoxes {
	/**
	 * Create a new instance of <code>Jun2dBoundingBoxes</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public Jun2dBoundingBoxes() {
		super();
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxes</code> with polyline and bit size.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category Instance creation
	 */
	public static Jun2dBoundingBoxes FromPolyline_(Jun2dPoint[] pointCollection) {
		return Jun2dBoundingBoxes.FromPolyline_bitSize_(pointCollection, new Dimension(16, 16));
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxes</code> with polyline and bit size.
	 * 
	 * @param pointCollection java.awt.Point[]
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category Instance creation
	 */
	public static Jun2dBoundingBoxes FromPolyline_(Point[] pointCollection) {
		return Jun2dBoundingBoxes.FromPolyline_bitSize_(pointCollection, new Dimension(16, 16));
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxes</code> with polyline and bit size.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param bitSize java.awt.Dimension
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category Instance creation
	 */
	public static Jun2dBoundingBoxes FromPolyline_bitSize_(Jun2dPoint[] pointCollection, Dimension bitSize) {
		Jun2dBoundingBoxes boxes = new Jun2dBoundingBoxes();
		if (pointCollection == null || pointCollection.length == 0) {
			return boxes;
		}
		Point[] points = new Point[pointCollection.length];
		int[] xPoints = new int[pointCollection.length];
		int[] yPoints = new int[pointCollection.length];
		for (int i = 0; i < pointCollection.length; i++) {
			points[i] = pointCollection[i]._toPoint();
			xPoints[i] = points[i].x;
			yPoints[i] = points[i].y;
		}
		Dimension dot = new Dimension(Math.max(bitSize.width, 0), Math.max(bitSize.height, 0));
		if (dot.width * dot.height <= 0) {
			return boxes;
		}
		Rectangle box = new Rectangle(points[0].x, points[0].y, 1, 1);
		for (int i = 1; i < points.length; i++) {
			box.add(new Rectangle(points[i].x, points[i].y, 1, 1));
		}
		Point origin = box.getLocation();
		Point corner = new Point(box.x + box.width, box.y + box.height);
		Dimension extent = new Dimension((int) Math.floor(Math.round(box.width / (double) dot.width)), (int) Math.floor(Math.round(box.height / (double) dot.height)));
		extent = new Dimension(Math.max(extent.width, 0), Math.max(extent.height, 0));
		if (extent.width * extent.height <= 0) {
			return boxes;
		}

		StImage pixmap = new StImage(box.width, box.height);
		Graphics2D aGraphics = (Graphics2D) pixmap.image().getGraphics();
		try {
			aGraphics.setColor(Color.white);
			aGraphics.fillRect(0, 0, extent.width, extent.height);
			aGraphics.setColor(Color.black);
			aGraphics.translate(-box.x, -box.y);
			Polygon aPolygon = new Polygon(xPoints, yPoints, xPoints.length);
			aGraphics.fill(aPolygon);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		StImage image = JunImageAdjuster.Adjust_extent_(pixmap, extent);
		image = image.convertToPalette_(JunImageProcessor._WhiteBlackPalette());

		boxes.add_(new Jun2dBoundingBox(StRectangle.Origin_corner_(origin, corner).toRectangle()));
		for (int y = 0; y < image.height(); y++) {
			for (int x = 0; x < image.width(); x++) {
				if (image.getPixel(x, y) == Color.black.getRGB()) {
					boxes.add_(Jun2dBoundingBox.Origin_extent_(new Jun2dPoint(origin.x + x * dot.width, origin.y + y * dot.height), new Jun2dPoint(dot.width, dot.height)));
				}
			}
		}

		final Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		boundingBoxes.add_(Jun2dBoundingBox.Origin_extent_(new Jun2dPoint(box.getLocation()), new Jun2dPoint(box.width, box.height)));
		boxes.differenceBoxes().do_(new StBlockClosure() {
			public Object value_(Object each) {
				boundingBoxes.add_((Jun2dBoundingBox) each);
				return null;
			}
		});
		return boundingBoxes.differenceBoxes();
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxes</code> with polyline and bit size.
	 * 
	 * @param pointCollection java.awt.Point[]
	 * @param bitSize java.awt.Dimension
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category Instance creation
	 */
	public static Jun2dBoundingBoxes FromPolyline_bitSize_(Point[] pointCollection, Dimension bitSize) {
		Jun2dPoint[] points = new Jun2dPoint[pointCollection.length];
		for (int i = 0; i < pointCollection.length; i++) {
			points[i] = new Jun2dPoint(pointCollection[i].x, pointCollection[i].y);
		}
		return Jun2dBoundingBoxes.FromPolyline_bitSize_(points, bitSize);
	}

	/**
	 * Answer the this area size
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		double area = 0.0d;
		JunBoundingObject[] myComponents = this.boundingObjects();
		for (int i = 0; i < myComponents.length; i++) {
			JunBoundingBox boundingBox = (JunBoundingBox) myComponents[i];
			area = area + boundingBox.area();
		}
		return area;
	}

	/**
	 * Answer my current bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall boundingBall() {
		if (this.isEmpty()) {
			return new Jun2dBoundingBall(null);
		}
		Jun2dBoundingBall boundingBall = null;
		JunBoundingObject[] objects = this.boundingObjects();
		for (int i = 0; i < objects.length; i++) {
			Jun2dBoundingBox each = (Jun2dBoundingBox) objects[i];
			if (boundingBall == null) {
				boundingBall = each.boundingBall();
			} else {
				boundingBall = each.boundingBall().merge_(boundingBall);
			}
		}
		return boundingBall;
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		if (this.isEmpty()) {
			return new Jun2dBoundingBox();
		}
		Jun2dBoundingBox boundingBox = null;
		JunBoundingObject[] objects = this.boundingObjects();
		for (int i = 0; i < objects.length; i++) {
			Jun2dBoundingBox each = (Jun2dBoundingBox) objects[i];
			if (boundingBox == null) {
				boundingBox = each;
			} else {
				boundingBox = each.merge_(boundingBox);
			}
		}
		return boundingBox;
	}

	/**
	 * Answer my bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox[]
	 * @category accessing
	 */
	public Jun2dBoundingBox[] boundingBoxes() {
		return (Jun2dBoundingBox[]) this._boundingObjects().toArray(new Jun2dBoundingBox[this._boundingObjects().size()]);
	}

	/**
	 * Convert to a Jun2dBoundingBall.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category converting
	 */
	public Jun2dBoundingBall asBoundingBall() {
		return this.boundingBall();
	}

	/**
	 * Convert to a Jun2dBoundingBox.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category converting
	 */
	public Jun2dBoundingBox asBoundingBox() {
		return this.boundingBox();
	}

	/**
	 * Convert to a Rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.geometry.pluralities.JunBoundingBoxes#asRectangle()
	 * @category converting
	 */
	public Rectangle asRectangle() {
		return this.asBoundingBox().asRectangle();
	}

	/**
	 * Convert to an array of Rectangle.
	 * 
	 * @return java.awt.Rectangle[]
	 * @see jp.co.sra.jun.geometry.pluralities.JunBoundingBoxes#asRectangles()
	 * @category converting
	 */
	public Rectangle[] asRectangles() {
		final Collection aCollection = new ArrayList();
		this.do_(new StBlockClosure() {
			public Object value_(Object obj) {
				JunBoundingBox boundingBox = (JunBoundingBox) obj;
				aCollection.add(boundingBox.asRectangle());
				return null;
			}
		});
		return (Rectangle[]) aCollection.toArray(new Rectangle[aCollection.size()]);
	}

	/**
	 * Answer the receiver's complement boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category functions
	 */
	public Jun2dBoundingBoxes complementBoxes() {
		Jun2dBoundingBoxes complementBoxes = this.unionBoxes();
		Jun2dBoundingBox intersectionBox = this.intersectionBox();
		if (intersectionBox == null) {
			return complementBoxes;
		}

		Jun2dBoundingBox[] boundingObjects = complementBoxes.boundingBoxes();
		for (int i = 0; i < boundingObjects.length; i++) {
			if (boundingObjects[i].equals(intersectionBox)) {
				complementBoxes._boundingObjects().remove(boundingObjects[i]);
			}
		}
		complementBoxes = (Jun2dBoundingBoxes) complementBoxes.reject_(new StBlockClosure() {
			public Object value_(Object aBox) {
				return new Boolean(((Jun2dBoundingBox) aBox).isEmpty());
			}
		});
		return complementBoxes;
	}

	/**
	 * Answer the difference boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category functions
	 */
	public Jun2dBoundingBoxes differenceBoxes() {
		if (this.isEmpty()) {
			return (Jun2dBoundingBoxes) this.copy();
		}

		Jun2dBoundingBox[] rectangles = this.boundingBoxes();
		Jun2dBoundingBox firstRectangle = rectangles[0];
		Jun2dBoundingBoxes differenceBoxes = new Jun2dBoundingBoxes();
		differenceBoxes.add_(firstRectangle);
		for (int i = 1; i < rectangles.length; i++) {
			Jun2dBoundingBox aRectangle = rectangles[i];
			Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
			JunBoundingObject[] myComponents = differenceBoxes.boundingObjects();
			for (int j = 0; j < myComponents.length; j++) {
				Jun2dBoundingBox each = (Jun2dBoundingBox) myComponents[j];
				if (each.intersects_(firstRectangle)) {
					boundingBoxes.addAll_(each.boxesOutside_(aRectangle));
				} else {
					boundingBoxes.add_(each);
				}
			}

			differenceBoxes = boundingBoxes;
		}
		differenceBoxes = (Jun2dBoundingBoxes) differenceBoxes.reject_(new StBlockClosure() {
			public Object value_(Object aBox) {
				return new Boolean(((Jun2dBoundingBox) aBox).isEmpty());
			}
		});
		return differenceBoxes;
	}

	/**
	 * Answer the intersection box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox intersectionBox() {
		if (this.isEmpty()) {
			return null;
		}
		Jun2dBoundingBox intersectionBox = null;
		JunBoundingObject[] objects = this.boundingObjects();
		for (int i = 0; i < objects.length; i++) {
			Jun2dBoundingBox each = (Jun2dBoundingBox) objects[i];
			if (intersectionBox == null) {
				intersectionBox = each;
			} else {
				if (intersectionBox.intersects_(each)) {
					intersectionBox = intersectionBox.intersect_(each);
				}
			}
		}
		if (intersectionBox.isEmpty()) {
			return null;
		}
		return intersectionBox;
	}

	/**
	 * Answer the intersection boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category functions
	 */
	public Jun2dBoundingBoxes intersectionBoxes() {
		Jun2dBoundingBox intersectionBox = this.intersectionBox();
		if (intersectionBox == null) {
			return new Jun2dBoundingBoxes();
		}

		Jun2dBoundingBoxes intersectionBoxes = new Jun2dBoundingBoxes();
		intersectionBoxes.add_(intersectionBox);
		return intersectionBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes with specified level.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category functions
	 */
	public Jun2dBoundingBoxes subdivide2Level_(int anInteger) {
		Jun2dBoundingBoxes boundingBoxes = this;
		for (int i = 0; i < anInteger; i++) {
			boundingBoxes = boundingBoxes.subdivide2();
		}
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's unify.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox unify() {
		return this.asBoundingBox();
	}

	/**
	 * Answer the receiver's union boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category functions
	 */
	public Jun2dBoundingBoxes unionBoxes() {
		Jun2dBoundingBoxes unionBoxes = new Jun2dBoundingBoxes();
		Jun2dBoundingBox intersectionBox = this.intersectionBox();
		if (intersectionBox == null) {
		} else {
			Jun2dBoundingBox[] tmp = this.boundingBoxes();
			Jun2dBoundingBox[] boxes = new Jun2dBoundingBox[tmp.length + 1];
			for (int i = 0; i < tmp.length; i++) {
				boxes[i] = tmp[i];
			}
			boxes[boxes.length - 1] = intersectionBox;

			for (int i = 0; i < boxes.length; i++) {
				Jun2dBoundingBox aBox = boxes[i];
				Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
				boundingBoxes.add_(aBox);
				Jun2dBoundingBox[] myComponents = unionBoxes.boundingBoxes();
				for (int j = 0; j < myComponents.length; j++) {
					Jun2dBoundingBox each = myComponents[j];
					if (each.intersects_(aBox)) {
						boundingBoxes.addAll_(each.boxesOutside_(aBox));
					} else {
						boundingBoxes.add_(each);
					}
				}
				unionBoxes = boundingBoxes;
			}
		}
		unionBoxes = (Jun2dBoundingBoxes) unionBoxes.reject_(new StBlockClosure() {
			public Object value_(Object aBox) {
				return new Boolean(((Jun2dBoundingBox) aBox).isEmpty());
			}
		});
		return unionBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide() {
		return this.subdivide4();
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide2() {
		Collection boundingBoxArray = new ArrayList(this.size() * 2);
		for (int n = 0; n < this.size(); n++) {
			Jun2dBoundingBox[] boundingBoxes = this.boundingBox().subdivide2().boundingBoxes();
			boundingBoxArray.add(boundingBoxes[0]);
			boundingBoxArray.add(boundingBoxes[1]);
		}
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		boundingBoxes.boundingBoxes_(boundingBoxArray);
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide4() {
		Collection boundingBoxArray = new ArrayList(this.size() * 4);
		for (int n = 0; n < this.size(); n++) {
			Jun2dBoundingBox[] boundingBoxes = this.boundingBox().subdivide4().boundingBoxes();
			boundingBoxArray.add(boundingBoxes[0]);
			boundingBoxArray.add(boundingBoxes[1]);
			boundingBoxArray.add(boundingBoxes[2]);
			boundingBoxArray.add(boundingBoxes[3]);
		}
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		boundingBoxes.boundingBoxes_(boundingBoxArray);
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes with specified level.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivideLevel_(int anInteger) {
		Jun2dBoundingBoxes boundingBoxes = this;
		for (int i = 0; i < anInteger; i++) {
			boundingBoxes = boundingBoxes.subdivide();
		}
		return boundingBoxes;
	}

	/**
	 * Answer true if the receiver is a 2d geometry element.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer <code>true</code> if the receiver touches the specified bounding object, otherwise <code>false</code>.
	 * 
	 * @param aBoundingObject jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @return boolean
	 * @category testing
	 */
	public boolean touches_(final Jun2dBoundingBox aBoundingObject) {
		return this.detect_ifNone_(new StBlockClosure() {
			public Object value(Object each) {
				return new Boolean(((Jun2dBoundingBox) each).touches_(aBoundingObject));
			}
		}, new StBlockClosure() {
			public Object value(Object each) {
				return null;
			}
		}) != null;
	}

	/**
	 * Apply a transformation 'aTransformation' to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dBoundingBoxes transform_(Jun2dTransformation aTransformation) {
		Jun2dBoundingBoxes transformedCopy = new Jun2dBoundingBoxes();
		JunBoundingObject[] boundingObjects = this.boundingObjects();
		for (int i = 0; i < boundingObjects.length; i++) {
			this.add_(boundingObjects[i].transform_(aTransformation));
		}
		return transformedCopy;
	}
}
