package jp.co.sra.jun.geometry.boundaries;

import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun3dBoundingBox class
 * 
 *  @author    nisinaka
 *  @created   1998/10/01 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/18 (by Mitsuhiro Asada)
 *  @updated   2007/05/29 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun693 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dBoundingBox.java,v 8.20 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun3dBoundingBox extends JunBoundingBox {
	/**
	 * Create a new instance of Jun3dBoundingBox and initialize it.
	 * 
	 * @category Instance creation
	 */
	public Jun3dBoundingBox() {
		super();
	}

	/**
	 * Create a new instance of Jun3dBoundingBox and initialize it with the
	 * collection of points.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Instance creation
	 */
	public Jun3dBoundingBox(Jun3dPoint[] pointCollection) {
		if (pointCollection == null || pointCollection.length == 0) {
			return;
		}

		this.origin_corner_(pointCollection[0], pointCollection[0]);
		for (int i = 1; i < pointCollection.length; i++) {
			this.add_(pointCollection[i]);
		}
	}

	/**
	 * Create a new instance of Jun3dBoundingBox and initialize it with the
	 * <code>origin</code> point and the <code>corner</code> point.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param cornerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Instance creation
	 */
	public static Jun3dBoundingBox Origin_corner_(Jun3dPoint originPoint, Jun3dPoint cornerPoint) {
		Jun3dBoundingBox aBoundingBox = new Jun3dBoundingBox();
		aBoundingBox.origin_corner_(originPoint, cornerPoint);
		return aBoundingBox;
	}

	/**
	 * Create a new instance of Jun3dBoundingBox and initialize it with the
	 * <code>origin</code> point and the <code>extent</code> point.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Instance creation
	 */
	public static Jun3dBoundingBox Origin_extent_(Jun3dPoint originPoint, Jun3dPoint extentPoint) {
		Jun3dBoundingBox aBoundingBox = new Jun3dBoundingBox();
		aBoundingBox.origin_extent_(originPoint, extentPoint);
		return aBoundingBox;
	}

	/**
	 * Create a new instance of Jun3dBoundingBox and initialize it with the
	 * <code>vertex</code> points.
	 * 
	 * @param vertexPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param vertexPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Instance creation
	 */
	public static Jun3dBoundingBox Vertex_vertex_(Jun3dPoint vertexPoint1, Jun3dPoint vertexPoint2) {
		Jun3dBoundingBox aBoundingBox = new Jun3dBoundingBox();
		aBoundingBox.vertex_vertex_(vertexPoint1, vertexPoint2);
		return aBoundingBox;
	}

	/**
	 * Answer a new Jun2dBoundingBox with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Constants access
	 */
	public static Jun3dBoundingBox Unity() {
		return Jun3dBoundingBox.Origin_corner_(Jun3dPoint.Zero(), Jun3dPoint.Unity());
	}

	/**
	 * Answer a new Jun2dBoundingBox with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Constants access
	 */
	public static Jun3dBoundingBox Zero() {
		return Jun3dBoundingBox.Origin_corner_(Jun3dPoint.Zero(), Jun3dPoint.Zero());
	}

	/**
	 * Initialize this object.
	 * 
	 * @category initialize-release
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 */
	protected void initialize() {
		super.initialize();
		this._origin(Jun3dPoint.Zero());
		this._corner(Jun3dPoint.Zero());
	}

	/**
	 * Answer my area size.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return (this.width() * this.height() + this.height() * this.depth() + this.depth() * this.width()) * 2;
	}

	/**
	 * Answer the bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category accessing
	 */
	public Jun3dBoundingBall boundingBall() {
		Jun3dPoint centerPoint = this.center();
		return new Jun3dBoundingBall(centerPoint, this.origin().distance_(centerPoint));
	}

	/**
	 * Answer the bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return this;
	}

	/**
	 * Answer the center point of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint center() {
		return this.origin().center_(this.corner());
	}

	/**
	 * Answer the <code>corner</code> point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint corner() {
		return (Jun3dPoint) this._corner();
	}

	/**
	 * Set the <code>corner</code> point.
	 * 
	 * @param cornerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void corner_(Jun3dPoint cornerPoint) {
		this.origin_corner_(this.origin(), cornerPoint);
	}

	/**
	 * Answer the depth of the receiver.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#depth()
	 * @category accessing
	 */
	public double depth() {
		return this.corner().z() - this.origin().z();
	}

	/**
	 * Answer the extent of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint extent() {
		return this.corner().minus_(this.origin());
	}

	/**
	 * Answer the inner bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category accessing
	 */
	public Jun3dBoundingBall innerBoundingBall() {
		Jun3dPoint centerPoint = this.center();
		return new Jun3dBoundingBall(centerPoint, Math.min(Math.min(this.width(), this.height()), this.depth()) / 2);
	}

	/**
	 * Answer the inner bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox innerBoundingBox() {
		return this;
	}

	/**
	 * Answer the receiver's octa bounding balls.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls
	 * @category accessing 
	 */
	public Jun3dBoundingBalls octaBoundingBalls() {
		Jun3dBoundingBalls boundingBalls = new Jun3dBoundingBalls();
		Jun3dBoundingBoxes octaBoundingBoxes = this.octaBoundingBoxes();
		Jun3dBoundingBox[] boundingBoxes = octaBoundingBoxes.boundingBoxes();
		for (int i = 0; i < boundingBoxes.length; i++) {
			boundingBalls.add_(boundingBoxes[i].outerBoundingBall());
		}
		return boundingBalls;
	}

	/**
	 * Answer the receiver's octa bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category accessing 
	 */
	public Jun3dBoundingBoxes octaBoundingBoxes() {
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		Jun3dBoundingBox boundingBox = null;
		double y = Double.NaN;
		double z = Double.NaN;
		boundingBoxes.add_(boundingBox = Jun3dBoundingBox.Origin_corner_(this.origin(), this.center()));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun3dPoint(0, 0, (z = this.depth() / 2))));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun3dPoint(0, (y = this.height() / 2), z)));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun3dPoint(0, y, 0)));
		boundingBoxes.add_(boundingBox = Jun3dBoundingBox.Origin_corner_(this.center(), this.corner()));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun3dPoint(0, 0 - y, 0)));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun3dPoint(0, 0 - y, 0 - z)));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun3dPoint(0, 0, 0 - z)));

		return boundingBoxes;
	}

	/**
	 * Answer the <code>origin</code> point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint origin() {
		return (Jun3dPoint) this._origin();
	}

	/**
	 * Set the <code>origin</code> point.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void origin_(Jun3dPoint originPoint) {
		this.origin_corner_(originPoint, this.corner());
	}

	/**
	 * Set the <code>origin</code> point and the <code>corner</code> point.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param cornerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void origin_corner_(Jun3dPoint originPoint, Jun3dPoint cornerPoint) {
		this._origin(originPoint);
		this._corner(cornerPoint);
	}

	/**
	 * Set the <code>origin</code> point and the <code>corner</code> point. The
	 * <code>corner</code> point is calculated with the <code>extent</code>.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void origin_extent_(Jun3dPoint originPoint, Jun3dPoint extentPoint) {
		this.origin_corner_(originPoint, originPoint.plus_(extentPoint));
	}

	/**
	 * Answer the receiver's outer bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category accessing
	 */
	public Jun3dBoundingBall outerBoundingBall() {
		return this.boundingBall();
	}

	/**
	 * Answer the receiver's outer bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox outerBoundingBox() {
		return this.boundingBox();
	}

	/**
	 * Answer the receiver's p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p1() {
		return new Jun3dPoint(this.origin().x(), this.origin().y(), this.origin().z());
	}

	/**
	 * Answer the receiver's p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p2() {
		return new Jun3dPoint(this.corner().x(), this.origin().y(), this.origin().z());
	}

	/**
	 * Answer the receiver's p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p3() {
		return new Jun3dPoint(this.corner().x(), this.corner().y(), this.origin().z());
	}

	/**
	 * Answer the receiver's p4 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p4() {
		return new Jun3dPoint(this.origin().x(), this.corner().y(), this.origin().z());
	}

	/**
	 * Answer the receiver's p5 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p5() {
		return new Jun3dPoint(this.corner().x(), this.corner().y(), this.corner().z());
	}

	/**
	 * Answer the receiver's p6 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p6() {
		return new Jun3dPoint(this.origin().x(), this.corner().y(), this.corner().z());
	}

	/**
	 * Answer the receiver's p7 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p7() {
		return new Jun3dPoint(this.origin().x(), this.origin().y(), this.corner().z());
	}

	/**
	 * Answer the receiver's p8 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p8() {
		return new Jun3dPoint(this.corner().x(), this.origin().y(), this.corner().z());
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] pp1() {
		return new Jun3dPoint[] { this.p1(), this.p4(), this.p3(), this.p2() };
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] pp2() {
		return new Jun3dPoint[] { this.p1(), this.p2(), this.p8(), this.p7() };
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] pp3() {
		return new Jun3dPoint[] { this.p1(), this.p7(), this.p6(), this.p4() };
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] pp4() {
		return new Jun3dPoint[] { this.p5(), this.p3(), this.p4(), this.p6() };
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] pp5() {
		return new Jun3dPoint[] { this.p5(), this.p8(), this.p2(), this.p3() };
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] pp6() {
		return new Jun3dPoint[] { this.p5(), this.p6(), this.p7(), this.p8() };
	}

	/**
	 * Answer the receiver's point arrays.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[][] pps() {
		return new Jun3dPoint[][] { this.pp1(), this.pp2(), this.pp3(), this.pp4(), this.pp5(), this.pp6() };
	}

	/**
	 * Set the <code>origin</code> point and the <code>corner</code> point. Two
	 * points are compared and the smaller one will be the
	 * <code>origin</code>.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void vertex_vertex_(Jun3dPoint aPoint1, Jun3dPoint aPoint2) {
		this.origin_corner_(aPoint1.min_(aPoint2), aPoint1.max_(aPoint2));
	}

	/**
	 * Answer the this volume value.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#volume()
	 * @category accessing
	 */
	public double volume() {
		return this.width() * this.height() * this.depth();
	}

	/**
	 * Add the aPoint to the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category adding
	 */
	public void add_(Jun3dPoint aPoint) {
		if (this.containsPoint_(aPoint) == false) {
			Jun3dBoundingBox box = this.merge_(Jun3dBoundingBox.Origin_corner_(aPoint, aPoint));
			this.origin_corner_(box.origin(), box.corner());
		}
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dLines()
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		Jun3dPoint originPoint = this.origin();
		Jun3dPoint cornerPoint = this.corner();
		return new Jun3dLine[] {
				new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()).to_(new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z())),
				new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()).to_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z())),
				new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()).to_(new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z())),
				new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()).to_(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z())),
				new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()).to_(new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z())),
				new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()).to_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z())),
				new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()).to_(new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z())),
				new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()).to_(new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z())),
				new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()).to_(new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z())),
				new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()).to_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z())),
				new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()).to_(new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z())),
				new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()).to_(new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z())) };
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dTriangles()
	 * @category converting 
	 */
	public Jun3dTriangle[] asArrayOf3dTriangles() {
		Jun3dPoint originPoint = this.origin();
		Jun3dPoint cornerPoint = this.corner();
		Jun3dTriangle[] triangles = new Jun3dTriangle[12];
		triangles[0] = Jun3dTriangle.On_on_on_(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()));
		triangles[1] = Jun3dTriangle.On_on_on_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()));
		triangles[2] = Jun3dTriangle.On_on_on_(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()));
		triangles[3] = Jun3dTriangle.On_on_on_(new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()));
		triangles[4] = Jun3dTriangle.On_on_on_(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()));
		triangles[5] = Jun3dTriangle.On_on_on_(new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()));
		triangles[6] = Jun3dTriangle.On_on_on_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()));
		triangles[7] = Jun3dTriangle.On_on_on_(new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()));
		triangles[8] = Jun3dTriangle.On_on_on_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()));
		triangles[9] = Jun3dTriangle.On_on_on_(new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()));
		triangles[10] = Jun3dTriangle.On_on_on_(new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()));
		triangles[11] = Jun3dTriangle.On_on_on_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()));
		return triangles;
	}

	/**
	 * Convert the receiver as an array of <code>JunPlane</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		Jun3dPoint originPoint = this.origin();
		Jun3dPoint cornerPoint = this.corner();
		JunPlane[] planes = new JunPlane[6];
		planes[0] = new JunPlane(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()));
		planes[1] = new JunPlane(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()));
		planes[2] = new JunPlane(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()));
		planes[3] = new JunPlane(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()));
		planes[4] = new JunPlane(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()));
		planes[5] = new JunPlane(new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()), new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()), new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()));
		return planes;
	}

	/**
	 * Convert to a <code>Jun3dBoundingBox</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category converting
	 */
	public Jun3dBoundingBox asBoundingBox() {
		return this;
	}

	/**
	 * Convert to a <code>Jun3dBoundingBoxes</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category converting
	 */
	public Jun3dBoundingBoxes asBoundingBoxes() {
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		Collection collection = new ArrayList();
		collection.add(this);
		boundingBoxes.boundingBoxes_(collection);
		return boundingBoxes;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject compoundObject = JunOpenGL3dObject.Origin_corner_(this.origin(), this.corner());
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_alpha_(Jun3dBoundingBox.this.defaultColor(), Jun3dBoundingBox.this.defaultAlpha());
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert the receiver as an array of points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category converting
	 */
	public Jun3dPoint[] asPointArray() {
		return new Jun3dPoint[] {
				this.origin(),
				new Jun3dPoint(this.corner().x(), this.origin().y(), this.origin().z()),
				new Jun3dPoint(this.corner().x(), this.corner().y(), this.origin().z()),
				new Jun3dPoint(this.origin().x(), this.corner().y(), this.origin().z()),
				this.corner(),
				new Jun3dPoint(this.corner().x(), this.origin().y(), this.corner().z()),
				new Jun3dPoint(this.origin().x(), this.origin().y(), this.corner().z()),
				new Jun3dPoint(this.origin().x(), this.corner().y(), this.corner().z()) };
	}

	/**
	 * Convert to a Rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#asRectangle()
	 * @category converting
	 */
	public Rectangle asRectangle() {
		Point origin = this.origin().as2dPoint()._toPoint();
		Point extent = this.extent().as2dPoint()._toPoint();
		return new Rectangle(origin.x, origin.y, extent.x, extent.y);
	}

	/**
	 * Convert to an array of Rectangle.
	 * 
	 * @return java.awt.Rectangle[]
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#asRectangles()
	 * @category converting
	 */
	public Rectangle[] asRectangles() {
		return this.asBoundingBoxes().asRectangles();
	}

	/**
	 * Answer the boxes out side with the specified box.
	 * 
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category functions
	 */
	public Jun3dBoundingBoxes boxesOutside_(Jun3dBoundingBox boundingBox) {
		Jun3dBoundingBoxes boxesOutside = new Jun3dBoundingBoxes();
		if (this.intersects_(boundingBox)) {
			double zOrigin = this.origin().z();
			double zCorner = this.corner().z();

			if (this.origin().z() <= boundingBox.origin().z() && boundingBox.corner().z() <= this.corner().z()) {
				zOrigin = boundingBox.origin().z();
				zCorner = boundingBox.corner().z();
				Jun3dBoundingBox aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin()), new Jun3dPoint(this.corner().x(), this.corner().y(), zOrigin));
				if (this.origin().z() != zOrigin) {
					boxesOutside.add_(aBox);
				}
				aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin().x(), this.origin().y(), zCorner), new Jun3dPoint(this.corner()));
				if (zCorner != this.corner().z()) {
					boxesOutside.add_(aBox);
				}
			} else {
				if (this.origin().z() < boundingBox.origin().z()) {
					zOrigin = boundingBox.origin().z();
					zCorner = this.corner().z();
					Jun3dBoundingBox aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin()), new Jun3dPoint(this.corner().x(), this.corner().y(), zOrigin));
					if (this.origin().z() != zOrigin) {
						boxesOutside.add_(aBox);
					}
				} else {
					if (boundingBox.corner().z() < this.corner().z()) {
						zOrigin = boundingBox.corner().z();
						zCorner = this.origin().z();
						Jun3dBoundingBox aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin().x(), this.origin().y(), zOrigin), new Jun3dPoint(this.corner()));
						if (zOrigin != this.corner().z()) {
							boxesOutside.add_(aBox);
						}
					}
				}
			}

			double yOrigin = Double.NaN;
			if (boundingBox.origin().y() > this.origin().y()) {
				yOrigin = boundingBox.origin().y();
				Jun3dBoundingBox aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin().x(), this.origin().y(), zOrigin), new Jun3dPoint(this.corner().x(), yOrigin, zCorner));
				boxesOutside.add_(aBox);
			} else {
				yOrigin = this.origin().y();
			}

			double yCorner = Double.NaN;
			if (boundingBox.corner().y() < this.corner().y()) {
				yCorner = boundingBox.corner().y();
				Jun3dBoundingBox aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin().x(), yCorner, zOrigin), new Jun3dPoint(this.corner().x(), this.corner().y(), zCorner));
				boxesOutside.add_(aBox);
			} else {
				yCorner = this.corner().y();
			}

			if (boundingBox.origin().x() > this.origin().x()) {
				Jun3dBoundingBox aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(this.origin().x(), yOrigin, zOrigin), new Jun3dPoint(boundingBox.origin().x(), yCorner, zCorner));
				boxesOutside.add_(aBox);
				if (boundingBox.corner().x() < this.corner().x()) {
					aBox = Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(corner().x(), yOrigin, zOrigin), new Jun3dPoint(this.corner().x(), yCorner, zCorner));
					boxesOutside.add_(aBox);
				}
			}
		} else {
			Jun3dBoundingBox aBox = (Jun3dBoundingBox) this.copy();
			boxesOutside.add_(aBox);
		}
		return boxesOutside;
	}

	/**
	 * Answer the receiver's contained point for the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint containedPointFor_(Jun3dPoint aPoint) {
		return aPoint.min_(this.origin()).max_(this.corner());
	}

	/**
	 * Answer a copy of this object expanded by "delta".
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox expandedBy_(double delta) {
		return Origin_corner_(this.origin().minus_(delta), this.corner().plus_(delta));
	}

	/**
	 * Answer a copy of this object expanded by "delta".
	 * 
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox expandedBy_(Jun3dBoundingBox aBoundingBox) {
		return Origin_corner_(this.origin().minus_(aBoundingBox.origin()), this.corner().plus_(aBoundingBox.corner()));
	}

	/**
	 * Answer a copy of this object inset by "delta".
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox insetBy_(double delta) {
		return Origin_corner_(this.origin().plus_(delta), this.corner().minus_(delta));
	}

	/**
	 * Answer a copy of this object inset by "delta".
	 * 
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox insetBy_(Jun3dBoundingBox aBoundingBox) {
		return Origin_corner_(this.origin().plus_(aBoundingBox.origin()), this.corner().minus_(aBoundingBox.corner()));
	}

	/**
	 * Answer the intersection part with the Jun3dBoundingBox.
	 * 
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox intersect_(Jun3dBoundingBox boundingBox) {
		return Origin_corner_(this.origin().max_(boundingBox.origin()), this.corner().min_(boundingBox.corner()));
	}

	/**
	 * Merge the Jun3dBoundingBox and the receiver.
	 * 
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox merge_(Jun3dBoundingBox boundingBox) {
		return Origin_corner_(this.origin().min_(boundingBox.origin()), this.corner().max_(boundingBox.corner()));
	}

	/**
	 * Answer the receiver's unify.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox unify() {
		return this;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category subdividing
	 */
	public Jun3dBoundingBoxes subdivide() {
		return this.subdivide8();
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category subdividing
	 */
	public Jun3dBoundingBoxes subdivide2() {
		Jun3dPoint originPoint = this.origin();
		Jun3dPoint cornerPoint = this.corner();
		Jun3dPoint centerPoint = this.center();
		Collection boundingBoxArray = new ArrayList(2);
		if (this.depth() > this.width() && this.depth() > this.height()) {
			boundingBoxArray.add(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()).box_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), centerPoint.z())));
			boundingBoxArray.add(new Jun3dPoint(originPoint.x(), originPoint.y(), centerPoint.z()).box_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z())));
		} else {
			if (this.height() > this.width()) {
				boundingBoxArray.add(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()).box_(new Jun3dPoint(cornerPoint.x(), centerPoint.y(), cornerPoint.z())));
				boundingBoxArray.add(new Jun3dPoint(originPoint.x(), centerPoint.y(), originPoint.z()).box_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z())));
			} else {
				boundingBoxArray.add(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()).box_(new Jun3dPoint(centerPoint.x(), cornerPoint.y(), cornerPoint.z())));
				boundingBoxArray.add(new Jun3dPoint(centerPoint.x(), originPoint.y(), originPoint.z()).box_(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z())));
			}
		}
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		boundingBoxes.boundingBoxes_(boundingBoxArray);
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category subdividing
	 */
	public Jun3dBoundingBoxes subdivide8() {
		Jun3dPoint originPoint = this.origin();
		Jun3dPoint cornerPoint = this.corner();
		Jun3dPoint centerPoint = this.center();
		Collection boundingBoxArray = new ArrayList(8);
		boundingBoxArray.add(new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z()).box_(centerPoint));
		boundingBoxArray.add(new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z()).box_(centerPoint));
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		boundingBoxes.boundingBoxes_(boundingBoxArray);
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes with specified level.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category subdividing
	 */
	public Jun3dBoundingBoxes subdivideLevel_(int anInteger) {
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		Collection anArray = new ArrayList();
		anArray.add(this);
		boundingBoxes.boundingBoxes_(anArray);
		for (int i = 0; i < anInteger; i++) {
			boundingBoxes = boundingBoxes.subdivide();
		}
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes with specified level.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category subdividing
	 */
	public Jun3dBoundingBoxes subdivide2Level_(int anInteger) {
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		Collection anArray = new ArrayList();
		anArray.add(this);
		boundingBoxes.boundingBoxes_(anArray);
		for (int i = 0; i < anInteger; i++) {
			boundingBoxes = boundingBoxes.subdivide2();
		}
		return boundingBoxes;
	}

	/**
	 * Answer true if the receiver contains the bounding box.
	 * 
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return boolean
	 * @category testing
	 */
	public boolean contains_(Jun3dBoundingBox boundingBox) {
		return this.origin().isLessThanOrEqualTo_(boundingBox.origin()) && boundingBox.corner().isLessThanOrEqualTo_(this.corner());
	}

	/**
	 * Answer true if the receiver contains or touches aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsOrTouchesPoint_(Jun3dPoint aPoint) {
		return this.origin().isLessThanOrEqualTo_(aPoint) && aPoint.isLessThanOrEqualTo_(this.corner());
	}

	/**
	 * Answer true if the receiver contains aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun3dPoint aPoint) {
		return this.origin().isLessThanOrEqualTo_(aPoint) && aPoint.isLessThan_(this.corner());
	}

	/**
	 * Answer true if the receiver intersects with the Jun3dBoundingBox.
	 * 
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return boolean
	 * @category testing
	 */
	public boolean intersects_(Jun3dBoundingBox aBoundingBox) {
		return this.origin().isLessThan_(aBoundingBox.corner()) && aBoundingBox.origin().isLessThan_(this.corner()) && this.origin().isLessThan_(this.corner()) && aBoundingBox.origin().isLessThan_(aBoundingBox.corner());
	}

	/**
	 * Answer true if the receiver intersects or touches with aBoundingBox.
	 * 
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return boolean
	 * @category testing
	 */
	public boolean intersectsOrTouches_(Jun3dBoundingBox aBoundingBox) {
		return this.intersects_(aBoundingBox) || this.touches_(aBoundingBox);
	}

	/**
	 * Answer true if the receiver touches with aBoundingBox.
	 * 
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @return boolean
	 * @category testing
	 */
	public boolean touches_(Jun3dBoundingBox aBoundingBox) {
		return this.origin().isLessThanOrEqualTo_(aBoundingBox.corner()) && aBoundingBox.origin().isLessThanOrEqualTo_(this.corner()) && this.origin().isLessThanOrEqualTo_(this.corner()) && aBoundingBox.origin().isLessThanOrEqualTo_(aBoundingBox.corner());
	}

	/**
	 * Answer true if the receiver touches the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean touchesPoint_(Jun3dPoint aPoint) {
		return ((this.isEqualNumber_to_(this.origin().x(), aPoint.x()) || this.isEqualNumber_to_(this.corner().x(), aPoint.x())) && (this.origin().y() <= aPoint.y() && aPoint.y() <= this.corner().y() && this.origin().z() <= aPoint.z() && aPoint.z() <= this
				.corner().z()))
				|| (((this.isEqualNumber_to_(this.origin().y(), aPoint.y()) || this.isEqualNumber_to_(this.corner().y(), aPoint.y())) && (this.origin().x() <= aPoint.x() && aPoint.x() <= this.corner().x() && this.origin().z() <= aPoint.z() && aPoint.z() <= this
						.corner().z())) || ((this.isEqualNumber_to_(this.origin().z(), aPoint.z()) || this.isEqualNumber_to_(this.corner().z(), aPoint.z())) && (this.origin().x() <= aPoint.x() && aPoint.x() <= this.corner().x()
						&& this.origin().y() <= aPoint.y() && aPoint.y() <= this.corner().y())));
	}

	/**
	 * Answer true if the receiver is a 3d geometry element.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Apply a transformation 'aJunTransformation' to the receiver.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category transforming
	 */
	public Jun3dBoundingBox transform_(Jun3dTransformation aTransformation) {
		return Jun3dBoundingBox.Origin_corner_(this.origin().transform_(aTransformation), this.corner().transform_(aTransformation));
	}

	/**
	 * Answer the new Jun3dBoundingBox which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category transforming
	 */
	public Jun3dBoundingBox translatedBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Answer the new Jun3dBoundingBox which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category transforming
	 */
	public Jun3dBoundingBox translatedBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the new Jun3dBoundingBox which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category transforming
	 */
	public Jun3dBoundingBox scaledBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Answer the new Jun3dBoundingBox which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category transforming
	 */
	public Jun3dBoundingBox scaledBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new Jun3dPoint which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category transforming
	 */
	public Jun3dBoundingBox rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun3dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer an rounded copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category truncation and round off
	 */
	public Jun3dBoundingBox rounded() {
		return Jun3dBoundingBox.Origin_corner_(this.origin().rounded(), this.corner().rounded());
	}

	/**
	 * Answer an truncated copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category truncation and round off
	 */
	public Jun3dBoundingBox truncated() {
		return Jun3dBoundingBox.Origin_corner_(this.origin().truncated(), this.corner().truncated());
	}
}
