package jp.co.sra.jun.geometry.boundaries;

import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;

/**
 * Jun2dBoundingBox class
 * 
 *  @author    nisinaka
 *  @created   1998/10/01 (by nisinaka)
 *  @updated   1999/08/05 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/18 (by Mitsuhiro Asada)
 *  @updated   2006/10/10 (by nisinaka)
 *  @updated   2007/05/29 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun693 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dBoundingBox.java,v 8.22 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun2dBoundingBox extends JunBoundingBox {
	/**
	 * Create a new instance of Jun2dBoundingBox.
	 * 
	 * @category Instance creation
	 */
	public Jun2dBoundingBox() {
		super();
	}

	/**
	 * Create a new instance of Jun2dBoundingBox and initialize it with the <code>aRectangle</code>.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @category Instance creation
	 */
	public Jun2dBoundingBox(Rectangle aRectangle) {
		this.origin_extent_(new Jun2dPoint(aRectangle.x, aRectangle.y), new Jun2dPoint(aRectangle.width, aRectangle.height));
	}

	/**
	 * Create a new instance of Jun2dBoundingBox and initialize it with the collection of points.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public Jun2dBoundingBox(Jun2dPoint[] pointCollection) {
		if (pointCollection == null || pointCollection.length == 0) {
			return;
		}

		this.origin_corner_(pointCollection[0], pointCollection[0]);
		for (int i = 1; i < pointCollection.length; i++) {
			this.add_(pointCollection[i]);
		}
	}

	/**
	 * Create a new instance of Jun2dBoundingBox and initialize it with the
	 * <code>origin</code> point and the <code>corner</code> point.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param cornerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public static Jun2dBoundingBox Origin_corner_(Jun2dPoint originPoint, Jun2dPoint cornerPoint) {
		Jun2dBoundingBox aBoundingBox = new Jun2dBoundingBox();
		aBoundingBox.origin_corner_(originPoint, cornerPoint);
		return aBoundingBox;
	}

	/**
	 * Create a new instance of Jun2dBoundingBox and initialize it with the
	 * <code>origin</code> point and the <code>extent</code> point.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public static Jun2dBoundingBox Origin_extent_(Jun2dPoint originPoint, Jun2dPoint extentPoint) {
		Jun2dBoundingBox aBoundingBox = new Jun2dBoundingBox();
		aBoundingBox.origin_extent_(originPoint, extentPoint);
		return aBoundingBox;
	}

	/**
	 * Create a new instance of Jun2dBoundingBox and initialize it with the
	 * <code>vertex</code> points.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param vertexPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param vertexPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public static Jun2dBoundingBox Vertex_vertex_(Jun2dPoint vertexPoint1, Jun2dPoint vertexPoint2) {
		Jun2dBoundingBox aBoundingBox = new Jun2dBoundingBox();
		aBoundingBox.vertex_vertex_(vertexPoint1, vertexPoint2);
		return aBoundingBox;
	}

	/**
	 * Answer a new Jun2dBoundingBox with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category Constants access
	 */
	public static Jun2dBoundingBox Unity() {
		return Jun2dBoundingBox.Origin_corner_(Jun2dPoint.Zero(), Jun2dPoint.Unity());
	}

	/**
	 * Answer a new Jun2dBoundingBox with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category Constants access
	 */
	public static Jun2dBoundingBox Zero() {
		return Jun2dBoundingBox.Origin_corner_(Jun2dPoint.Zero(), Jun2dPoint.Zero());
	}

	/**
	 * Initialize this object.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this._origin(Jun2dPoint.Zero());
		this._corner(Jun2dPoint.Zero());
	}

	/**
	 * Answer my area size.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return this.width() * this.height();
	}

	/**
	 * Answer the bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall boundingBall() {
		Jun2dPoint centerPoint = this.center();
		return new Jun2dBoundingBall(centerPoint, this.origin().distance_(centerPoint));
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return this;
	}

	/**
	 * Answer the center point of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint center() {
		return this.origin().center_(this.corner());
	}

	/**
	 * Answer the <code>corner</code> point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint corner() {
		return (Jun2dPoint) this._corner();
	}

	/**
	 * Set the <code>corner</code> point.
	 * 
	 * @param cornerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void corner_(Jun2dPoint cornerPoint) {
		this.origin_corner_(this.origin(), cornerPoint);
	}

	/**
	 * Answer the depth of the receiver.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#depth()
	 * @category accessing
	 */
	public double depth() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer the extent of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint extent() {
		return this.corner().minus_(this.origin());
	}

	/**
	 * Answer the inner bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall innerBoundingBall() {
		Jun2dPoint centerPoint = this.center();
		return new Jun2dBoundingBall(centerPoint, Math.min(this.width(), this.height()) / 2);
	}

	/**
	 * Answer the inner bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox innerBoundingBox() {
		return this;
	}

	/**
	 * Answer the <code>origin</code> point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint origin() {
		return (Jun2dPoint) this._origin();
	}

	/**
	 * Set the <code>origin</code> point.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void origin_(Jun2dPoint originPoint) {
		this.origin_corner_(originPoint, this.corner());
	}

	/**
	 * Set the <code>origin</code> point and the <code>corner</code> point.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param cornerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void origin_corner_(Jun2dPoint originPoint, Jun2dPoint cornerPoint) {
		this._origin(originPoint);
		this._corner(cornerPoint);
	}

	/**
	 * Set the <code>origin</code> point and the <code>corner</code> point. The
	 * <code>corner</code> point is calculated with the <code>extent</code>.
	 * 
	 * @param originPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void origin_extent_(Jun2dPoint originPoint, Jun2dPoint extentPoint) {
		this.origin_corner_(originPoint, originPoint.plus_(extentPoint));
	}

	/**
	 * Answer the receiver's outer bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall outerBoundingBall() {
		return this.boundingBall();
	}

	/**
	 * Answer the receiver's outer bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox outerBoundingBox() {
		return this.boundingBox();
	}

	/**
	 * Answer the receiver's p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p1() {
		return new Jun2dPoint(this.origin().x(), this.origin().y());
	}

	/**
	 * Answer the receiver's p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p2() {
		return new Jun2dPoint(this.corner().x(), this.origin().y());
	}

	/**
	 * Answer the receiver's p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p3() {
		return new Jun2dPoint(this.corner().x(), this.corner().y());
	}

	/**
	 * Answer the receiver's p4 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p4() {
		return new Jun2dPoint(this.origin().x(), this.corner().y());
	}

	/**
	 * Answer the receiver's point array.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] pp1() {
		return new Jun2dPoint[] { this.p1(), this.p2(), this.p3(), this.p4() };
	}

	/**
	 * Answer the receiver's point arrays.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[][] pps() {
		return new Jun2dPoint[][] { this.pp1() };
	}

	/**
	 * Answer the receiver's tetra bounding balls.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls
	 * @category accessing 
	 */
	public Jun2dBoundingBalls tetraBoundingBalls() {
		Jun2dBoundingBalls boundingBalls = new Jun2dBoundingBalls();
		Jun2dBoundingBoxes tetraBoundingBoxes = this.tetraBoundingBoxes();
		Jun2dBoundingBox[] boundingBoxes = tetraBoundingBoxes.boundingBoxes();
		for (int i = 0; i < boundingBoxes.length; i++) {
			boundingBalls.add_(boundingBoxes[i].outerBoundingBall());
		}
		return boundingBalls;
	}

	/**
	 * Answer the receiver's tetra bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category accessing 
	 */
	public Jun2dBoundingBoxes tetraBoundingBoxes() {
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		Jun2dBoundingBox boundingBox = null;
		double x = Double.NaN;
		double y = Double.NaN;
		boundingBoxes.add_(boundingBox = Jun2dBoundingBox.Origin_corner_(this.origin(), this.center()));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun2dPoint((x = this.width() / 2), 0)));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun2dPoint(x, (y = this.height() / 2))));
		boundingBoxes.add_(boundingBox.translatedBy_(new Jun2dPoint(0, y)));

		return boundingBoxes;
	}

	/**
	 * Set the <code>origin</code> point and the <code>corner</code> point. Two
	 * points are compared and the smaller one will be the
	 * <code>origin</code>.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void vertex_vertex_(Jun2dPoint aPoint1, Jun2dPoint aPoint2) {
		this.origin_corner_(aPoint1.min_(aPoint2), aPoint1.max_(aPoint2));
	}

	/**
	 * Add the aPoint to the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category adding
	 */
	public void add_(Jun2dPoint aPoint) {
		if (this.containsPoint_(aPoint) == false) {
			Jun2dBoundingBox box = this.merge_(Jun2dBoundingBox.Origin_corner_(aPoint, aPoint));
			this.origin_corner_(box.origin(), box.corner());
		}
	}

	/**
	 * Convert the receiver as an array of <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf2dLines()
	 * @category converting
	 */
	public Jun2dLine[] asArrayOf2dLines() {
		return new Jun2dLine[] {
				new Jun2dLine(new Jun2dPoint(this.origin().x(), this.origin().y()), new Jun2dPoint(this.corner().x(), this.origin().y())),
				new Jun2dLine(new Jun2dPoint(this.corner().x(), this.origin().y()), new Jun2dPoint(this.corner().x(), this.corner().y())),
				new Jun2dLine(new Jun2dPoint(this.corner().x(), this.corner().y()), new Jun2dPoint(this.origin().x(), this.corner().y())),
				new Jun2dLine(new Jun2dPoint(this.origin().x(), this.corner().y()), new Jun2dPoint(this.origin().x(), this.origin().y())) };
	}

	/**
	 * Convert the receiver as an array of <code>JunPlane</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		Jun2dPoint originPoint = this.origin();
		Jun2dPoint cornerPoint = this.corner();
		JunPlane[] planes = new JunPlane[1];
		planes[0] = new JunPlane(new Jun3dPoint(originPoint.x(), originPoint.y(), 0), new Jun3dPoint(cornerPoint.x(), originPoint.y(), 0), new Jun3dPoint(originPoint.x(), cornerPoint.y(), 0));
		return planes;
	}

	/**
	 * Convert to an array of a <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category converting
	 */
	public Jun2dTriangle[] asArrayOfTriangles() {
		Jun2dPoint originPoint = this.origin();
		Jun2dPoint cornerPoint = this.corner();
		Jun2dTriangle[] triangles = new Jun2dTriangle[2];
		triangles[0] = Jun2dTriangle.On_on_on_(new Jun2dPoint(originPoint.x(), originPoint.y()), new Jun2dPoint(cornerPoint.x(), originPoint.y()), new Jun2dPoint(originPoint.x(), cornerPoint.y()));
		triangles[1] = Jun2dTriangle.On_on_on_(new Jun2dPoint(cornerPoint.x(), cornerPoint.y()), new Jun2dPoint(originPoint.x(), cornerPoint.y()), new Jun2dPoint(cornerPoint.x(), originPoint.y()));
		return triangles;
	}

	/**
	 * Convert to a bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category converting
	 */
	public Jun2dBoundingBox asBoundingBox() {
		return this;
	}

	/**
	 * Convert to bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category converting
	 */
	public Jun2dBoundingBoxes asBoundingBoxes() {
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		Collection collection = new ArrayList();
		collection.add(this);
		boundingBoxes.boundingBoxes_(collection);
		return boundingBoxes;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunOpenGL3dObject aPolygon = new JunOpenGL3dPolygon(new Jun3dPoint[] {
				new Jun3dPoint(this.origin(), 0),
				new Jun3dPoint(this.corner().x(), this.origin().y(), 0),
				new Jun3dPoint(this.corner(), 0),
				new Jun3dPoint(this.origin().x(), this.corner().y(), 0) });
		aPolygon.paint_alpha_(this.defaultColor(), this.defaultAlpha());
		compoundObject.add_(aPolygon.reversed());
		compoundObject.add_(aPolygon);
		return compoundObject;
	}

	/**
	 * Convert the receiver as an array of points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category converting
	 */
	public Jun2dPoint[] asPointArray() {
		return new Jun2dPoint[] { this.origin(), new Jun2dPoint(this.corner().x(), this.origin().y()), this.corner(), new Jun2dPoint(this.origin().x(), this.corner().y()) };
	}

	/**
	 * Convert to a Rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#asRectangle()
	 * @category converting
	 */
	public Rectangle asRectangle() {
		Point origin = this.origin()._toPoint();
		Point extent = this.extent()._toPoint();
		return new Rectangle(origin.x, origin.y, extent.x, extent.y);
	}

	/**
	 * Convert to an array of Rectangle.
	 * 
	 * @return java.awt.Rectangle[]
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#asRectangles()
	 * @category converting
	 */
	public Rectangle[] asRectangles() {
		return this.asBoundingBoxes().asRectangles();
	}

	/**
	 * Answer the boxes out side with the specified box.
	 * 
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category functions
	 */
	public Jun2dBoundingBoxes boxesOutside_(Jun2dBoundingBox boundingBox) {
		Jun2dBoundingBoxes boxesOutside = new Jun2dBoundingBoxes();
		if (this.intersects_(boundingBox)) {
			double yOrigin = Double.NaN;
			if (boundingBox.origin().y() > this.origin().y()) {
				yOrigin = boundingBox.origin().y();
				Jun2dBoundingBox aBox = Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(this.origin()), new Jun2dPoint(this.corner().x(), yOrigin));
				boxesOutside.add_(aBox);
			} else {
				yOrigin = this.origin().y();
			}

			double yCorner = Double.NaN;
			if (boundingBox.corner().y() < this.corner().y()) {
				yCorner = boundingBox.corner().y();
				Jun2dBoundingBox aBox = Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(this.origin().x(), yCorner), new Jun2dPoint(this.corner()));
				boxesOutside.add_(aBox);
			} else {
				yCorner = this.corner().y();
			}

			if (boundingBox.origin().x() > this.origin().x()) {
				Jun2dBoundingBox aBox = Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(this.origin().x(), yOrigin), new Jun2dPoint(boundingBox.origin().x(), yCorner));
				boxesOutside.add_(aBox);
			}

			if (boundingBox.corner().x() < this.corner().x()) {
				Jun2dBoundingBox aBox = Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(boundingBox.corner().x(), yOrigin), new Jun2dPoint(this.corner().x(), yCorner));
				boxesOutside.add_(aBox);
			}
		} else {
			Jun2dBoundingBox aBox = (Jun2dBoundingBox) this.copy();
			boxesOutside.add_(aBox);
		}
		return boxesOutside;
	}

	/**
	 * Answer the receiver's contained point for the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint containedPointFor_(Jun2dPoint aPoint) {
		return aPoint.min_(this.origin()).max_(this.corner());
	}

	/**
	 * Answer a copy of this object expanded by "delta".
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param delta double
	 * @category functions
	 */
	public Jun2dBoundingBox expandedBy_(double delta) {
		return Origin_corner_(this.origin().minus_(delta), this.corner().plus_(delta));
	}

	/**
	 * Answer a copy of this object expanded by "delta".
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox expandedBy_(Jun2dBoundingBox aBoundingBox) {
		return Origin_corner_(this.origin().minus_(aBoundingBox.origin()), this.corner().plus_(aBoundingBox.corner()));
	}

	/**
	 * Answer a copy of this object expanded by "delta".
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dBoundingBox expandedBy_(Jun2dPoint aPoint) {
		return Origin_corner_(this.origin().minus_(aPoint), this.corner().plus_(aPoint));
	}

	/**
	 * Answer a copy of this object inset by "delta".
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param delta double
	 * @category functions
	 */
	public Jun2dBoundingBox insetBy_(double delta) {
		return Origin_corner_(this.origin().plus_(delta), this.corner().minus_(delta));
	}

	/**
	 * Answer a copy of this object inset by "delta".
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param aBoundingBox jp.co.sra.jun.geometry.basic.Ju2ndBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox insetBy_(Jun2dBoundingBox aBoundingBox) {
		return Origin_corner_(this.origin().plus_(aBoundingBox.origin()), this.corner().minus_(aBoundingBox.corner()));
	}

	/**
	 * Answer the intersection part with the Jun2dBoundingBox.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox intersect_(Jun2dBoundingBox aBoundingBox) {
		return Origin_corner_(this.origin().max_(aBoundingBox.origin()), this.corner().min_(aBoundingBox.corner()));
	}

	/**
	 * Answer the intersecting points with line.
	 * 
	 * @return jp.co.sra.jun.geometory.basic.Jun2dPoint[]
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dPoint[] intersectingPointsWithLine_(Jun2dLine aLine) {
		Jun2dLine theLine = new Jun2dLine(aLine.first(), aLine.last());
		Jun2dPoint p1 = this.origin();
		Jun2dPoint p2 = new Jun2dPoint(this.origin().x(), this.corner().y());
		Jun2dPoint p3 = this.corner();
		Jun2dPoint p4 = new Jun2dPoint(this.corner().x(), this.origin().y());

		Jun2dLine[] lineCollectionOfRectangle = new Jun2dLine[4];
		lineCollectionOfRectangle[0] = new Jun2dLine(p1, p2);
		lineCollectionOfRectangle[1] = new Jun2dLine(p2, p3);
		lineCollectionOfRectangle[2] = new Jun2dLine(p3, p4);
		lineCollectionOfRectangle[3] = new Jun2dLine(p4, p1);

		Vector intersectingPointCollection = new Vector();
		for (int i = 0; i < lineCollectionOfRectangle.length; i++) {
			Jun2dLine eachLine = lineCollectionOfRectangle[i];
			if (eachLine.isParallelWithLine_(theLine)) {
				double distance = eachLine.distanceFromPoint_(theLine.first());
				if (Math.abs(distance) < JunGeometry.Accuracy()) {
					intersectingPointCollection.add(eachLine.first());
					intersectingPointCollection.add(eachLine.last());
				}
			} else {
				Jun2dPoint intersectingPoint = eachLine.intersectingPointWithLine_(theLine);
				if (intersectingPoint != null) {
					intersectingPointCollection.add(intersectingPoint);
				}
			}
		}

		Vector aStream = new Vector();
		Jun2dPoint[] points = new Jun2dPoint[intersectingPointCollection.size()];
		intersectingPointCollection.copyInto(points);
		for (int i = 0; i < points.length; i++) {
			Jun2dPoint eachPoint = points[i];
			for (int j = 0; j < lineCollectionOfRectangle.length; j++) {
				Jun2dLine eachLine = lineCollectionOfRectangle[j];
				if (eachLine.lineSegmentContainsPoint_(eachPoint)) {
					aStream.add(eachPoint);
				}
			}
		}

		Jun2dPoint[] results = new Jun2dPoint[aStream.size()];
		aStream.copyInto(results);
		return results;
	}

	/**
	 * Merge the Jun2dBoundingBox and the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox merge_(Jun2dBoundingBox boundingBox) {
		return Origin_corner_(this.origin().min_(boundingBox.origin()), this.corner().max_(boundingBox.corner()));
	}

	/**
	 * Answer the receiver's unify.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox unify() {
		return this;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide() {
		return this.subdivide4();
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide2() {
		Jun2dPoint originPoint = this.origin();
		Jun2dPoint cornerPoint = this.corner();
		Jun2dPoint centerPoint = this.center();
		Collection boundingBoxArray = new ArrayList(2);
		if (this.height() > this.width()) {
			boundingBoxArray.add(new Jun2dPoint(originPoint.x(), originPoint.y()).box_(new Jun2dPoint(cornerPoint.x(), centerPoint.y())));
			boundingBoxArray.add(new Jun2dPoint(originPoint.x(), centerPoint.y()).box_(new Jun2dPoint(cornerPoint.x(), cornerPoint.y())));
		} else {
			boundingBoxArray.add(new Jun2dPoint(originPoint.x(), originPoint.y()).box_(new Jun2dPoint(centerPoint.x(), cornerPoint.y())));
			boundingBoxArray.add(new Jun2dPoint(centerPoint.x(), originPoint.y()).box_(new Jun2dPoint(cornerPoint.x(), cornerPoint.y())));
		}
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		boundingBoxes.boundingBoxes_(boundingBoxArray);
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide4() {
		Jun2dPoint originPoint = this.origin();
		Jun2dPoint cornerPoint = this.corner();
		Jun2dPoint centerPoint = this.center();
		Collection boundingBoxArray = new ArrayList(4);
		boundingBoxArray.add(new Jun2dPoint(originPoint.x(), originPoint.y()).box_(centerPoint));
		boundingBoxArray.add(new Jun2dPoint(cornerPoint.x(), originPoint.y()).box_(centerPoint));
		boundingBoxArray.add(new Jun2dPoint(originPoint.x(), cornerPoint.y()).box_(centerPoint));
		boundingBoxArray.add(new Jun2dPoint(cornerPoint.x(), cornerPoint.y()).box_(centerPoint));
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		boundingBoxes.boundingBoxes_(boundingBoxArray);
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes with specified level.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivideLevel_(int anInteger) {
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		Collection anArray = new ArrayList();
		anArray.add(this);
		boundingBoxes.boundingBoxes_(anArray);
		for (int i = 0; i < anInteger; i++) {
			boundingBoxes = boundingBoxes.subdivide();
		}
		return boundingBoxes;
	}

	/**
	 * Answer the receiver's subdivide bounding boxes with specified level.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category subdividing
	 */
	public Jun2dBoundingBoxes subdivide2Level_(int anInteger) {
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		Collection anArray = new ArrayList();
		anArray.add(this);
		boundingBoxes.boundingBoxes_(anArray);
		for (int i = 0; i < anInteger; i++) {
			boundingBoxes = boundingBoxes.subdivide2();
		}
		return boundingBoxes;
	}

	/**
	 * Answer true if the receiver contains the bounding box.
	 * 
	 * @return boolean
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category testing
	 */
	public boolean contains_(Jun2dBoundingBox boundingBox) {
		return this.origin().isLessThanOrEqualTo_(boundingBox.origin()) && boundingBox.corner().isLessThanOrEqualTo_(this.corner());
	}

	/**
	 * Answer true if the receiver contains or touches aPoint.
	 * 
	 * @return boolean
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category testing
	 */
	public boolean containsOrTouchesPoint_(Jun2dPoint aPoint) {
		return this.origin().isLessThanOrEqualTo_(aPoint) && aPoint.isLessThanOrEqualTo_(this.corner());
	}

	/**
	 * Answer true if the receiver contains aPoint.
	 * 
	 * @return boolean
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		return this.origin().isLessThanOrEqualTo_(aPoint) && aPoint.isLessThan_(this.corner());
	}

	/**
	 * Answer true if the receiver intersects with the Jun2dBoundingBox.
	 * 
	 * @return boolean
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category testing
	 */
	public boolean intersects_(Jun2dBoundingBox aBoundingBox) {
		return this.origin().isLessThan_(aBoundingBox.corner()) && aBoundingBox.origin().isLessThan_(this.corner()) && this.origin().isLessThan_(this.corner()) && aBoundingBox.origin().isLessThan_(aBoundingBox.corner());
	}

	/**
	 * Answer true if the receiver intersects or touches with aBoundingBox.
	 * 
	 * @return boolean
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category testing
	 */
	public boolean intersectsOrTouches_(Jun2dBoundingBox aBoundingBox) {
		return this.intersects_(aBoundingBox) || this.touches_(aBoundingBox);
	}

	/**
	 * Answer true if the receiver touches with aBoundingBox.
	 * 
	 * @return boolean
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category testing
	 */
	public boolean touches_(Jun2dBoundingBox aBoundingBox) {
		return this.origin().isLessThanOrEqualTo_(aBoundingBox.corner()) && aBoundingBox.origin().isLessThanOrEqualTo_(this.corner()) && this.origin().isLessThanOrEqualTo_(this.corner()) && aBoundingBox.origin().isLessThanOrEqualTo_(aBoundingBox.corner());
	}

	/**
	 * Answer true if the receiver touches the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean touchesPoint_(Jun2dPoint aPoint) {
		return ((this.isEqualNumber_to_(this.origin().x(), aPoint.x()) || this.isEqualNumber_to_(this.corner().x(), aPoint.x())) && (this.origin().y() <= aPoint.y() && aPoint.y() <= this.corner().y()))
				|| ((this.isEqualNumber_to_(this.origin().y(), aPoint.y()) || this.isEqualNumber_to_(this.corner().y(), aPoint.y())) && (this.origin().x() <= aPoint.x() && aPoint.x() <= this.corner().x()));
	}

	/**
	 * Answer true if the receiver is a 2d geometry element.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer the new Jun2dBoundingBox which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category transforming
	 */
	public Jun2dBoundingBox rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun2dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun2dBoundingBox which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category transforming
	 */
	public Jun2dBoundingBox scaledBy_(double aNumber) {
		return this.transform_(Jun2dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new Jun2dBoundingBox which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category transforming
	 */
	public Jun2dBoundingBox scaledBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply a transformation 'aJunTransformation' to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dBoundingBox transform_(Jun2dTransformation aTransformation) {
		return Jun2dBoundingBox.Origin_corner_(this.origin().transform_(aTransformation), this.corner().transform_(aTransformation));
	}

	/**
	 * Answer the new Jun2dBoundingBox which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category transforming
	 */
	public Jun2dBoundingBox translatedBy_(double aNumber) {
		return this.transform_(Jun2dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the new Jun2dBoundingBox which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category transforming
	 */
	public Jun2dBoundingBox translatedBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Translate_(aPoint));
	}

	/**
	 * Answer an rounded copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category truncation and round off
	 */
	public Jun2dBoundingBox rounded() {
		return Jun2dBoundingBox.Origin_corner_(this.origin().rounded(), this.corner().rounded());
	}

	/**
	 * Answer an truncated copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category truncation and round off
	 */
	public Jun2dBoundingBox truncated() {
		return Jun2dBoundingBox.Origin_corner_(this.origin().truncated(), this.corner().truncated());
	}
}
