package jp.co.sra.jun.delaunay.twoD;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunPoint;

/**
 * Jun2dDelaunayVertex class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2002/01/14 (by Ryouichi Matsuda)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dDelaunayVertex.java,v 8.11 2008/02/20 06:30:54 nisinaka Exp $
 */
public class Jun2dDelaunayVertex extends Jun2dDelaunayElement {
	protected double x = Double.NaN;
	protected double y = Double.NaN;
	protected double z = Double.NaN;
	protected Jun2dDelaunayHalfEdge halfEdge;

	/**
	 * Create a new instance of <code>Jun2dDelaunayVertex</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public Jun2dDelaunayVertex() {
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayVertex</code> and initialize it.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @category Instance creation
	 */
	public Jun2dDelaunayVertex(double xNumber, double yNumber) {
		this.setX_(xNumber);
		this.setY_(yNumber);
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayVertex</code> and initialize it.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @param zNumber double
	 * @category Instance creation
	 */
	public Jun2dDelaunayVertex(double xNumber, double yNumber, double zNumber) {
		this.setX_(xNumber);
		this.setY_(yNumber);
		this.setZ_(zNumber);
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayVertex</code> and initialize it.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public Jun2dDelaunayVertex(JunPoint aPoint) {
		if (aPoint instanceof Jun2dPoint) {
			Jun2dPoint p = (Jun2dPoint) aPoint;
			this.setX_(p.x());
			this.setY_(p.y());
		}

		if (aPoint instanceof Jun3dPoint) {
			Jun3dPoint p = (Jun3dPoint) aPoint;
			this.setX_(p.x());
			this.setY_(p.y());
			this.setZ_(p.z());
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayVertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double angleWith_with_(Jun2dDelaunayVertex aJun2dDelaunayVertex1, Jun2dDelaunayVertex aJun2dDelaunayVertex2) {
		return Math.acos(this.cosOfAngleWith_with_(aJun2dDelaunayVertex1, aJun2dDelaunayVertex2));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayVertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double areaWith_with_(Jun2dDelaunayVertex aJun2dDelaunayVertex1, Jun2dDelaunayVertex aJun2dDelaunayVertex2) {
		return ((aJun2dDelaunayVertex1.x() - x) * (aJun2dDelaunayVertex2.y() - y)) - ((aJun2dDelaunayVertex2.x() - x) * (aJun2dDelaunayVertex1.y() - y));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 */
	public Jun2dPoint asJun2dPoint() {
		return new Jun2dPoint(x, y);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint asJun3dPoint() {
		return new Jun3dPoint(x, y, (z == Double.NaN) ? 0.0d : z);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return boolean
	 */
	public boolean contains_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return (Math.pow(aJun2dDelaunayVertex.x() - x, 2) + Math.pow(aJun2dDelaunayVertex.y() - y, 2) < DECIMAL_12);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return boolean
	 */
	public boolean containsX_y_(double xNumber, double yNumber) {
		return (Math.pow(xNumber - x, 2) + Math.pow(yNumber - y, 2) < DECIMAL_12);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayVertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double cosOfAngleWith_with_(Jun2dDelaunayVertex aJun2dDelaunayVertex1, Jun2dDelaunayVertex aJun2dDelaunayVertex2) {
		double d1;
		double d2;
		double dotProduct;

		d1 = this.squaredDistance_(aJun2dDelaunayVertex1);
		d2 = this.squaredDistance_(aJun2dDelaunayVertex2);
		dotProduct = ((aJun2dDelaunayVertex1.x() - x) * (aJun2dDelaunayVertex2.x() - x)) + ((aJun2dDelaunayVertex1.y() - y) * (aJun2dDelaunayVertex2.y() - y));

		return Math.min(Math.max(dotProduct / Math.sqrt(d1 * d2), -1.0d), 1.0d);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double distance_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return Math.sqrt(this.squaredDistance_(aJun2dDelaunayVertex));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double dotProduct_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return (x * aJun2dDelaunayVertex.x()) + (y * aJun2dDelaunayVertex.y());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object edgesDo_(final StBlockClosure aBlock) {
		return this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				Object returnObject = aBlock.value_(hEdge.edge);

				if (returnObject != null) {
					return returnObject; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 */
	public Jun2dDelaunayEdge edgeTo_(final Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return (Jun2dDelaunayEdge) this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				if (aJun2dDelaunayVertex == hEdge.pair().vertex()) {
					return hEdge.edge; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public Jun2dDelaunayHalfEdge halfEdge() {
		return halfEdge;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayHalfEdge
	 *        jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public void halfEdge_(Jun2dDelaunayHalfEdge aJun2dDelaunayHalfEdge) {
		halfEdge = aJun2dDelaunayHalfEdge;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object halfEdgesDo_(StBlockClosure aBlock) {
		Jun2dDelaunayHalfEdge hEdge;

		hEdge = halfEdge;

		do {
			Jun2dDelaunayHalfEdge nextHalfEdge;

			nextHalfEdge = hEdge.next().pair();

			Object returnObject = aBlock.value_(hEdge);

			if (returnObject != null) {
				return returnObject; // continuation exit
			}

			hEdge = nextHalfEdge;
		} while (hEdge != halfEdge);

		return null;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean isBoundaryVertex() {
		return false;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object loopsDo_(final StBlockClosure aBlock) {
		return this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				Jun2dDelaunayLoop l;

				l = hEdge.loop();

				if (l != null) {
					Object returnObject = aBlock.value_(l);

					if (returnObject != null) {
						return returnObject; // continuation exit
					}
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object neighborsDo_(final StBlockClosure aBlock) {
		return this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				Object returnObject = aBlock.value_(hEdge.pair().vertex());

				if (returnObject != null) {
					return returnObject; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 */
	public JunPoint point() {
		if (Double.isNaN(z)) {
			return new Jun2dPoint(x, y);
		} else {
			return new Jun3dPoint(x, y, z);
		}
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException DOCUMENT ME!
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Vertex(");
		aWriter.write(Double.toString(x));
		aWriter.write("@");
		aWriter.write(Double.toString(y));
		aWriter.write(")");
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double product_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return (x * aJun2dDelaunayVertex.y()) - (y * aJun2dDelaunayVertex.x());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return double
	 */
	public double squaredDistance_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return this.squaredDistanceFromX_y_(aJun2dDelaunayVertex.x(), aJun2dDelaunayVertex.y());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return double
	 */
	public double squaredDistanceFromX_y_(double xNumber, double yNumber) {
		return Math.pow(x - xNumber, 2) + Math.pow(y - yNumber, 2);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 */
	public double x() {
		return x;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 */
	public double y() {
		return y;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 */
	public double z() {
		return z;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayLoop jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	protected Jun2dDelaunayHalfEdge detectHalfEdgeOfLoop_(final Jun2dDelaunayLoop aJun2dDelaunayLoop) {
		return (Jun2dDelaunayHalfEdge) this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				if (hEdge.loop() == aJun2dDelaunayLoop) {
					return hEdge; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	protected Jun2dDelaunayHalfEdge detectHalfEdgeOfLoopContainsX_y_(final double xNumber, final double yNumber) {
		return (Jun2dDelaunayHalfEdge) this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				if (hEdge.loop().containsX_y_(xNumber, yNumber)) {
					return hEdge; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	protected Jun2dDelaunayHalfEdge halfEdgeTo_(final Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return (Jun2dDelaunayHalfEdge) this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				if (aJun2dDelaunayVertex == hEdge.pair().vertex()) {
					return hEdge.pair(); // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	protected Jun2dDelaunayHalfEdge halfEdgeToward_(final Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		final double scaleToTarget = this.distance_(aJun2dDelaunayVertex);
		final double vectorToTargetX = (aJun2dDelaunayVertex.x() - x) / scaleToTarget;
		final double vectorToTargetY = (aJun2dDelaunayVertex.y() - y) / scaleToTarget;

		return (Jun2dDelaunayHalfEdge) this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				Jun2dDelaunayVertex vertex;
				double scaleToNeighbor;
				double vectorToNeighborX;
				double vectorToNeighborY;

				vertex = hEdge.pair().vertex();
				scaleToNeighbor = distance_(vertex);
				vectorToNeighborX = (vertex.x() - x) / scaleToNeighbor;
				vectorToNeighborY = (vertex.y() - y) / scaleToNeighbor;

				if (hEdge.pair().vertex() == aJun2dDelaunayVertex) {
					return hEdge; // continuation exit
				}

				if (Math.abs((vectorToTargetX * vectorToNeighborY) - (vectorToNeighborX * vectorToTargetY)) < DECIMAL_12) {
					double t = (vectorToTargetX * vectorToNeighborX) + (vectorToTargetY * vectorToNeighborY);

					if ((-DECIMAL_12 <= t) && (t <= (1.0d + DECIMAL_12))) {
						return hEdge; // continuation exit
					}
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param vertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param vertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return boolean
	 */
	protected boolean isOnLineSegmentFrom_to_(Jun2dDelaunayVertex vertex1, Jun2dDelaunayVertex vertex2) {
		double yourScale;
		double yourVectorX;
		double yourVectorY;
		double myScale;
		double myVectorX;
		double myVectorY;

		if ((this == vertex1) || (this == vertex2)) {
			return true;
		}

		yourScale = vertex1.distance_(vertex2);
		yourVectorX = (vertex2.x() - vertex1.x()) / yourScale;
		yourVectorY = (vertex2.y() - vertex1.y()) / yourScale;
		myScale = vertex1.distance_(this);
		myVectorX = (this.x() - vertex1.x()) / myScale;
		myVectorY = (this.y() - vertex1.y()) / myScale;

		if (Math.abs((yourVectorX * myVectorY) - (myVectorX * yourVectorY)) < DECIMAL_12) {
			double t = (yourVectorX * myVectorX) + (yourVectorY * myVectorY);

			if ((0.0d <= t) && (t <= 1.0d)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 */
	protected Jun2dDelaunayEdge nearestEdgeFromX_y_(double xNumber, double yNumber) {
		return (this.nearestHalfEdgeFromX_y_(xNumber, yNumber).edge());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	protected Jun2dDelaunayHalfEdge nearestHalfEdgeFromX_y_(final double xNumber, final double yNumber) {
		final StValueHolder theNearest = new StValueHolder(); // Jun2dDelaunayHalfEdge
		final StValueHolder minSquaredDistance = new StValueHolder(); // double

		theNearest.value_(halfEdge);
		minSquaredDistance.value_(halfEdge.edge().squaredDistanceFromX_y_(xNumber, yNumber));
		this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				if (hEdge != halfEdge) {
					double squaredDistance;

					squaredDistance = hEdge.edge().squaredDistanceFromX_y_(xNumber, yNumber);

					if (squaredDistance < minSquaredDistance._doubleValue()) {
						minSquaredDistance.value_(squaredDistance);
						theNearest.value_(hEdge);
					}
				}

				return null;
			}
		});

		return (Jun2dDelaunayHalfEdge) theNearest.value();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	protected Jun2dDelaunayVertex nearestNeighborFromX_y_(final double xNumber, final double yNumber) {
		final StValueHolder theNearest = new StValueHolder(); // Jun2dDelaunayVertex
		final StValueHolder minSquaredDistance = new StValueHolder(); // double

		theNearest.value_(this);
		minSquaredDistance.value_(Math.pow(x - xNumber, 2) + Math.pow(y - yNumber, 2));
		this.neighborsDo_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;
				double squaredDistance;

				squaredDistance = Math.pow(vertex.x() - xNumber, 2) + Math.pow(vertex.y() - yNumber, 2);

				if (squaredDistance < minSquaredDistance._doubleValue()) {
					minSquaredDistance.value_(squaredDistance);
					theNearest.value_(vertex);
				}

				return null;
			}
		});

		return (Jun2dDelaunayVertex) theNearest.value();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 */
	protected void setX_(double xNumber) {
		x = xNumber;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param yNumber double
	 */
	protected void setY_(double yNumber) {
		y = yNumber;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param zNumber double
	 */
	protected void setZ_(double zNumber) {
		z = zNumber;
	}
}
