package jp.co.sra.jun.collections.support;

import jp.co.sra.smalltalk.*;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunCorrelation class
 * 
 *  @author    nisinaka
 *  @created   2001/04/10 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCorrelation.java,v 8.11 2008/02/20 06:30:53 nisinaka Exp $
 */
public class JunCorrelation extends JunAbstractObject {
	//
	public static final double ACCURACY = 1.0e-5d;
	protected Number[] collectionX = null;
	protected Number[] collectionY = null;
	protected boolean nMinusOne = true;
	protected double meanX = Double.NaN;
	protected double meanY = Double.NaN;
	protected double varianceX = Double.NaN;
	protected double varianceY = Double.NaN;
	protected double standardDeviationX = Double.NaN;
	protected double standardDeviationY = Double.NaN;
	protected double covariance = Double.NaN;
	protected double coefficient = Double.NaN;
	//
	protected double[] _collectionX = null;
	protected double[] _collectionY = null;

	/**
	 * Create a new instance of <code>JunCorrelation</code> and initialize it.
	 * 
	 * @param xCollection double[]
	 * @param yCollection double[]
	 * @category Instance creation
	 */
	public JunCorrelation(double[] xCollection, double[] yCollection) {
		super();
		this.collectionX_(xCollection);
		this.collectionY_(yCollection);
	}

	/**
	 * Create a new instance of <code>JunCorrelation</code> and initialize it.
	 * 
	 * @param xCollection float[]
	 * @param yCollection float[]
	 * @category Instance creation
	 */
	public JunCorrelation(float[] xCollection, float[] yCollection) {
		super();
		this.collectionX_(xCollection);
		this.collectionY_(yCollection);
	}

	/**
	 * Create a new instance of <code>JunCorrelation</code> and initialize it.
	 * 
	 * @param xCollection int[]
	 * @param yCollection int[]
	 * @category Instance creation
	 */
	public JunCorrelation(int[] xCollection, int[] yCollection) {
		super();
		this.collectionX_(xCollection);
		this.collectionY_(yCollection);
	}

	/**
	 * Create a new instance of <code>JunCorrelation</code> and initialize it.
	 * 
	 * @param xCollection long[]
	 * @param yCollection long[]
	 * @category Instance creation
	 */
	public JunCorrelation(long[] xCollection, long[] yCollection) {
		super();
		this.collectionX_(xCollection);
		this.collectionY_(yCollection);
	}

	/**
	 * Answer the coefficient value.
	 * 
	 * @return double
	 */
	public double coefficient() {
		if (Double.isNaN(coefficient)) {
			double denominator = this.standardDeviationX() * this.standardDeviationY();

			if (denominator <= ACCURACY) {
				denominator = ACCURACY;
			}

			coefficient = this.covariance() / denominator;
		}

		return coefficient;
	}

	/**
	 * Answer the collection X as an array of double.
	 * 
	 * @return DOCUMENT ME!
	 */
	public double[] collectionX() {
		if (_collectionX == null) {
			_collectionX = new double[collectionX.length];

			for (int i = 0; i < collectionX.length; i++) {
				_collectionX[i] = collectionX[i].doubleValue();
			}
		}

		return _collectionX;
	}

	/**
	 * Set the collection X as an array of double.
	 * 
	 * @param aCollection double[]
	 */
	public void collectionX_(double[] aCollection) {
		Double[] doubleArray = new Double[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			doubleArray[i] = new Double(aCollection[i]);
		}

		this.collectionX_(doubleArray);
		_collectionX = aCollection;
	}

	/**
	 * Set the collection X as an array of float.
	 * 
	 * @param aCollection float[]
	 */
	public void collectionX_(float[] aCollection) {
		Float[] floatArray = new Float[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			floatArray[i] = new Float(aCollection[i]);
		}

		this.collectionX_(floatArray);
	}

	/**
	 * Set the collection X as an array of int.
	 * 
	 * @param aCollection int[]
	 */
	public void collectionX_(int[] aCollection) {
		Integer[] intArray = new Integer[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			intArray[i] = new Integer(aCollection[i]);
		}

		this.collectionX_(intArray);
	}

	/**
	 * Set the collection X as an array of long.
	 * 
	 * @param aCollection long[]
	 */
	public void collectionX_(long[] aCollection) {
		Long[] longArray = new Long[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			longArray[i] = new Long(aCollection[i]);
		}

		this.collectionX_(longArray);
	}

	/**
	 * Set the collection X.
	 * 
	 * @param aCollection java.lang.Number[]
	 */
	public void collectionX_(Number[] aCollection) {
		collectionX = aCollection;
		_collectionX = null;
		meanX = Double.NaN;
		varianceX = Double.NaN;
		covariance = Double.NaN;
		coefficient = Double.NaN;
	}

	/**
	 * Answer the collection Y as an array of double.
	 * 
	 * @return DOCUMENT ME!
	 */
	public double[] collectionY() {
		if (_collectionY == null) {
			_collectionY = new double[collectionY.length];

			for (int i = 0; i < collectionY.length; i++) {
				_collectionY[i] = collectionY[i].doubleValue();
			}
		}

		return _collectionY;
	}

	/**
	 * Set the collection Y as an array of double.
	 * 
	 * @param aCollection double[]
	 */
	public void collectionY_(double[] aCollection) {
		Double[] doubleArray = new Double[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			doubleArray[i] = new Double(aCollection[i]);
		}

		this.collectionY_(doubleArray);
		_collectionX = aCollection;
	}

	/**
	 * Set the collection Y as an array of float.
	 * 
	 * @param aCollection float[]
	 */
	public void collectionY_(float[] aCollection) {
		Float[] floatArray = new Float[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			floatArray[i] = new Float(aCollection[i]);
		}

		this.collectionY_(floatArray);
	}

	/**
	 * Set the collection Y as an array of int.
	 * 
	 * @param aCollection int[]
	 */
	public void collectionY_(int[] aCollection) {
		Integer[] intArray = new Integer[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			intArray[i] = new Integer(aCollection[i]);
		}

		this.collectionY_(intArray);
	}

	/**
	 * Set the collection Y as an array of long.
	 * 
	 * @param aCollection long[]
	 */
	public void collectionY_(long[] aCollection) {
		Long[] longArray = new Long[aCollection.length];

		for (int i = 0; i < aCollection.length; i++) {
			longArray[i] = new Long(aCollection[i]);
		}

		this.collectionY_(longArray);
	}

	/**
	 * Set the collection Y.
	 * 
	 * @param aCollection java.lang.Number[]
	 */
	public void collectionY_(Number[] aCollection) {
		collectionY = aCollection;
		_collectionY = null;
		meanY = Double.NaN;
		varianceY = Double.NaN;
		covariance = Double.NaN;
		coefficient = Double.NaN;
	}

	/**
	 * Answer the covariance value.
	 * 
	 * @return double
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 */
	public double covariance() {
		if (Double.isNaN(covariance)) {
			double total = 0;
			double[] cx = this.collectionX();
			double[] cy = this.collectionY();

			if (cx.length != cy.length) {
				throw new SmalltalkException("no match error");
			}

			int size = cx.length;

			for (int i = 0; i < size; i++) {
				double valueX = cx[i] - this.meanX();
				double valueY = cy[i] - this.meanY();
				total += (valueX * valueY);
			}

			covariance = total / (this.nMinusOne() ? (size - 1) : size);
		}

		return covariance;
	}

	/**
	 * Answer the mean value of X.
	 * 
	 * @return double
	 */
	public double meanX() {
		if (Double.isNaN(meanX)) {
			double total = 0;
			double[] collection = this.collectionX();

			for (int i = 0; i < collection.length; i++) {
				total += collection[i];
			}

			meanX = total / collection.length;
		}

		return meanX;
	}

	/**
	 * Answer the mean value of Y.
	 * 
	 * @return double
	 */
	public double meanY() {
		if (Double.isNaN(meanY)) {
			double total = 0;
			double[] collection = this.collectionY();

			for (int i = 0; i < collection.length; i++) {
				total += collection[i];
			}

			meanY = total / collection.length;
		}

		return meanY;
	}

	/**
	 * Answer whether to use 'n' or 'n-1'.
	 * 
	 * @return boolean
	 */
	public boolean nMinusOne() {
		return nMinusOne;
	}

	/**
	 * Set the flag whether to use 'n' or 'n-1'.
	 * 
	 * @param aBoolean boolean
	 */
	public void nMinusOne_(boolean aBoolean) {
		nMinusOne = aBoolean;
	}

	/**
	 * Answer the standard deviation value of X.
	 * 
	 * @return double
	 */
	public double standardDeviationX() {
		if (Double.isNaN(standardDeviationX)) {
			standardDeviationX = Math.sqrt(this.varianceX());
		}

		return standardDeviationX;
	}

	/**
	 * Answer the standard deviation value of Y.
	 * 
	 * @return double
	 */
	public double standardDeviationY() {
		if (Double.isNaN(standardDeviationY)) {
			standardDeviationY = Math.sqrt(this.varianceY());
		}

		return standardDeviationY;
	}

	/**
	 * Answer the variance value of X.
	 * 
	 * @return double
	 */
	public double varianceX() {
		if (Double.isNaN(varianceX)) {
			double total = 0;
			double[] collection = this.collectionX();
			int size = collection.length;

			for (int i = 0; i < size; i++) {
				double value = collection[i] - this.meanX();
				total += (value * value);
			}

			varianceX = total / (this.nMinusOne() ? (size - 1) : size);
		}

		return varianceX;
	}

	/**
	 * Answer the variance value of Y.
	 * 
	 * @return double
	 */
	public double varianceY() {
		if (Double.isNaN(varianceY)) {
			double total = 0;
			double[] collection = this.collectionY();
			int size = collection.length;

			for (int i = 0; i < collection.length; i++) {
				double value = collection[i] - this.meanY();
				total += (value * value);
			}

			varianceY = total / (this.nMinusOne() ? (size - 1) : size);
		}

		return varianceY;
	}
}
