package jp.co.sra.gl4jun;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;

import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.RepaintManager;
import javax.swing.SwingUtilities;

import jp.co.sra.smalltalk.SmalltalkException;

/**
 * GLjJCanvas class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1999/09/01 (by MATSUDA Ryouichi)
 *  @updated   2006/07/18 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: GLjJCanvas.java,v 8.11 2008/02/20 06:30:52 nisinaka Exp $
 */
public class GLjJCanvas extends JComponent {

	protected int renderingMode;

	protected transient GLjRenderingContext renderingContext;
	protected transient boolean resizeRequest;
	protected transient boolean moveRequest;

	/**
	 * Create a new instance of GLjJCanvas and initialize it.
	 *
	 * @category Instance creation
	 */
	public GLjJCanvas() {
		this.initialize();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		this.setPreferredSize(new Dimension(256, 256));
		this.setRenderingMode(GLjRenderingMode.IMAGE);
	}

	/**
	 * Delete the rendering context before removed.
	 * 
	 * @see java.awt.Component#removeNotify()
	 * @category initialize-release
	 */
	public void removeNotify() {
		if (renderingContext != null) {
			Container c = this.getParent();
			while (c != null && (c instanceof java.applet.Applet) == false) {
				c = c.getParent();
			}

			renderingContext.gljDelete(c != null);
			renderingContext = null;
		}

		super.removeNotify();
	}

	/**
	 * Set the rendering mode.
	 * 
	 * @param aMode int
	 * @category accessing
	 */
	public void setRenderingMode(int aMode) {
		switch (aMode) {
			case GLjRenderingMode.NOTHING:
			case GLjRenderingMode.IMAGE:
				renderingMode = aMode;
				RepaintManager.currentManager(this).setDoubleBufferingEnabled(true);
				break;
			case GLjRenderingMode.DIRECT:
			case GLjRenderingMode.SINGLE_BUFFER:
			case GLjRenderingMode.DOUBLE_BUFFER:
				renderingMode = aMode;
				RepaintManager.currentManager(this).setDoubleBufferingEnabled(false);
				break;
			default:
				throw new SmalltalkException("Invalid rendering mode : " + aMode);
		}
	}

	/**
	 * Set the request flags when moved or resized.
	 * Setting flags by a component listener may be done after the repaint.
	 * 
	 * @param x int
	 * @param y int
	 * @param width int
	 * @param height int
	 * @see java.awt.Component#setBounds(int, int, int, int)
	 */
	public void setBounds(int x, int y, int width, int height) {
		Rectangle bounds = this.getBounds();
		if (bounds.x != x || bounds.y != y) {
			moveRequest = true;
		}
		if (bounds.width != width || bounds.height != height) {
			resizeRequest = true;
		}

		super.setBounds(x, y, width, height);
	}

	/**
	 * Answer my default rendering context.
	 * 
	 * @return jp.co.sra.gl4jun.GLjRenderingContext
	 * @category defaults
	 */
	protected GLjRenderingContext defaultRenderingContext() {
		Component root = SwingUtilities.getRoot(this);
		return new GLjRenderingContext(this, (JFrame) root);
	}

	/**
	 * Paint the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category painting
	 */
	protected void paintComponent(Graphics aGraphics) {
		super.paintComponent(aGraphics);
		this.render(aGraphics);
		this.superimposeOn(aGraphics);
	}

	/**
	 * Render the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category rendering
	 */
	protected void render(Graphics aGraphics) {
		if (this.isShowing() == false) {
			return;
		}

		synchronized (GLjInterface.Current()) {
			if (renderingContext == null) {
				renderingContext = this.defaultRenderingContext();
				renderingContext.setMode(renderingMode);
				resizeRequest = false;
				moveRequest = true;
				renderingContext.gljInitialize();
			}
		}

		if (renderingContext.getMode() != GLjRenderingMode.NOTHING) {
			synchronized (renderingContext.OpenGLInterface) {
				if (resizeRequest) {
					renderingContext.gljResize(this.getSize());
					resizeRequest = false;
				}

				if (moveRequest) {
					Point rootPoint = this.getRootPane().getLocationOnScreen();
					Point thisPoint = this.getLocationOnScreen();
					Point location = new Point(thisPoint.x - rootPoint.x, thisPoint.y - rootPoint.y);
					renderingContext.gljSetLocation(location);
					moveRequest = false;
				}

				renderingContext.gljCurrent();
				this.render(renderingContext);

				if (renderingContext.getMode() == GLjRenderingMode.IMAGE) {
					Image anImage = renderingContext.gljImageRGB();
					aGraphics.drawImage(anImage, 0, 0, null);
				}

				renderingContext.gljFlush();
			}
		}
	}

	/**
	 * Render the receiver on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.gl4jun.GLjRenderingContext
	 * @category rendering
	 */
	protected void render(GLjRenderingContext aRenderingContext) {
		this.render(aRenderingContext.OpenGLInterface);
	}

	/**
	 * Render the receiver on the OpenGL interface.
	 * 
	 * @param openGLInterface jp.co.sra.gl4jun.GLjInterface
	 * @category rendering
	 */
	protected void render(GLjInterface openGLInterface) {
	}

	/**
	 * Superimpose on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category superimpose
	 */
	protected void superimposeOn(Graphics aGraphics) {
	}

}
