#import <Cocoa/Cocoa.h>
#import <OpenGL/gl.h>

#include <jawt.h>
#include <jawt_md.h>
#include "jp_co_sra_gl4jun_GLjInterface.h"
#include "rendering_mode.h"
#include "version.h"

/**
 * glj-mac.m:
 *
 * 	@author:	Hoshi Takanori
 * 	@version:	2.0
 * 	@created:	1999/09/06 (by Hoshi Takanori)
 * 	@updated:	2005/03/03 (by Hoshi Takanori)
 *
 * 	$Id: glj-mac.m,v 2.3 2005/03/08 08:47:56 hoshi Exp $
 */

struct JUN_HANDLE {
	int mode;
	NSView *view;
	NSOpenGLContext *context;
	void *buffer;
	int width;
	int height;
	int x;
	int y;
	Boolean resize;
};

static NSOpenGLPixelFormatAttribute singleBuffer[] = {
	NSOpenGLPFAWindow,
	NSOpenGLPFADepthSize, 16,
	NSOpenGLPFAStencilSize, 8,
	0
};

static NSOpenGLPixelFormatAttribute doubleBuffer[] = {
	NSOpenGLPFAWindow,
	NSOpenGLPFADoubleBuffer,
	NSOpenGLPFADepthSize, 16,
	NSOpenGLPFAStencilSize, 8,
	0
};

static NSOpenGLPixelFormatAttribute offscreenBuffer[] = {
	NSOpenGLPFAOffScreen,
	NSOpenGLPFAColorSize, 32,
	NSOpenGLPFADepthSize, 16,
	NSOpenGLPFAStencilSize, 8,
	0
};

// ==================================================================
/**
 *
 */
JNIEXPORT jstring JNICALL Java_jp_co_sra_gl4jun_GLjInterface_getVersion
  (JNIEnv *env, jclass cls)
{
	return (*env)->NewStringUTF(env, JUNGL_JNI_VERSION);
}

// ==================================================================
/**
 *
 */
static jboolean rendering2image
  (jint *image, int width, int height, GLenum format)
{
	GLint swapbytes, lsbfirst, rowlength;
	GLint skiprows, skippixels, alignment;
	unsigned char *buffer, *pBuffer;
	unsigned char byte1, byte2, byte3, byte4;
	jint color;
	int x, y;

	// create buffer
	switch (format) {
	case GL_RGB:
		buffer = malloc(width * height * 3);
		break;
	case GL_RGBA:
		buffer = malloc(width * height * 4);
		break;
	default:
		buffer = NULL;
		break;
	}
	if (buffer == NULL) {
		return JNI_FALSE;
	}

	// copy buffer
	glGetIntegerv(GL_UNPACK_SWAP_BYTES, &swapbytes);
	glGetIntegerv(GL_UNPACK_LSB_FIRST, &lsbfirst);
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &rowlength);
	glGetIntegerv(GL_UNPACK_SKIP_ROWS, &skiprows);
	glGetIntegerv(GL_UNPACK_SKIP_PIXELS, &skippixels);
	glGetIntegerv(GL_UNPACK_ALIGNMENT, &alignment);
	glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
	glPixelStorei(GL_UNPACK_LSB_FIRST, GL_FALSE);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glReadPixels(0, 0, width, height, format, GL_UNSIGNED_BYTE, buffer);
	glPixelStorei(GL_UNPACK_SWAP_BYTES, swapbytes);
	glPixelStorei(GL_UNPACK_LSB_FIRST, lsbfirst);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, rowlength);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, skiprows);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, skippixels);
	glPixelStorei(GL_UNPACK_ALIGNMENT, alignment);

	// convert buffer
	switch (format) {
	case GL_RGB:
		for (y = height - 1; 0 <= y ; y--) {
			pBuffer = buffer + (width * 3 * y);
			for (x = 0; x < width; x++) {
				byte1 = *pBuffer++;
				byte2 = *pBuffer++;
				byte3 = *pBuffer++;
				color = byte3 | byte2 << 8 | byte1 << 16 | 0xFF << 24;
				*image++ = color;
			}
		}
		break;
	case GL_RGBA:
		for (y = height - 1; 0 <= y ; y--) {
			pBuffer = buffer + (width * 4 * y);
			for (x = 0; x < width; x++) {
				byte1 = *pBuffer++;
				byte2 = *pBuffer++;
				byte3 = *pBuffer++;
				byte4 = *pBuffer++;
				color = byte3 | byte2 << 8 | byte1 << 16 | byte4 << 24;
				*image++ = color;
			}
		}
		break;
	default:
		break;
	}

	// release buffer
	free(buffer);

	return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljReadImage
  (JNIEnv *env, jobject obj, jint jHandle, jintArray jImage, jboolean isRGBA)
{
	struct JUN_HANDLE *handle;
	jint *image;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;
	image = (*env)->GetIntArrayElements(env, jImage, 0);

	// copy buffer
	if (isRGBA == JNI_TRUE) {
		rendering2image(image, handle->width, handle->height, GL_RGBA);
	} else {
		// hanged up when set RGB.
		rendering2image(image, handle->width, handle->height, GL_RGBA);
		//rendering2image(image, handle->width, handle->height, GL_RGB);
	}

	// release handle
	(*env)->ReleaseIntArrayElements(env, jImage, image, 0);

	return JNI_TRUE;
}

// ==================================================================
/**
 *
 */
static jboolean get_view
  (JNIEnv *env, jobject component, NSView **view)
{
	JAWT awt;
	JAWT_DrawingSurface *ds;
	JAWT_DrawingSurfaceInfo *dsi;
	JAWT_MacOSXDrawingSurfaceInfo *dsi_mac;
	jint lock;

	awt.version = JAWT_VERSION_1_3;
	if (JAWT_GetAWT(env, &awt) == JNI_FALSE) {
		return JNI_FALSE;
	}

	ds = awt.GetDrawingSurface(env, component);
	if (ds == NULL) {
		return JNI_FALSE;
	}

	lock = ds->Lock(ds);
	if ((lock & JAWT_LOCK_ERROR) != 0) {
		awt.FreeDrawingSurface(ds);
		return JNI_FALSE;
	}

	dsi = ds->GetDrawingSurfaceInfo(ds);
	if (dsi == NULL) {
		ds->Unlock(ds);
		awt.FreeDrawingSurface(ds);
		return JNI_FALSE;
	}

	dsi_mac = (JAWT_MacOSXDrawingSurfaceInfo *) dsi->platformInfo;
	*view = dsi_mac->cocoaViewRef;

	ds->Unlock(ds);
	awt.FreeDrawingSurface(ds);

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jint JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljCreateContext__Ljava_lang_Object_2III
  (JNIEnv *env, jobject obj, jobject component, jint width, jint height, jint mode)
{
	NSView *view = nil;
	NSOpenGLPixelFormat *format;
	NSOpenGLContext *context;
	void *buffer = NULL;
	struct JUN_HANDLE *handle;

	// create resource
	switch (mode) {
	case RENDER_SINGLE_BUFFER:
	case RENDER_DOUBLE_BUFFER:
		if (get_view(env, component, &view) == JNI_FALSE) {
			return 0;
		}
		format = [[NSOpenGLPixelFormat alloc] initWithAttributes:
				(mode == RENDER_SINGLE_BUFFER) ? singleBuffer : doubleBuffer];
		context = [[NSOpenGLContext alloc] initWithFormat: format shareContext: nil];
		[format release];
		break;
	case RENDER_DIRECT:
		if (get_view(env, component, &view) == JNI_FALSE) {
			return 0;
		}
		// no break
	case RENDER_IMAGE:
		format = [[NSOpenGLPixelFormat alloc] initWithAttributes: offscreenBuffer];
		context = [[NSOpenGLContext alloc] initWithFormat: format shareContext: nil];
		[format release];
		buffer = malloc(width * height * 4);
		if (buffer == NULL) {
			[context release];
			return 0;
		}
		[context setOffScreen: buffer width: width height: height rowbytes: width * 4];
		break;
	default:
		return 0;
	}

	// set handle
	handle = malloc(sizeof(struct JUN_HANDLE));
	if (handle == NULL) {
		return 0;
	}
	handle->mode = mode;
	handle->view = view;
	handle->context = context;
	handle->buffer = buffer;
	handle->width = width;
	handle->height = height;
	handle->x = 0;
	handle->y = 0;
	handle->resize = 0;

	return (jint)handle;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljDeleteContext
  (JNIEnv *env, jobject obj, jint jHandle, jboolean isApplet)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// delete resource
	switch (handle->mode) {
	case RENDER_SINGLE_BUFFER:
	case RENDER_DOUBLE_BUFFER:
		[handle->context clearDrawable];
		[handle->context release];
		break;
	case RENDER_DIRECT:
	case RENDER_IMAGE:
		[handle->context clearDrawable];
		[handle->context release];
		free(handle->buffer);
		break;
	}

	// release handle
	free(handle);

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljFlushBuffer
  (JNIEnv *env, jobject obj, jint jHandle)
{
	struct JUN_HANDLE *handle;
	CGColorSpaceRef cs;
	CGDataProviderRef data;
	CGImageRef image;
	CGContextRef ctx;
	CGRect rect;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// flush buffer
	switch (handle->mode) {
	case RENDER_SINGLE_BUFFER:
	case RENDER_DOUBLE_BUFFER:
	case RENDER_IMAGE:
		[handle->context flushBuffer];
		[NSOpenGLContext clearCurrentContext];
		break;
	case RENDER_DIRECT:
		glFinish();
		[handle->context flushBuffer];
		[NSOpenGLContext clearCurrentContext];
		if (! [handle->view inLiveResize]
		 && [handle->view lockFocusIfCanDraw]) {
			cs = CGColorSpaceCreateDeviceRGB();
			data = CGDataProviderCreateWithData(NULL,
					handle->buffer, handle->width * handle->height * 4, NULL);
			image = CGImageCreate(handle->width, handle->height,
					8, 32, handle->width * 4, cs, kCGImageAlphaNoneSkipFirst,
					data, NULL, FALSE, kCGRenderingIntentDefault);
			CGDataProviderRelease(data);
			CGColorSpaceRelease(cs);
			ctx = [[NSGraphicsContext currentContext] graphicsPort];
			CGContextScaleCTM(ctx, 1.0, -1.0);
			CGContextTranslateCTM(ctx, 0, - handle->height);
			rect = CGRectMake(0, 0, handle->width, handle->height);
			CGContextDrawImage(ctx, rect, image);
			CGImageRelease(image);
			[handle->view unlockFocus];
		}
		break;
	}

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljMakeCurrent
  (JNIEnv *env, jobject obj, jint jHandle)
{
	struct JUN_HANDLE *handle;
	int width, height;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;
	width = handle->width;
	height = handle->height;

	if (handle->resize == 0) {
		// make current
		if (handle->mode == RENDER_SINGLE_BUFFER
		 || handle->mode == RENDER_DOUBLE_BUFFER) {
			[handle->context setView: handle->view];
		}
		[handle->context makeCurrentContext];
		[handle->context update];
		glViewport(0, 0, width, height);
	} else {
		switch (handle->mode) {
		case RENDER_SINGLE_BUFFER:
		case RENDER_DOUBLE_BUFFER:
			[handle->context setView: handle->view];
			[handle->context makeCurrentContext];
			[handle->context update];
			glViewport(0, 0, width, height);
			break;
		case RENDER_IMAGE:
		case RENDER_DIRECT:
			[handle->context clearDrawable];
			free(handle->buffer);
			handle->buffer = malloc(width * height * 4);
			[handle->context setOffScreen: handle->buffer width: width height: height rowbytes: width * 4];
			[handle->context makeCurrentContext];
			[handle->context update];
			glViewport(0, 0, width, height);
			break;
		}
		handle->resize = 0;
	}

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljSetLocation
  (JNIEnv *env, jobject obj, jint jHandle, jint x, jint y)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// set location
	handle->x = x;
	handle->y = y;

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljSetSize
  (JNIEnv *env, jobject obj, jint jHandle, jint width, jint height)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// set handle
	if (handle->width != width || handle->height != height) {
		handle->width = width;
		handle->height = height;
		handle->resize = 1;
	}

	return JNI_TRUE;
}
