/*
 * This file is part of jDiffChaser.
 *
 *  jDiffChaser is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  jDiffChaser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with jDiffChaser; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdiffchaser.publish;

import java.awt.image.BufferedImage;
import java.util.Arrays;
import java.util.logging.Logger;
import javax.swing.ImageIcon;
import org.jdiffchaser.scenarihandling.TestSet;
import org.jdiffchaser.utils.ImageUtilities;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;


public class HtmlReporter {
    
    private static final Logger LOGGER = Logger.getLogger(HtmlReporter.class.getName());
    
    private static final SimpleDateFormat SDF = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss");
    
    private static final ImageIcon JDIFFCHASER_LOGO = 
                         new ImageIcon(HtmlReporter.class.getResource("jdiffchaserlogo_xs.png"));    
    private static BufferedImage JDIFFCHASER_BUFF_LOGO;
    static{
        JDIFFCHASER_BUFF_LOGO = new BufferedImage(JDIFFCHASER_LOGO.getIconWidth(),
                                                  JDIFFCHASER_LOGO.getIconHeight(),
                                                  BufferedImage.TYPE_INT_ARGB);
        JDIFFCHASER_BUFF_LOGO.getGraphics().drawImage(JDIFFCHASER_LOGO.getImage(), 0, 0, null);
    }
    
    private String failedDir;
    private int totalTestCount;
    private int totalTestPlayed;
    private int totalTestSucceded;
    private String firstHostComment;
    private String secondHostComment;
    
    public HtmlReporter(String failedDir, 
                        String firstHostComment,
                        String secondHostComment,
                        int totalTestCount, int totalTestPlayed, int totalTestSucceded) {
        
        this.firstHostComment  = firstHostComment;
        this.secondHostComment = secondHostComment;
        this.failedDir         = failedDir;
        this.totalTestCount    = totalTestCount;
        this.totalTestPlayed   = totalTestPlayed;
        this.totalTestSucceded = totalTestSucceded;
    }
    
    public void createReport(String title) throws ReportException, IOException, InterruptedException{
        StringBuffer sbuf = new StringBuffer();
        sbuf.append(      "<html>"
                        + "<body link=\"white\" vlink=\"white\" alink=\"white\">"
                        + "<table width=\"580\" border=\"0\"><tr><td width=\"200\"><a href=\"http://jdiffchaser.sourceforge.net\"><img src=\"jdiffchaserlogo_xs.png\" border=\"0\"></a></td><td align=\"center\">"
                        + "<center><h1>" + title + "</h1></center>"
                        + "<center><h1>GUI Tests</h1></center>"
                        + "<center><h2>" + SDF.format(new Date()) + "</h2></center>"
                        + "</td></tr></table>"
                        + "<hr>"
                        + "Client #1 is " + this.firstHostComment + "<br>"
                        + "Client #2 is " + this.secondHostComment
                        + "<br>"
                        + "<table>");
        
        File failedDirFile = new File(this.failedDir);
        LOGGER.finer("failedDir is " + this.failedDir);
        File[] appsDirs = failedDirFile.listFiles();
        LOGGER.finer("app dirs are " + Arrays.asList(appsDirs));
        if(appsDirs.length==0){
            sbuf.append( "<tr><td colspan=\"3\">&nbsp;</td></tr>"
                       + "<tr><td colspan=\"3\" bgcolor=\"#b5ff84\">"
                       + "<h2>Congratulations, no differences detected!</h2></tr>");
        }else{        
            int failures = this.totalTestPlayed - this.totalTestSucceded;
            sbuf.append( "<tr><td colspan=\"3\">&nbsp;</td></tr>"
                       + "<tr><td colspan=\"3\" bgcolor=\"yellow\">"
                       + "<h2><center>- Expected to play " + this.totalTestCount + " tests -</center></h2></td></tr>"
                       + "<tr><td colspan=\"3\" bgcolor=\"yellow\">"
                       + "<h2><center>" + failures + " different screen" + (failures>1?"s":"") +  " detected among "
                       + this.totalTestPlayed + " test" + (this.totalTestPlayed>1?"s":"") 
                       + " played</center></h2></td></tr>");
            for(int i=0; i<appsDirs.length; i++){
                File appDir = appsDirs[i];
                if(!appDir.isDirectory()){ //if index.html already exists, bypass
                    continue;
                }
                sbuf.append(  "<tr><td colspan=\"3\">&nbsp;</td></tr>"
                            + "<tr><td colspan=\"3\" bgcolor=\"pink\" ><h2>");
                sbuf.append(appDir.getName());
                sbuf.append("</h2></td></tr>");
                File[] scenariiDirs = appDir.listFiles();
                LOGGER.finer("app dir is " + appDir);
                LOGGER.finer("Scenariis dirs are " + Arrays.asList(scenariiDirs));
                for(int j=0; j<scenariiDirs.length; j++){
                                       
                    File scenarioDir = scenariiDirs[j];
                    
                    File scenarioThumbnailsDir = new File(scenarioDir.getPath() + File.separator + "thumbnails");
                    scenarioThumbnailsDir.mkdir();                            
                    
                    sbuf.append(  "<tr><td colspan=\"3\">&nbsp;</td></tr>"
                                + "<tr><td width=\"340\" bgcolor=\"#bcbcff\"><b>");
                    sbuf.append(scenarioDir.getName());
                    sbuf.append("</b></td>");

                    File[] images = scenarioDir.listFiles();
                    LOGGER.finer("scenario dir is " + scenarioDir.getPath());
                    LOGGER.finer("Images are " + Arrays.asList(images));
                    File diffImg = findFile(images, TestSet.DIFF_IMG_NAME + '.' + TestSet.IMG_EXTENSION);
                    
                    List imagesWithoutDiff = getListWithout(images, diffImg.getName());
                    
                    String image1Base  = ((File)imagesWithoutDiff.get(0)).getName();
                    String image2Base  = ((File)imagesWithoutDiff.get(1)).getName();
                    String diffImgBase = diffImg.getName();
                            
                    String image1Th      = image1Base.substring(0, image1Base.lastIndexOf('.')) 
                                                                                        + "Th." + TestSet.IMG_EXTENSION;
                    String image2Th      = image2Base.substring(0, image2Base.lastIndexOf('.')) 
                                                                                        + "Th." + TestSet.IMG_EXTENSION;
                    String diffImgNameTh = diffImgBase.substring(0, diffImgBase.lastIndexOf('.')) 
                                                                                        + "Th." + TestSet.IMG_EXTENSION;
                    
                    storeScaledImageFile(ImageUtilities.readImage((File)imagesWithoutDiff.get(0)),
                                         image1Th, scenarioThumbnailsDir, 80, 48);
                    
                    storeScaledImageFile(ImageUtilities.readImage((File)imagesWithoutDiff.get(1)),
                                         image2Th, scenarioThumbnailsDir, 80, 48);

                    storeScaledImageFile(ImageUtilities.readImage(diffImg),
                                         diffImgNameTh, scenarioThumbnailsDir, 160, 104);
                    
                    String bigImagesDir  = appDir.getName() + '/' + scenarioDir.getName() + '/';
                    String thumbnailsDir = bigImagesDir + scenarioThumbnailsDir.getName() + '/';
                    
                    createMagnifyHtmlFile(failedDir+'/'+bigImagesDir, scenarioDir.getName(), null, image1Base, image2Base);
                    createMagnifyHtmlFile(failedDir+'/'+bigImagesDir, scenarioDir.getName(), image1Base, image2Base, diffImgBase);
                    createMagnifyHtmlFile(failedDir+'/'+bigImagesDir, scenarioDir.getName(), image2Base, diffImgBase, null);
                                                                                        
                    sbuf.append("<td><a href=\"").append(bigImagesDir).append(toHtmlFilename(image1Base));
                    sbuf.append("\"><img src=\"").append(thumbnailsDir).append(image1Th);
                    sbuf.append("\" width=\"80\" height=\"48\" alt=\"image1\"></a></td>");
                    sbuf.append("<td rowspan=\"2\"><a href=\"").append(bigImagesDir).append(toHtmlFilename(diffImgBase));
                    sbuf.append("\"><img src=\"").append(thumbnailsDir).append(diffImgNameTh);
                    sbuf.append("\" width=\"160\" height=\"104\" alt=\"image of diffs\"></a></td></tr>");
                    sbuf.append("<tr><td >&nbsp;</td><td><a href=\"").append(bigImagesDir).append(toHtmlFilename(image2Base));
                    sbuf.append("\"><img src=\"").append(thumbnailsDir).append(image2Th);
                    sbuf.append("\" width=\"80\" height=\"48\" alt=\"image2\"></a></td></tr>");

                }
            }
        }
        
        sbuf.append("</table></body></html>");

        try{
            FileWriter fw = new FileWriter(failedDir+'/'+"index.html");
            fw.write(sbuf.toString());
            fw.close();
        }catch(IOException ioe){
            throw new ReportException("Unable to create HTML report file", ioe);
        }
        
        String imgPath = failedDir + File.separator + "jdiffchaserlogo_xs.png";
        ImageUtilities.storeImage(JDIFFCHASER_BUFF_LOGO, imgPath); 
    }
        
    private void createMagnifyHtmlFile(String toDir,
                                       String scenarioName,
                                       String previousImgFilename, 
                                       String currentImgFilename,
                                       String nextImgFilename) throws IOException{
        FileWriter fw = new FileWriter(toDir+toHtmlFilename(currentImgFilename));
        fw.write(createMagnifyHtmlContent(scenarioName, previousImgFilename, currentImgFilename, nextImgFilename));
        fw.close();
    }
    
    public static String createMagnifyHtmlContent(String scenarioName,
                                                  String previousImgFilename, 
                                                  String currentImgFilename,
                                                  String nextImgFilename){
    String toRet =  
        "<html>"
        + "\n<title>"
        + "\n" + scenarioName + ": " + currentImgFilename
        + "\n</title>"
        + "\n<body>"
        + "\n<center><a href=\"../../index.html\">back to report</a></center>"
        + "\n<center>" + scenarioName + ": " + currentImgFilename + "</center>"             
        + "\n<table border=\"0\" width=\"100%\">"
        + "\n<tr>"
        + "\n<td><center>"
        + (previousImgFilename!=null? "<a href=\"" + toHtmlFilename(previousImgFilename) + "\">Previous</a>" : "")
        + "&nbsp;&nbsp;"
        + (nextImgFilename!=null?"<a href=\"" + toHtmlFilename(nextImgFilename) + "\">Next</a>" : "")
        + "</center></td>"
        + "\n</tr>"
        + "\n<tr>"
        + "\n<td><center><img src=\"" + currentImgFilename + "\" border=\"0\"></center></td>"
        + "\n</tr>"
        + "\n<tr>"
        + "\n<td><center>"
        + (previousImgFilename!=null? "<a href=\"" + toHtmlFilename(previousImgFilename) + "\">Previous</a>": "")
        + "&nbsp;&nbsp;"
        + (nextImgFilename!=null?"<a href=\"" + toHtmlFilename(nextImgFilename) + "\">Next</a>" : "")
        + "</center></td>"
        + "\n</tr>"
        + "\n</table>"
        + "\n</body>"
        + "\n</html>";
    
        return toRet;
    }
    
    private static String toHtmlFilename(String imageName){
        return imageName.substring(0, imageName.indexOf('.')) + ".html";
    }
    
    private void storeScaledImageFile(RenderedImage srcImg, String filename, 
                                      File toDir, int width, int height) throws IOException{
        String imgPath = toDir.getPath() + File.separator + filename;
        ImageUtilities.storeImage(ImageUtilities.getScaledInstance(srcImg, width, height, true), imgPath); 
    }
    
    private static List getListWithout(File[] files, String name){
        List list = new ArrayList();
        for(int i=0; i<files.length; i++){
            if(!name.equals(files[i].getName())){
                list.add(files[i]);
            }
        }
        return list;
    }
    
    private static File findFile(File[] files, String name){
        File fileToReturn = null;
        for(int i=0; i<files.length; i++){
            if(name.equals(files[i].getName())){
                fileToReturn = files[i];
                break;
            }
        }
        return fileToReturn;
    }
    
}
