/*
 * $Id:TwisterRegistry.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister;

import net.sf.jame.core.extension.ConfigurableExtension;
import net.sf.jame.core.extension.ConfigurableExtensionRegistry;
import net.sf.jame.core.extension.Extension;
import net.sf.jame.core.extension.ExtensionNotFoundException;
import net.sf.jame.core.extension.ExtensionRegistry;
import net.sf.jame.twister.effect.extension.EffectExtensionConfig;
import net.sf.jame.twister.effect.extension.EffectExtensionRegistry;
import net.sf.jame.twister.effect.extension.EffectExtensionRuntime;
import net.sf.jame.twister.frame.filter.extension.FrameFilterExtensionConfig;
import net.sf.jame.twister.frame.filter.extension.FrameFilterExtensionRegistry;
import net.sf.jame.twister.frame.filter.extension.FrameFilterExtensionRuntime;
import net.sf.jame.twister.frame.layer.filter.extension.LayerFilterExtensionConfig;
import net.sf.jame.twister.frame.layer.filter.extension.LayerFilterExtensionRegistry;
import net.sf.jame.twister.frame.layer.filter.extension.LayerFilterExtensionRuntime;
import net.sf.jame.twister.frame.layer.image.extension.ImageExtensionConfig;
import net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRegistry;
import net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime;
import net.sf.jame.twister.tree.extension.NodeBuilderExtensionRegistry;
import net.sf.jame.twister.tree.extension.NodeBuilderExtensionRuntime;
import net.sf.jame.twister.xmlexport.extension.ActionXMLExporterExtensionRegistry;
import net.sf.jame.twister.xmlexport.extension.ActionXMLExporterExtensionRuntime;
import net.sf.jame.twister.xmlexport.extension.ExtensionConfigXMLExporterExtensionRegistry;
import net.sf.jame.twister.xmlexport.extension.ExtensionConfigXMLExporterExtensionRuntime;
import net.sf.jame.twister.xmlimport.extension.ActionXMLImporterExtensionRegistry;
import net.sf.jame.twister.xmlimport.extension.ActionXMLImporterExtensionRuntime;
import net.sf.jame.twister.xmlimport.extension.ExtensionConfigXMLImporterExtensionRegistry;
import net.sf.jame.twister.xmlimport.extension.ExtensionConfigXMLImporterExtensionRuntime;

/**
 * The twister registry.
 * 
 * @author Andrea Medeghini
 */
public class TwisterRegistry {
	private ConfigurableExtensionRegistry<FrameFilterExtensionRuntime<?>, FrameFilterExtensionConfig> frameFilterRegistry;
	private ConfigurableExtensionRegistry<LayerFilterExtensionRuntime<?>, LayerFilterExtensionConfig> imageFilterRegistry;
	private ConfigurableExtensionRegistry<ImageExtensionRuntime<?>, ImageExtensionConfig> imageRegistry;
	private ConfigurableExtensionRegistry<EffectExtensionRuntime<?>, EffectExtensionConfig> effectRegistry;
	private ExtensionRegistry<NodeBuilderExtensionRuntime> nodeBuilderRegistry;
	private ExtensionRegistry<ExtensionConfigXMLExporterExtensionRuntime> XMLExtensionConfigExporterRegistry;
	private ExtensionRegistry<ExtensionConfigXMLImporterExtensionRuntime> XMLExtensionConfigImporterRegistry;
	private ExtensionRegistry<ActionXMLExporterExtensionRuntime> XMLActionExporterRegistry;
	private ExtensionRegistry<ActionXMLImporterExtensionRuntime> XMLActionImporterRegistry;

	private static class RegistryHolder {
		private static final TwisterRegistry instance = new TwisterRegistry();
	}

	private TwisterRegistry() {
		setFrameFilterRegistry(new FrameFilterExtensionRegistry());
		setLayerFilterRegistry(new LayerFilterExtensionRegistry());
		setImageRegistry(new ImageExtensionRegistry());
		setEffectRegistry(new EffectExtensionRegistry());
		setNodeBuilderRegistry(new NodeBuilderExtensionRegistry());
		setXMLExtensionConfigExporterRegistry(new ExtensionConfigXMLExporterExtensionRegistry());
		setXMLExtensionConfigImporterRegistry(new ExtensionConfigXMLImporterExtensionRegistry());
		setXMLActionExporterRegistry(new ActionXMLExporterExtensionRegistry());
		setXMLActionImporterRegistry(new ActionXMLImporterExtensionRegistry());
	}

	/**
	 * @return
	 */
	public static TwisterRegistry getInstance() {
		return RegistryHolder.instance;
	}

	/**
	 * Returns a filter extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public ConfigurableExtension<FrameFilterExtensionRuntime<?>, FrameFilterExtensionConfig> getFrameFilterExtension(final String extensionId) throws ExtensionNotFoundException {
		return frameFilterRegistry.getConfigurableExtension(extensionId);
	}

	/**
	 * Returns an image extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public ConfigurableExtension<ImageExtensionRuntime<?>, ImageExtensionConfig> getImageExtension(final String extensionId) throws ExtensionNotFoundException {
		return imageRegistry.getConfigurableExtension(extensionId);
	}

	/**
	 * Returns an image filter extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public ConfigurableExtension<LayerFilterExtensionRuntime<?>, LayerFilterExtensionConfig> getLayerFilterExtension(final String extensionId) throws ExtensionNotFoundException {
		return imageFilterRegistry.getConfigurableExtension(extensionId);
	}

	/**
	 * Returns an effect extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public ConfigurableExtension<EffectExtensionRuntime<?>, EffectExtensionConfig> getEffectExtension(final String extensionId) throws ExtensionNotFoundException {
		return effectRegistry.getConfigurableExtension(extensionId);
	}

	/**
	 * Returns a node builder extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public Extension<NodeBuilderExtensionRuntime> getNodeBuilderExtension(final String extensionId) throws ExtensionNotFoundException {
		return nodeBuilderRegistry.getExtension(extensionId);
	}

	/**
	 * Returns a XML exporter extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public Extension<ExtensionConfigXMLExporterExtensionRuntime> getXMLExtensionConfigExporterExtension(final String extensionId) throws ExtensionNotFoundException {
		return XMLExtensionConfigExporterRegistry.getExtension(extensionId);
	}

	/**
	 * Returns a XML importer extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public Extension<ExtensionConfigXMLImporterExtensionRuntime> getXMLExtensionConfigImporterExtension(final String extensionId) throws ExtensionNotFoundException {
		return XMLExtensionConfigImporterRegistry.getExtension(extensionId);
	}

	/**
	 * Returns a XML exporter extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public Extension<ActionXMLExporterExtensionRuntime> getXMLActionExporterExtension(final String extensionId) throws ExtensionNotFoundException {
		return XMLActionExporterRegistry.getExtension(extensionId);
	}

	/**
	 * Returns a XML importer extension.
	 * 
	 * @param extensionId the extensionId.
	 * @return the extension.
	 * @throws ExtensionNotFoundException
	 */
	public Extension<ActionXMLImporterExtensionRuntime> getXMLActionImporterExtension(final String extensionId) throws ExtensionNotFoundException {
		return XMLActionImporterRegistry.getExtension(extensionId);
	}

	private void setFrameFilterRegistry(final ConfigurableExtensionRegistry<FrameFilterExtensionRuntime<?>, FrameFilterExtensionConfig> frameFilterRegistry) {
		this.frameFilterRegistry = frameFilterRegistry;
	}

	private void setLayerFilterRegistry(final ConfigurableExtensionRegistry<LayerFilterExtensionRuntime<?>, LayerFilterExtensionConfig> imageFilterRegistry) {
		this.imageFilterRegistry = imageFilterRegistry;
	}

	private void setEffectRegistry(final ConfigurableExtensionRegistry<EffectExtensionRuntime<?>, EffectExtensionConfig> effectRegistry) {
		this.effectRegistry = effectRegistry;
	}

	private void setImageRegistry(final ConfigurableExtensionRegistry<ImageExtensionRuntime<?>, ImageExtensionConfig> imageRegistry) {
		this.imageRegistry = imageRegistry;
	}

	private void setNodeBuilderRegistry(final ExtensionRegistry<NodeBuilderExtensionRuntime> nodeBuilderRegistry) {
		this.nodeBuilderRegistry = nodeBuilderRegistry;
	}

	private void setXMLExtensionConfigExporterRegistry(final ExtensionRegistry<ExtensionConfigXMLExporterExtensionRuntime> XMLExtensionConfigExporterRegistry) {
		this.XMLExtensionConfigExporterRegistry = XMLExtensionConfigExporterRegistry;
	}

	private void setXMLExtensionConfigImporterRegistry(final ExtensionRegistry<ExtensionConfigXMLImporterExtensionRuntime> XMLExtensionConfigImporterRegistry) {
		this.XMLExtensionConfigImporterRegistry = XMLExtensionConfigImporterRegistry;
	}

	private void setXMLActionExporterRegistry(final ExtensionRegistry<ActionXMLExporterExtensionRuntime> XMLActionExporterRegistry) {
		this.XMLActionExporterRegistry = XMLActionExporterRegistry;
	}

	private void setXMLActionImporterRegistry(final ExtensionRegistry<ActionXMLImporterExtensionRuntime> XMLActionImporterRegistry) {
		this.XMLActionImporterRegistry = XMLActionImporterRegistry;
	}

	/**
	 * @return the frameFilterRegistry
	 */
	public ConfigurableExtensionRegistry<FrameFilterExtensionRuntime<?>, FrameFilterExtensionConfig> getFrameFilterRegistry() {
		return frameFilterRegistry;
	}

	/**
	 * @return the imageFilterRegistry
	 */
	public ConfigurableExtensionRegistry<LayerFilterExtensionRuntime<?>, LayerFilterExtensionConfig> getLayerFilterRegistry() {
		return imageFilterRegistry;
	}

	/**
	 * @return the effectRegistry
	 */
	public ConfigurableExtensionRegistry<EffectExtensionRuntime<?>, EffectExtensionConfig> getEffectRegistry() {
		return effectRegistry;
	}

	/**
	 * @return the imageRegistry
	 */
	public ConfigurableExtensionRegistry<ImageExtensionRuntime<?>, ImageExtensionConfig> getImageRegistry() {
		return imageRegistry;
	}

	/**
	 * @return the nodeBuilderRegistry
	 */
	public ExtensionRegistry<NodeBuilderExtensionRuntime> getNodeBuilderRegistry() {
		return nodeBuilderRegistry;
	}

	/**
	 * @return the XMLExtensionConfigExporterRegistry
	 */
	public ExtensionRegistry<ExtensionConfigXMLExporterExtensionRuntime> getXMLExtensionConfigExporterRegistry() {
		return XMLExtensionConfigExporterRegistry;
	}

	/**
	 * @return the XMLExtensionConfigImporterRegistry
	 */
	public ExtensionRegistry<ExtensionConfigXMLImporterExtensionRuntime> getXMLExtensionConfigImporterRegistry() {
		return XMLExtensionConfigImporterRegistry;
	}

	/**
	 * @return the XMLActionExporterRegistry
	 */
	public ExtensionRegistry<ActionXMLExporterExtensionRuntime> getXMLActionExporterRegistry() {
		return XMLActionExporterRegistry;
	}

	/**
	 * @return the XMLActionImporterRegistry
	 */
	public ExtensionRegistry<ActionXMLImporterExtensionRuntime> getXMLActionImporterRegistry() {
		return XMLActionImporterRegistry;
	}
}
