/*
 * $Id:MovieClipService.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.clip;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.List;

import net.sf.jame.service.DataTableException;
import net.sf.jame.service.Session;
import net.sf.jame.service.SessionException;

/**
 * @author Andrea Medeghini
 */
public class MovieClipService {
	private final MovieClipFileCache fileCache;
	private final MovieClipDataTable dataTable;

	/**
	 * @param workdir
	 * @throws IOException
	 */
	public MovieClipService(final File workdir) throws IOException {
		fileCache = new MovieClipFileCache(new File(workdir, "clip"));
		dataTable = new MovieClipDataTable();
	}

	/**
	 * @param session
	 * @param clip
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void create(final Session session, final MovieClipDataRow clip) throws SessionException, DataTableException {
		dataTable.create(session, clip);
	}

	/**
	 * @param session
	 * @param clip
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void delete(final Session session, final MovieClipDataRow clip) throws SessionException, DataTableException {
		dataTable.delete(session, clip);
		try {
			fileCache.delete(clip.getClipId());
		}
		catch (final IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param session
	 * @param clip
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void save(final Session session, final MovieClipDataRow clip) throws SessionException, DataTableException {
		dataTable.save(session, clip);
	}

	/**
	 * @param session
	 * @param id
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public boolean validate(final Session session, final int id) throws SessionException, DataTableException {
		final MovieClipDataRow clip = dataTable.load(session, id);
		return fileCache.exists(clip.getClipId());
	}

	/**
	 * @param session
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public List<MovieClipDataRow> loadAll(final Session session) throws SessionException, DataTableException {
		return dataTable.loadAll(session);
	}

	/**
	 * @param session
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void deleteAll(final Session session) throws SessionException, DataTableException {
		final List<MovieClipDataRow> clips = loadAll(session);
		for (final MovieClipDataRow clip : clips) {
			delete(session, clip);
		}
	}

	/**
	 * @param session
	 * @throws SessionException
	 */
	public void cleanup(final Session session) throws SessionException {
		final File[] files = fileCache.list();
		for (final File file : files) {
			try {
				final int id = Integer.parseInt(file.getName().substring(0, file.getName().indexOf(fileCache.getSuffix())));
				dataTable.load(session, id);
			}
			catch (final NumberFormatException e) {
				if (file.exists() && !file.delete()) {
					throw new SessionException("Can't delete the file: " + file.getName());
				}
			}
			catch (final DataTableException e) {
				if (file.exists() && !file.delete()) {
					throw new SessionException("Can't delete the file: " + file.getName());
				}
			}
		}
	}

	/**
	 * @param clipId
	 * @return
	 * @throws FileNotFoundException
	 */
	public InputStream getInputStream(final int clipId) throws FileNotFoundException {
		return fileCache.getInputStream(clipId);
	}

	/**
	 * @param clipId
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getOutputStream(final int clipId) throws FileNotFoundException {
		return fileCache.getOutputStream(clipId);
	}

	/**
	 * @param clipId
	 * @return
	 * @throws FileNotFoundException
	 */
	public RandomAccessFile getRandomAccessFile(final int clipId) throws FileNotFoundException {
		return fileCache.getRandomAccessFile(clipId);
	}

	/**
	 * @param session
	 * @param clipId
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public MovieClipDataRow getClip(final Session session, final int clipId) throws SessionException, DataTableException {
		return dataTable.load(session, clipId);
	}

	/**
	 * @param session
	 * @param clip
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void saveStatus(final Session session, final MovieClipDataRow clip) throws SessionException, DataTableException {
		dataTable.saveStatus(session, clip);
	}

	/**
	 * @param session
	 * @param clip
	 * @throws SessionException
	 * @throws DataTableException
	 * @throws IOException
	 */
	public void clean(final Session session, final MovieClipDataRow clip) throws SessionException, DataTableException, IOException {
		fileCache.delete(clip.getClipId());
	}

	/**
	 * @param session
	 * @throws DataTableException
	 * @throws SessionException
	 */
	public void init(final Session session) throws DataTableException, SessionException {
		dataTable.init(session);
	}
}
