/*
 * $Id:AsyncService.java 488 2008-01-27 09:21:39Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import net.sf.jame.service.clip.MovieClipDataRow;
import net.sf.jame.service.encoder.extension.EncoderExtensionRuntime;
import net.sf.jame.service.job.RenderJobDataRow;
import net.sf.jame.service.profile.RenderProfileDataRow;
import net.sf.jame.service.spool.DefaultJobService;
import net.sf.jame.service.spool.JobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.JobService;
import net.sf.jame.service.spool.SpoolJobInterface;
import net.sf.jame.service.spool.extension.SpoolExtensionConfig;
import net.sf.jame.service.spool.extension.SpoolExtensionRuntime;
import net.sf.jame.service.spool.impl.CopyProcessorJob;
import net.sf.jame.service.spool.impl.CopyProcessorJobFactory;
import net.sf.jame.service.spool.impl.PostProcessorJob;
import net.sf.jame.service.spool.impl.PostProcessorJobFactory;
import net.sf.jame.twister.DefaultThreadFactory;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class AsyncService {
	private static final Logger logger = Logger.getLogger(AsyncService.class);
	private final DefaultThreadFactory workerFactory = new DefaultThreadFactory("ServiceWorkerThread", true, Thread.NORM_PRIORITY);
	private final ServiceWorkerExecutor workerExecutor = new ServiceWorkerExecutor();
	private final HashMap<Integer, JobInterface> postProcessJobs = new HashMap<Integer, JobInterface>();
	private final HashMap<Integer, JobInterface> copyProcessJobs = new HashMap<Integer, JobInterface>();
	private final HashMap<Integer, JobInterface> processJobs = new HashMap<Integer, JobInterface>();
	private JobService<? extends SpoolJobInterface> processService;
	private final JobService<PostProcessorJob> postProcessService;
	private final JobService<CopyProcessorJob> copyProcessService;
	private final Service service;

	/**
	 * @param service
	 * @param jobService
	 */
	public AsyncService(final Service service, final JobService<? extends SpoolJobInterface> jobService) {
		this.service = service;
		workerExecutor.start();
		service.addServiceListener(new SpoolServiceListener());
		postProcessService = new DefaultJobService<PostProcessorJob>(new PostProcessorJobFactory(service));
		copyProcessService = new DefaultJobService<CopyProcessorJob>(new CopyProcessorJobFactory(service));
		postProcessService.start();
		copyProcessService.start();
		setJobService(jobService);
	}

	/**
	 * @param jobService
	 */
	public void setJobService(final JobService<? extends SpoolJobInterface> jobService) {
		if (processService != null) {
			processService.stop();
		}
		processService = jobService;
		processService.start();
	}

	/**
	 * @return
	 */
	public JobService<? extends SpoolJobInterface> getJobService() {
		return processService;
	}

	/**
	 * 
	 */
	public void start() {
		postProcessService.start();
		copyProcessService.start();
		if (processService != null) {
			processService.start();
		}
	}

	/**
	 * 
	 */
	public void stop() {
		copyProcessService.stop();
		postProcessService.stop();
		if (processService != null) {
			processService.stop();
		}
	}

	/**
	 * @return
	 */
	public Service getService() {
		return service;
	}

	/**
	 * @param callback
	 */
	public void loadClips(final ServiceCallback<List<MovieClipDataRow>> callback) {
		workerExecutor.execute(new ServiceWorker<List<MovieClipDataRow>>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceWorker#execute()
			 */
			@Override
			public List<MovieClipDataRow> execute() throws Exception {
				return service.loadClips();
			}
		});
	}

	/**
	 * @param callback
	 * @param clip
	 * @throws ServiceException
	 */
	public void createClip(final ServiceVoidCallback callback, final MovieClipDataRow clip) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.createClip(clip);
			}
		});
	}

	/**
	 * @param callback
	 * @param clip
	 * @throws ServiceException
	 */
	public void saveClip(final ServiceVoidCallback callback, final MovieClipDataRow clip) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.saveClip(clip);
			}
		});
	}

	/**
	 * @param callback
	 * @param clip
	 * @throws ServiceException
	 */
	public void deleteClip(final ServiceVoidCallback callback, final MovieClipDataRow clip) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.deleteClip(clip);
			}
		});
	}

	/**
	 * @param callback
	 * @param clipId
	 * @throws ServiceException
	 */
	public void getClip(final ServiceCallback<MovieClipDataRow> callback, final int clipId) {
		workerExecutor.execute(new ServiceWorker<MovieClipDataRow>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public MovieClipDataRow execute() throws Exception {
				return service.getClip(clipId);
			}
		});
	}

	/**
	 * @param callback
	 * @param clipId
	 * @throws ServiceException
	 */
	public void loadProfiles(final ServiceCallback<List<RenderProfileDataRow>> callback, final int clipId) {
		workerExecutor.execute(new ServiceWorker<List<RenderProfileDataRow>>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceWorker#execute()
			 */
			@Override
			public List<RenderProfileDataRow> execute() throws Exception {
				return service.loadProfiles(clipId);
			}
		});
	}

	/**
	 * @param callback
	 * @param clipId
	 */
	public void resetProfiles(final ServiceCallback<List<RenderProfileDataRow>> callback, final int clipId) {
		workerExecutor.execute(new ServiceWorker<List<RenderProfileDataRow>>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceWorker#execute()
			 */
			@Override
			public List<RenderProfileDataRow> execute() throws Exception {
				return service.resetProfiles(clipId);
			}
		});
	}

	/**
	 * @param callback
	 * @param profile
	 * @throws ServiceException
	 */
	public void createProfile(final ServiceVoidCallback callback, final RenderProfileDataRow profile) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.createProfile(profile);
			}
		});
	}

	/**
	 * @param callback
	 * @param profile
	 * @throws ServiceException
	 */
	public void saveProfile(final ServiceVoidCallback callback, final RenderProfileDataRow profile) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.saveProfile(profile);
			}
		});
	}

	/**
	 * @param callback
	 * @param profile
	 * @throws ServiceException
	 */
	public void deleteProfile(final ServiceVoidCallback callback, final RenderProfileDataRow profile) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.deleteProfile(profile);
			}
		});
	}

	/**
	 * @param callback
	 * @param profileId
	 * @throws ServiceException
	 */
	public void getProfile(final ServiceCallback<RenderProfileDataRow> callback, final int profileId) {
		workerExecutor.execute(new ServiceWorker<RenderProfileDataRow>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public RenderProfileDataRow execute() throws Exception {
				return service.getProfile(profileId);
			}
		});
	}

	/**
	 * @param callback
	 * @throws ServiceException
	 */
	public void loadJobs(final ServiceCallback<List<RenderJobDataRow>> callback) {
		workerExecutor.execute(new ServiceWorker<List<RenderJobDataRow>>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceWorker#execute()
			 */
			@Override
			public List<RenderJobDataRow> execute() throws Exception {
				return service.loadJobs();
			}
		});
	}

	/**
	 * @param callback
	 * @param job
	 * @throws ServiceException
	 */
	public void createJob(final ServiceVoidCallback callback, final RenderJobDataRow job) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.createJob(job);
			}
		});
	}

	/**
	 * @param callback
	 * @param job
	 * @throws ServiceException
	 */
	public void saveJob(final ServiceVoidCallback callback, final RenderJobDataRow job) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.saveJob(job);
			}
		});
	}

	/**
	 * @param callback
	 * @param job
	 * @throws ServiceException
	 */
	public void deleteJob(final ServiceVoidCallback callback, final RenderJobDataRow job) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.deleteJob(job);
			}
		});
	}

	/**
	 * @param callback
	 * @param jobId
	 * @throws ServiceException
	 */
	public void getJob(final ServiceCallback<RenderJobDataRow> callback, final int jobId) {
		workerExecutor.execute(new ServiceWorker<RenderJobDataRow>(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public RenderJobDataRow execute() throws Exception {
				return service.getJob(jobId);
			}
		});
	}

	/**
	 * @param callback
	 * @throws ServiceException
	 */
	public void startJobs(final ServiceVoidCallback callback) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.startJobs();
			}
		});
	}

	/**
	 * @param callback
	 * @throws ServiceException
	 */
	public void stopJobs(final ServiceVoidCallback callback) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.stopJobs();
			}
		});
	}

	/**
	 * @param callback
	 * @throws ServiceException
	 */
	public void deleteJobs(final ServiceVoidCallback callback) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.deleteJobs();
			}
		});
	}

	/**
	 * @param callback
	 * @param profileId
	 */
	public void startJobs(final ServiceVoidCallback callback, final int profileId) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.startJobs(profileId);
			}
		});
	}

	/**
	 * @param callback
	 * @param profileId
	 */
	public void stopJobs(final ServiceVoidCallback callback, final int profileId) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.stopJobs(profileId);
			}
		});
	}

	/**
	 * @param callback
	 * @param profileId
	 * @throws ServiceException
	 */
	public void deleteJobs(final ServiceVoidCallback callback, final int profileId) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.deleteJobs(profileId);
			}
		});
	}

	/**
	 * @param callback
	 * @param profileId
	 * @throws ServiceException
	 */
	public void createJobs(final ServiceVoidCallback callback, final int profileId) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.createJobs(profileId);
			}
		});
	}

	/**
	 * @param callback
	 * @param job
	 * @throws ServiceException
	 */
	public void jobCompleted(final ServiceVoidCallback callback, final RenderJobDataRow job) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.jobCompleted(job);
			}
		});
	}

	/**
	 * @param callback
	 * @param clip
	 */
	public void cleanClip(final ServiceVoidCallback callback, final MovieClipDataRow clip) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.cleanClip(clip);
			}
		});
	}

	/**
	 * @param callback
	 * @param profile
	 */
	public void cleanProfile(final ServiceVoidCallback callback, final RenderProfileDataRow profile) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.cleanProfile(profile);
			}
		});
	}

	/**
	 * @param callback
	 * @param job
	 */
	public void cleanJob(final ServiceVoidCallback callback, final RenderJobDataRow job) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.cleanJob(job);
			}
		});
	}

	/**
	 * @param callback
	 */
	public void resumeJobs(final ServiceVoidCallback callback) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.resumeJobs();
			}
		});
	}

	/**
	 * @param callback
	 * @param clip
	 * @param file
	 * @throws ServiceException
	 */
	public void importClip(final ServiceVoidCallback callback, final MovieClipDataRow clip, final File file) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.importClip(clip, file);
			}
		});
	}

	/**
	 * @param callback
	 * @param clip
	 * @param file
	 * @throws ServiceException
	 */
	public void exportClip(final ServiceVoidCallback callback, final MovieClipDataRow clip, final File file) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.exportClip(clip, file);
			}
		});
	}

	/**
	 * @param callback
	 * @param profile
	 * @param encoder
	 * @param path
	 */
	public void exportProfile(final ServiceVoidCallback callback, final RenderProfileDataRow profile, final EncoderExtensionRuntime<?> encoder, final File path) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.exportProfile(profile, encoder, path);
			}
		});
	}

	/**
	 * @param callback
	 * @param listener
	 */
	public void addServiceListener(final ServiceVoidCallback callback, final ServiceListener listener) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.addServiceListener(listener);
			}
		});
	}

	/**
	 * @param callback
	 * @param listener
	 */
	public void removeServiceListener(final ServiceVoidCallback callback, final ServiceListener listener) {
		workerExecutor.execute(new ServiceVoidWorker(callback) {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidWorker#execute()
			 */
			@Override
			public void execute() throws Exception {
				service.removeServiceListener(listener);
			}
		});
	}

	private class ServiceWorkerExecutor implements Runnable {
		private final List<Runnable> activeWorkers = new ArrayList<Runnable>();
		private final List<Runnable> workers = new ArrayList<Runnable>();
		private final Object lock = new Object();
		private Thread thread;

		/**
		 * 
		 */
		public void start() {
			if (thread == null) {
				thread = workerFactory.newThread(this);
				thread.start();
			}
		}

		/**
		 * 
		 */
		public void stop() {
			if (thread != null) {
				thread.interrupt();
				try {
					thread.join();
					thread = null;
				}
				catch (final InterruptedException e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				while (!Thread.currentThread().isInterrupted()) {
					synchronized (lock) {
						activeWorkers.clear();
						if (workers.size() == 0) {
							lock.wait();
						}
						if (workers.size() > 0) {
							activeWorkers.addAll(workers);
							workers.clear();
						}
					}
					try {
						for (final Runnable worker : activeWorkers) {
							worker.run();
							Thread.yield();
						}
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					Thread.yield();
				}
			}
			catch (final InterruptedException e) {
				e.printStackTrace();
			}
		}

		/**
		 * @param worker
		 */
		public void execute(final Runnable worker) {
			synchronized (lock) {
				workers.add(worker);
				lock.notify();
			}
		}
	}

	public static class ServiceWorker<T> implements Runnable {
		private final ServiceCallback<T> callback;

		/**
		 * @param callback
		 * @param hasValue
		 */
		public ServiceWorker(final ServiceCallback<T> callback) {
			this.callback = callback;
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public final void run() {
			try {
				final T value = this.execute();
				this.callback.executed(value);
			}
			catch (final Exception e) {
				e.printStackTrace();
				this.callback.failed(e);
			}
		}

		/**
		 * @return
		 * @throws Exception
		 */
		public T execute() throws Exception {
			return null;
		}
	}

	public static class ServiceVoidWorker implements Runnable {
		private final ServiceVoidCallback callback;

		/**
		 * @param callback
		 */
		public ServiceVoidWorker(final ServiceVoidCallback callback) {
			this.callback = callback;
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public final void run() {
			try {
				execute();
				callback.executed();
			}
			catch (final Exception e) {
				e.printStackTrace();
				callback.failed(e);
			}
		}

		/**
		 * @throws Exception
		 */
		public void execute() throws Exception {
		}
	}

	public static interface ServiceCallback<T> {
		/**
		 * @param throwable
		 */
		public void failed(Throwable throwable);

		/**
		 * @param value
		 */
		public void executed(T value);
	}

	public static interface ServiceVoidCallback {
		/**
		 * @param throwable
		 */
		public void failed(Throwable throwable);

		/**
		 * 
		 */
		public void executed();
	}

	private class SpoolServiceListener implements ServiceListener {
		/**
		 * @see net.sf.jame.service.ServiceListener#clipCreated(net.sf.jame.service.clip.MovieClipDataRow)
		 */
		public void clipCreated(final MovieClipDataRow clip) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#clipDeleted(net.sf.jame.service.clip.MovieClipDataRow)
		 */
		public void clipDeleted(final MovieClipDataRow clip) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#clipUpdated(net.sf.jame.service.clip.MovieClipDataRow)
		 */
		public void clipUpdated(final MovieClipDataRow clip) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobCreated(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobCreated(final RenderJobDataRow job) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobDeleted(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobDeleted(final RenderJobDataRow job) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobStarted(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobStarted(final RenderJobDataRow job) {
			if (job.isPostProcess()) {
				final String jobId = postProcessService.createJob(new SpoolJobListener());
				final PostProcessorJob tmpJob = postProcessService.getJob(jobId);
				tmpJob.setJobDataRow(job);
				postProcessJobs.put(job.getJobId(), tmpJob);
				logger.info("Job " + tmpJob + " created");
			}
			else if (job.isCopyProcess()) {
				final String jobId = copyProcessService.createJob(new SpoolJobListener());
				final CopyProcessorJob tmpJob = copyProcessService.getJob(jobId);
				tmpJob.setJobDataRow(job);
				copyProcessJobs.put(job.getJobId(), tmpJob);
				logger.info("Job " + tmpJob + " created");
			}
			else {
				final String jobId = processService.createJob(new SpoolJobListener());
				final SpoolJobInterface tmpJob = processService.getJob(jobId);
				tmpJob.setJobDataRow(job);
				processJobs.put(job.getJobId(), tmpJob);
				logger.info("Job " + tmpJob + " created");
			}
			if (job.isPostProcess()) {
				final JobInterface tmpJob = postProcessJobs.get(job.getJobId());
				if (tmpJob != null) {
					postProcessService.runJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " started");
				}
			}
			else if (job.isCopyProcess()) {
				final JobInterface tmpJob = copyProcessJobs.get(job.getJobId());
				if (tmpJob != null) {
					copyProcessService.runJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " started");
				}
			}
			else {
				final JobInterface tmpJob = processJobs.get(job.getJobId());
				if (tmpJob != null) {
					processService.runJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " started");
				}
			}
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobAborted(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobAborted(final RenderJobDataRow job) {
			if (job.isPostProcess()) {
				final JobInterface tmpJob = postProcessJobs.get(job.getJobId());
				if (tmpJob != null) {
					postProcessService.abortJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " stopped");
				}
			}
			else if (job.isCopyProcess()) {
				final JobInterface tmpJob = copyProcessJobs.get(job.getJobId());
				if (tmpJob != null) {
					copyProcessService.abortJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " stopped");
				}
			}
			else {
				final JobInterface tmpJob = processJobs.get(job.getJobId());
				if (tmpJob != null) {
					processService.abortJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " stopped");
				}
			}
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobStopped(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobStopped(final RenderJobDataRow job) {
			if (job.isPostProcess()) {
				final JobInterface tmpJob = postProcessJobs.remove(job.getJobId());
				if (tmpJob != null) {
					postProcessService.deleteJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " deleted");
				}
			}
			else if (job.isCopyProcess()) {
				final JobInterface tmpJob = copyProcessJobs.remove(job.getJobId());
				if (tmpJob != null) {
					copyProcessService.deleteJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " deleted");
				}
			}
			else {
				final JobInterface tmpJob = processJobs.remove(job.getJobId());
				if (tmpJob != null) {
					processService.deleteJob(tmpJob.getJobId());
					logger.info("Job " + tmpJob + " deleted");
				}
			}
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobUpdated(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobUpdated(final RenderJobDataRow job) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#jobResumed(net.sf.jame.service.job.RenderJobDataRow)
		 */
		public void jobResumed(final RenderJobDataRow job) {
			if (job.isPostProcess()) {
				final String jobId = postProcessService.createJob(new SpoolJobListener());
				final PostProcessorJob tmpJob = postProcessService.getJob(jobId);
				tmpJob.setJobDataRow(job);
				postProcessJobs.put(job.getJobId(), tmpJob);
				logger.info("Job " + tmpJob + " created");
			}
			else if (job.isCopyProcess()) {
				final String jobId = copyProcessService.createJob(new SpoolJobListener());
				final CopyProcessorJob tmpJob = copyProcessService.getJob(jobId);
				tmpJob.setJobDataRow(job);
				copyProcessJobs.put(job.getJobId(), tmpJob);
				logger.info("Job " + tmpJob + " created");
			}
			else {
				final String jobId = processService.createJob(new SpoolJobListener());
				final SpoolJobInterface tmpJob = processService.getJob(jobId);
				tmpJob.setJobDataRow(job);
				processJobs.put(job.getJobId(), tmpJob);
				logger.info("Job " + tmpJob + " created");
			}
//			jobStarted(job);
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#profileCreated(net.sf.jame.service.profile.RenderProfileDataRow)
		 */
		public void profileCreated(final RenderProfileDataRow profile) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#profileDeleted(net.sf.jame.service.profile.RenderProfileDataRow)
		 */
		public void profileDeleted(final RenderProfileDataRow profile) {
		}

		/**
		 * @see net.sf.jame.service.ServiceListener#profileUpdated(net.sf.jame.service.profile.RenderProfileDataRow)
		 */
		public void profileUpdated(final RenderProfileDataRow profile) {
		}
	}

	private class SpoolJobListener implements JobListener {
		/**
		 * @see net.sf.jame.service.spool.JobListener#stateChanged(net.sf.jame.p2p.job.test.DummyJob)
		 */
		public void stateChanged(final JobInterface job) {
			if (logger.isDebugEnabled()) {
				logger.debug("Job state changed " + job);
			}
			try {
				service.saveJobStatus(((SpoolJobInterface) job).getJobDataRow());
				service.fireJobUpdated(((SpoolJobInterface) job).getJobDataRow());
			}
			catch (final ServiceException e) {
				e.printStackTrace();
			}
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#started(net.sf.jame.p2p.job.test.DummyJob)
		 */
		public void started(final JobInterface job) {
			logger.info("Job started " + job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stopped(net.sf.jame.p2p.job.test.DummyJob)
		 */
		public void stopped(final JobInterface job) {
			logger.info("Job stopped " + job);
			if (!job.isAborted()) {
				jobCompleted(new ServiceVoidCallback() {
					public void executed() {
						logger.info("Job completed " + job);
					}

					public void failed(final Throwable throwable) {
						logger.info("Job completed failed: " + job);
					}
				}, ((SpoolJobInterface) job).getJobDataRow());
			}
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#terminated(net.sf.jame.service.spool.JobInterface)
		 */
		public void terminated(final JobInterface job) {
			logger.info("Job terminated " + job);
		}
	}
}
