/*
 * $Id:SpoolJobServiceTest.java 474 2008-01-13 18:30:34Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.networking.test.spool;

import java.io.File;
import java.io.OutputStream;
import java.util.HashMap;

import net.sf.jame.core.xml.XML;
import net.sf.jame.core.xml.XMLNodeBuilder;
import net.sf.jame.networking.jxta.JXTADiscoveryService;
import net.sf.jame.networking.jxta.JXTANetworkService;
import net.sf.jame.networking.jxta.JXTASpoolJobService;
import net.sf.jame.networking.spool.DistributedServiceProcessor;
import net.sf.jame.networking.spool.LocalServiceProcessor;
import net.sf.jame.service.ConnectionFactory;
import net.sf.jame.service.DefaultConnectionFactory;
import net.sf.jame.service.Service;
import net.sf.jame.service.ServiceListener;
import net.sf.jame.service.Session;
import net.sf.jame.service.clip.MovieClip;
import net.sf.jame.service.clip.MovieClipDataRow;
import net.sf.jame.service.extensions.encoder.JPEGEncoderConfig;
import net.sf.jame.service.extensions.encoder.JPEGEncoderRuntime;
import net.sf.jame.service.job.RenderJobDataRow;
import net.sf.jame.service.profile.RenderProfile;
import net.sf.jame.service.profile.RenderProfileDataRow;
import net.sf.jame.service.spool.DefaultJobService;
import net.sf.jame.service.spool.JobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.impl.DistributedJob;
import net.sf.jame.service.spool.impl.DistributedJobFactory;
import net.sf.jame.service.spool.impl.DistributedSpoolJob;
import net.sf.jame.service.spool.impl.DistributedSpoolJobFactory;
import net.sf.jame.service.spool.impl.LocalSpoolJob;
import net.sf.jame.service.spool.impl.LocalSpoolJobFactory;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipXMLExporter;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterConfigBuilder;
import net.sf.jame.twister.TwisterSequence;
import net.sf.jame.twister.renderer.Surface;

import org.apache.log4j.Logger;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author Andrea Medeghini
 */
public class SpoolJobServiceTest {
	private static final Logger logger = Logger.getLogger(SpoolJobServiceTest.class);

	private void deleteFiles(final File path) {
		final File[] files = path.listFiles();
		for (final File file : files) {
			if (file.isDirectory()) {
				deleteFiles(file);
			}
			file.delete();
		}
	}

	@Test
	public void testSpool() throws Exception {
		final Surface surface = new Surface(200, 200);
		final File workDir = new File("workdir");
		final File tmpDir = new File(workDir, "tmp");
		tmpDir.mkdirs();
		deleteFiles(tmpDir);
		final ConnectionFactory factory = new DefaultConnectionFactory(workDir);
		final Session session = new Session(factory);
		final Service service = new Service(session, workDir);
		final MovieClipDataRow movieClip = new MovieClipDataRow(new MovieClip());
		movieClip.setClipName("test clip");
		service.createClip(movieClip);
		final RenderProfileDataRow renderProfile = new RenderProfileDataRow(new RenderProfile());
		renderProfile.setProfileName("test profile");
		renderProfile.setClipName(movieClip.getClipName());
		renderProfile.setClipId(movieClip.getClipId());
		renderProfile.setImageWidth(640);
		renderProfile.setImageHeight(480);
		renderProfile.setOffsetX(0);
		renderProfile.setOffsetY(0);
		renderProfile.setStartTime(0);
		renderProfile.setStopTime(0);
		service.createProfile(renderProfile);
		final TwisterClip clip = new TwisterClip();
		final TwisterSequence sequence = new TwisterSequence();
		clip.addSequence(sequence);
		final TwisterConfigBuilder configBuilder = new TwisterConfigBuilder();
		final TwisterConfig config = configBuilder.createDefaultConfig();
		sequence.setFinalConfig(config);
		final TwisterClipXMLExporter exporter = new TwisterClipXMLExporter();
		final Document doc = XML.createDocument();
		final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
		final Element element = exporter.exportToElement(clip, builder);
		doc.appendChild(element);
		final OutputStream os = service.getClipOutputStream(movieClip.getClipId());
		XML.saveDocument(os, "twister-clip.xml", doc);
		os.close();
		final DistributedServiceProcessor processor1 = new DistributedServiceProcessor(new DefaultJobService<DistributedJob>(new DistributedJobFactory(tmpDir)), 10);
		final LocalServiceProcessor processor2 = new LocalServiceProcessor(service, new DefaultJobService<LocalSpoolJob>(new LocalSpoolJobFactory(service)), 10);
		final JXTASpoolJobService jobService = new JXTASpoolJobService(new JXTADiscoveryService(new JXTANetworkService(tmpDir, "http://jame.sf.net", "JAME-Service", "Andrea Medeghini", "1.0", processor1), processor2), new DistributedSpoolJobFactory(service));
		jobService.start();
		processor1.start();
		processor2.start();
		final HashMap<Integer, DistributedSpoolJob> jobs = new HashMap<Integer, DistributedSpoolJob>();
		service.addServiceListener(new ServiceListener() {
			public void clipCreated(final MovieClipDataRow clip) {
			}

			public void clipDeleted(final MovieClipDataRow clip) {
			}

			public void clipUpdated(final MovieClipDataRow clip) {
			}

			public void jobCreated(final RenderJobDataRow job) {
				final String jobId = jobService.createJob(new TestListener());
				final DistributedSpoolJob spoolJob = (DistributedSpoolJob) jobService.getJob(jobId);
				spoolJob.setJobDataRow(job);
				jobs.put(job.getJobId(), spoolJob);
				logger.info("Job " + spoolJob + " created");
			}

			public void jobDeleted(final RenderJobDataRow job) {
				final DistributedSpoolJob spoolJob = jobs.get(job.getJobId());
				jobService.deleteJob(spoolJob.getJobId());
				logger.info("Job " + spoolJob + " deleted");
			}

			public void jobStarted(final RenderJobDataRow job) {
				final DistributedSpoolJob spoolJob = jobs.get(job.getJobId());
				jobService.runJob(spoolJob.getJobId());
				logger.info("Job " + spoolJob + " started");
			}

			public void jobStopped(final RenderJobDataRow job) {
				final DistributedSpoolJob spoolJob = jobs.get(job.getJobId());
				jobService.abortJob(spoolJob.getJobId());
				logger.info("Job " + spoolJob + " stopped");
			}

			public void jobAborted(final RenderJobDataRow job) {
			}

			public void jobUpdated(final RenderJobDataRow job) {
			}

			public void jobResumed(final RenderJobDataRow job) {
			}

			public void profileCreated(final RenderProfileDataRow profile) {
			}

			public void profileDeleted(final RenderProfileDataRow profile) {
			}

			public void profileUpdated(final RenderProfileDataRow profile) {
			}
		});
		service.createJobs(renderProfile.getProfileId());
		service.startJobs(renderProfile.getProfileId());
		Thread.sleep(5000);
		try {
			while (true) {
				boolean terminated = true;
				for (final DistributedSpoolJob job : jobs.values()) {
					if (!job.isTerminated()) {
						terminated = false;
					}
				}
				if (terminated) {
					break;
				}
				Thread.sleep(10000);
			}
		}
		catch (final InterruptedException e) {
		}
		service.stopJobs(renderProfile.getProfileId());
		for (final DistributedSpoolJob job : jobs.values()) {
			service.jobCompleted(job.getJobDataRow());
		}
		final JPEGEncoderRuntime encoder = new JPEGEncoderRuntime();
		encoder.setConfig(new JPEGEncoderConfig());
		service.exportProfile(renderProfile, encoder, new File("test.jpeg"));
		processor1.stop();
		processor2.stop();
		jobService.stop();
		session.close();
	}

	private static class TestListener implements JobListener {
		/**
		 * @see net.sf.jame.service.spool.JobListener#stateChanged(net.sf.jame.p2p.job.test.DummyJob)
		 */
		public void stateChanged(final JobInterface job) {
			logger.info("Job state changed " + job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#started(net.sf.jame.p2p.job.test.DummyJob)
		 */
		public void started(final JobInterface job) {
			logger.info("Job started " + job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stopped(net.sf.jame.p2p.job.test.DummyJob)
		 */
		public void stopped(final JobInterface job) {
			logger.info("Job stopped " + job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#terminated(net.sf.jame.service.spool.JobInterface)
		 */
		public void terminated(final JobInterface job) {
			logger.info("Job terminated " + job);
		}
	}
}
