/*
 * $Id:Engine.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.media.swing;

import java.awt.Canvas;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.Window;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.geom.Point2D;
import java.awt.image.BufferStrategy;
import java.awt.image.VolatileImage;

import net.sf.jame.media.Context;
import net.sf.jame.media.Controller;
import net.sf.jame.media.EngineMouseEvent;
import net.sf.jame.media.Movie;
import net.sf.jame.media.Pipeline;

public final class Engine implements Runnable {
	private static final boolean debug = false;
	private Thread thread;
	Movie movie;
	private final Context context;
	Pipeline pipeline;
	private Controller controller;
	Dimension component_size;
	private Dimension movie_size;
	boolean stop = false;
	private int frame = 0;
	private float center_x = 0;
	private float center_y = 0;
	Component component;
	private VolatileImage image;
	// private TiledImage image;
	private Graphics2D graphics;
	private BufferStrategy strategy;

	public Engine(final Context context, final Window component) {
		this.context = context;
		this.component = component;
	}

	public Engine(final Context context, final Canvas component) {
		this.context = context;
		this.component = component;
	}

	public void start() {
		if (thread == null) {
			thread = new Thread(this);
			thread.start();
		}
	}

	public void stop() {
		if ((thread != null) && thread.isAlive()) {
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
		}
		thread = null;
	}

	public void run() {
		/*
		 * Font[] fonts = GraphicsEnvironment.getLocalGraphicsEnvironment().getAllFonts(); for (int i = 0; i < fonts.length; i++) { System.out.println(fonts[i]); }
		 */
		if (context.debug()) {
			context.println("engine started");
		}
		if (component instanceof Window) {
			((Window) component).createBufferStrategy(2);
			strategy = ((Window) component).getBufferStrategy();
		}
		else if (component instanceof Canvas) {
			((Canvas) component).createBufferStrategy(2);
			strategy = ((Canvas) component).getBufferStrategy();
		}
		final keyAdapter keyadapter = new keyAdapter();
		final mouseAdapter mouseadapter = new mouseAdapter();
		final componentAdapter componentadapter = new componentAdapter();
		final mouseMotionAdapter mousemotionadapter = new mouseMotionAdapter();
		pipeline = new Pipeline(context, context.getColor());
		controller = new EngineController();
		movie = context.getMovie();
		movie.load(component);
		movie_size = movie.getSize();
		component_size = component.getSize();
		final Point2D center = movie.getCenter();
		center_x = (float) center.getX();
		center_y = (float) center.getY();
		allocate(component_size);
		scale(component_size);
		if (context.debug()) {
			context.println("size = " + component_size.width + " x " + component_size.height);
			context.println("rate = " + movie.getFrameRate() + " frame/s");
		}
		component.addKeyListener(keyadapter);
		component.addMouseListener(mouseadapter);
		component.addComponentListener(componentadapter);
		component.addMouseMotionListener(mousemotionadapter);
		Thread.currentThread().setPriority(Thread.MAX_PRIORITY - 1);
		movie.build(controller, null, null, null);
		movie.init();
		ciclo();
		movie.kill();
		movie.flush();
		pipeline.kill();
		component.removeKeyListener(keyadapter);
		component.removeMouseListener(mouseadapter);
		component.removeComponentListener(componentadapter);
		component.removeMouseMotionListener(mousemotionadapter);
		if (context.debug()) {
			context.println("engine stopped");
		}
		context.exit(0);
	}

	public Controller getController() {
		return controller;
	}

	private void ciclo() {
		try {
			int skip = 1;
			long start_time = 0;
			long total_time = 0;
			long frame_time = 0;
			long sleep_time = 0;
			long mean_time = 0;
			final long min_time = (1000 << 16) / movie.getFrameRate();
			long max_time = min_time;
			long low_time = max_time / 3;
			long high_time = max_time - (max_time / (skip + 2));
			// long last_time[] = new long[4];
			// int index = 0;
			movie.setFrame(0);
			while (!Thread.interrupted()) {
				start_time = System.currentTimeMillis();
				check(component_size);
				synchronized (pipeline) {
					pipeline.rendering(graphics, component_size.width, component_size.height, movie);
				}
				for (int i = 0; i < skip; i++) {
					if (!stop) {
						movie.nextFrame();
					}
					if (movie.getFrame() == movie.getFrames()) {
						if (context.loop()) {
							movie.reset();
							movie.setFrame(0);
						}
						else {
							stop = true;
						}
					}
				}
				if (component.isDisplayable()) {
					final Graphics g = strategy.getDrawGraphics();
					g.drawImage(image, 0, 0, null);
					g.dispose();
					strategy.show();
				}
				frame_time = (System.currentTimeMillis() - start_time) << 16;
				frame += skip;
				total_time += frame_time;
				mean_time = total_time / frame;
				if ((frame_time >= high_time) || ((mean_time >= high_time) && (skip < 20))) {
					skip += 1;
					max_time += min_time;
					low_time = max_time / 3;
					high_time = max_time - (max_time / (skip + 3));
					if (context.debug() && Engine.debug) {
						context.println("skip = " + skip);
					}
				}
				else if ((frame_time < low_time) && (mean_time < low_time) && (skip > 1)) {
					skip -= 1;
					max_time -= min_time;
					low_time = max_time / 3;
					high_time = max_time - (max_time / (skip + 3));
					if (context.debug() && Engine.debug) {
						context.println("skip = " + skip);
					}
				}
				frame_time = (System.currentTimeMillis() - start_time) << 16;
				// last_time[(index + 1) & 0x03] = (System.currentTimeMillis() - start_time) << 16;
				// sleep_time = max_time - ((last_time[0] + last_time[1] + last_time[2] + last_time[3] + 1) >> 2);
				if (context.debug() && Engine.debug) {
					context.println("mean time = " + (mean_time >> 16));
				}
				sleep_time = max_time - frame_time;
				if (sleep_time > 0) {
					Thread.sleep(sleep_time >> 16);
				}
				else {
					Thread.yield();
				}
			}
		}
		catch (final InterruptedException e) {
		}
		catch (final NullPointerException e) {
			e.printStackTrace();
		}
	}

	private void allocate(final Dimension size) {
		image = component.getGraphicsConfiguration().createCompatibleVolatileImage(size.width, size.height);
		// image = component.getGraphicsConfiguration().createCompatibleImage(size.width, size.height);
		graphics = (Graphics2D) image.getGraphics();
		setRenderingHints(graphics);
	}

	void reallocate(final Dimension size) {
		graphics.dispose();
		allocate(size);
		if (context.debug()) {
			context.println("buffer re-allocated");
		}
	}

	private void check(final Dimension size) {
		if (image.validate(component.getGraphicsConfiguration()) == VolatileImage.IMAGE_INCOMPATIBLE) {
			if (context.debug()) {
				context.println("incompatible image!");
			}
			reallocate(size);
		}
	}

	void scale(final Dimension size) {
		movie.getTransform().setToScale(size.width / (float) movie_size.width, size.height / (float) movie_size.height);
		movie.translate(-center_x, -center_y);
	}

	private void setRenderingHints(final Graphics2D graphics) {
		graphics.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
		graphics.setRenderingHint(RenderingHints.KEY_DITHERING, RenderingHints.VALUE_DITHER_DISABLE);
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		graphics.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
		graphics.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
		graphics.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
		graphics.setRenderingHint(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_SPEED);
		graphics.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
		graphics.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_PURE);
	}

	private class EngineController implements Controller {
		public void play() {
			stop = false;
		}

		public void stop() {
			stop = true;
		}

		public void gotoAndPlay(final int frame) {
			movie.setFrame(frame % movie.getFrames());
			stop = false;
		}

		public void gotoAndStop(final int frame) {
			movie.setFrame(frame % movie.getFrames());
			stop = true;
		}
	}

	private class mouseAdapter extends MouseAdapter {
		public mouseAdapter() {
		}

		@Override
		public void mousePressed(final MouseEvent e) {
			pipeline.enqueueEvent(new EngineMouseEvent(EngineMouseEvent.PRESSED, new Point2D.Float(e.getX(), e.getY())));
		}

		@Override
		public void mouseReleased(final MouseEvent e) {
			pipeline.enqueueEvent(new EngineMouseEvent(EngineMouseEvent.RELEASED, new Point2D.Float(e.getX(), e.getY())));
		}
	}

	private class mouseMotionAdapter extends MouseMotionAdapter {
		public mouseMotionAdapter() {
		}

		@Override
		public void mouseMoved(final MouseEvent e) {
			pipeline.enqueueEvent(new EngineMouseEvent(EngineMouseEvent.MOVED, new Point2D.Float(e.getX(), e.getY())));
		}

		@Override
		public void mouseDragged(final MouseEvent e) {
			pipeline.enqueueEvent(new EngineMouseEvent(EngineMouseEvent.DRAGGED, new Point2D.Float(e.getX(), e.getY())));
		}
	}

	private class keyAdapter extends KeyAdapter {
		public keyAdapter() {
		}

		@Override
		public void keyPressed(final KeyEvent e) {
		}
	}

	private class componentAdapter extends ComponentAdapter {
		public componentAdapter() {
		}

		@Override
		public void componentResized(final ComponentEvent e) {
			synchronized (pipeline) {
				component_size = component.getSize();
				reallocate(component_size);
				scale(component_size);
			}
		}
	}
}
