/*
 * $Id:Timeline.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.media;

import java.util.ListIterator;

public final class Timeline {
	private final List layers = new List();
	private boolean dirty = false;
	private int frames = 0;
	private int frame = 0;

	public Timeline() {
	}

	public Timeline(final List list) {
		layers.addAll(list);
	}

	@Override
	public Object clone() throws CloneNotSupportedException {
		return new Timeline((List) layers.clone());
	}

	@Override
	public String toString() {
		return "timeline " + hashCode();
	}

	public void removeLayer(final Layer layer) {
		layers.remove(layer);
		dirty = false;
	}

	public void addLayerFirst(final Layer layer) {
		layers.addFirst(layer);
		dirty = false;
	}

	public void addLayerLast(final Layer layer) {
		layers.addLast(layer);
		dirty = false;
	}

	public void removeLayerFirst() {
		layers.removeFirst();
		dirty = false;
	}

	public void removeLayerLast() {
		layers.removeLast();
		dirty = false;
	}

	public void removeLayers() {
		layers.clear();
		dirty = false;
	}

	public void setLayer(final int index, final Layer layer) {
		layers.set(index, layer);
		dirty = false;
	}

	public Layer getLayer(final int index) {
		return (Layer) layers.get(index);
	}

	public int getLayers() {
		return layers.size();
	}

	public int getFrames() {
		return frames;
	}

	public int getFrame() {
		return frame;
	}

	public boolean isFirstFrame() {
		return (frame == 0);
	}

	public boolean isLastFrame() {
		return (frame == (frames - 1));
	}

	boolean isChanged() {
		boolean changed = false;
		final ListIterator li = layers.listIterator();
		while (li.hasNext()) {
			if (((Layer) li.next()).isChanged()) {
				changed = true;
				break;
			}
		}
		return changed;
	}

	void buildFrames() {
		frames = 0;
		final ListIterator li = layers.listIterator();
		while (li.hasNext()) {
			frames = Math.max(frames, ((Layer) li.next()).getFrames());
		}
		dirty = true;
	}

	void setFrame(final int frame) {
		if ((!dirty) || (isChanged())) {
			buildFrames();
		}
		this.frame = frame;
		if ((this.frame >= 0) && (this.frame < frames)) {
			final ListIterator li = layers.listIterator();
			while (li.hasNext()) {
				((Layer) li.next()).setFrame(this.frame);
			}
		}
	}

	void nextFrame() {
		if ((!dirty) || (isChanged())) {
			buildFrames();
		}
		frame = frame + 1;
		if ((frame >= 0) && (frame < frames)) {
			final ListIterator li = layers.listIterator();
			while (li.hasNext()) {
				((Layer) li.next()).nextFrame();
			}
		}
	}

	void prevFrame() {
		if ((!dirty) || (isChanged())) {
			buildFrames();
		}
		frame = frame - 1;
		if ((frame >= 0) && (frame < frames)) {
			final ListIterator li = layers.listIterator();
			while (li.hasNext()) {
				((Layer) li.next()).prevFrame();
			}
		}
	}

	void build(final Controller engine, final Movie movie) {
		final ListIterator li = layers.listIterator();
		while (li.hasNext()) {
			((Layer) li.next()).build(engine, movie);
		}
	}

	void init() {
		final ListIterator li = layers.listIterator();
		while (li.hasNext()) {
			((Layer) li.next()).init();
		}
	}

	void kill() {
		final ListIterator li = layers.listIterator();
		while (li.hasNext()) {
			((Layer) li.next()).kill();
		}
	}

	void reset() {
		final ListIterator li = layers.listIterator();
		while (li.hasNext()) {
			((Layer) li.next()).reset();
		}
	}
}
