/*
 * $Id:MandelbrotFractalRuntimeElement.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.fractal;

import net.sf.jame.core.config.ListConfigElementEvents;
import net.sf.jame.core.config.ListRuntimeElement;
import net.sf.jame.core.config.RuntimeElement;
import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.core.config.ValueConfigElementEvents;
import net.sf.jame.mandelbrot.fractal.incolouring.IncolouringFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.incolouring.IncolouringFormulaRuntimeElement;
import net.sf.jame.mandelbrot.fractal.outcolouring.OutcolouringFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.outcolouring.OutcolouringFormulaRuntimeFormula;
import net.sf.jame.mandelbrot.fractal.rendering.RenderingFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.rendering.RenderingFormulaRuntimeFormula;
import net.sf.jame.mandelbrot.fractal.transforming.TransformingFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.transforming.TransformingFormulaRuntimeFormula;

/**
 * @author Andrea Medeghini
 */
public class MandelbrotFractalRuntimeElement extends RuntimeElement {
	private IncolouringFormulaListElementListener incolouringFormulasListener;
	private OutcolouringFormulaListElementListener outcolouringFormulasListener;
	private RenderingFormulaElementListener renderingFormulaListener;
	private TransformingFormulaElementListener transformingFormulaListener;
	private MandelbrotFractalConfigElement fractalElement;
	private RenderingFormulaRuntimeFormula renderingFormula;
	private TransformingFormulaRuntimeFormula transformingFormula;
	private final ListRuntimeElement<IncolouringFormulaRuntimeElement> incolouringFormulaListElement = new ListRuntimeElement<IncolouringFormulaRuntimeElement>();
	private final ListRuntimeElement<OutcolouringFormulaRuntimeFormula> outcolouringFormulaListElement = new ListRuntimeElement<OutcolouringFormulaRuntimeFormula>();
	private boolean renderingFormulaChanged;
	private boolean transformingFormulaChanged;
	private boolean incolouringFormulaChanged;
	private boolean outcolouringFormulaChanged;

	/**
	 * 
	 */
	public MandelbrotFractalRuntimeElement(final MandelbrotFractalConfigElement fractalElement) {
		if (fractalElement == null) {
			throw new IllegalArgumentException("fractalElement is null");
		}
		this.fractalElement = fractalElement;
		createRenderingFormulaRuntime(fractalElement);
		renderingFormulaListener = new RenderingFormulaElementListener();
		fractalElement.getRenderingFormulaSingleElement().addChangeListener(renderingFormulaListener);
		createTransformingFormulaRuntime(fractalElement);
		transformingFormulaListener = new TransformingFormulaElementListener();
		fractalElement.getTransformingFormulaSingleElement().addChangeListener(transformingFormulaListener);
		createIncolouringFormulaRuntimes(fractalElement);
		incolouringFormulasListener = new IncolouringFormulaListElementListener();
		fractalElement.getIncolouringFormulaListElement().addChangeListener(incolouringFormulasListener);
		createOutcolouringFormulaRuntimes(fractalElement);
		outcolouringFormulasListener = new OutcolouringFormulaListElementListener();
		fractalElement.getOutcolouringFormulaListElement().addChangeListener(outcolouringFormulasListener);
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#dispose()
	 */
	@Override
	public void dispose() {
		if ((fractalElement != null) && (incolouringFormulasListener != null)) {
			fractalElement.getIncolouringFormulaListElement().removeChangeListener(incolouringFormulasListener);
		}
		incolouringFormulasListener = null;
		if ((fractalElement != null) && (outcolouringFormulasListener != null)) {
			fractalElement.getOutcolouringFormulaListElement().removeChangeListener(outcolouringFormulasListener);
		}
		outcolouringFormulasListener = null;
		if ((fractalElement != null) && (renderingFormulaListener != null)) {
			fractalElement.getRenderingFormulaSingleElement().removeChangeListener(renderingFormulaListener);
		}
		renderingFormulaListener = null;
		if ((fractalElement != null) && (transformingFormulaListener != null)) {
			fractalElement.getTransformingFormulaSingleElement().removeChangeListener(transformingFormulaListener);
		}
		transformingFormulaListener = null;
		fractalElement = null;
		if (renderingFormula != null) {
			renderingFormula.dispose();
			renderingFormula = null;
		}
		if (transformingFormula != null) {
			transformingFormula.dispose();
			transformingFormula = null;
		}
		incolouringFormulaListElement.dispose();
		outcolouringFormulaListElement.dispose();
		super.dispose();
	}

	private void createRenderingFormulaRuntime(final MandelbrotFractalConfigElement fractalElement) {
		final RenderingFormulaConfigElement renderingFormulaElement = fractalElement.getRenderingFormulaConfigElement();
		if (renderingFormulaElement != null) {
			final RenderingFormulaRuntimeFormula renderingFormula = new RenderingFormulaRuntimeFormula(renderingFormulaElement);
			setRenderingFormula(renderingFormula);
		}
	}

	private void createTransformingFormulaRuntime(final MandelbrotFractalConfigElement fractalElement) {
		final TransformingFormulaConfigElement transformingFormulaElement = fractalElement.getTransformingFormulaConfigElement();
		if (transformingFormulaElement != null) {
			final TransformingFormulaRuntimeFormula transformingFormula = new TransformingFormulaRuntimeFormula(transformingFormulaElement);
			setTransformingFormula(transformingFormula);
		}
	}

	private void createIncolouringFormulaRuntimes(final MandelbrotFractalConfigElement fractalElement) {
		for (int i = 0; i < fractalElement.getIncolouringFormulaConfigElementCount(); i++) {
			final IncolouringFormulaConfigElement incolouringFormulaElement = fractalElement.getIncolouringFormulaConfigElement(i);
			if (incolouringFormulaElement != null) {
				final IncolouringFormulaRuntimeElement incolouringFormula = new IncolouringFormulaRuntimeElement(incolouringFormulaElement);
				appendIncolouringFormula(incolouringFormula);
			}
		}
	}

	private void createOutcolouringFormulaRuntimes(final MandelbrotFractalConfigElement fractalElement) {
		for (int i = 0; i < fractalElement.getOutcolouringFormulaConfigElementCount(); i++) {
			final OutcolouringFormulaConfigElement outcolouringFormulaElement = fractalElement.getOutcolouringFormulaConfigElement(i);
			if (outcolouringFormulaElement != null) {
				final OutcolouringFormulaRuntimeFormula outcolouringFormula = new OutcolouringFormulaRuntimeFormula(outcolouringFormulaElement);
				appendOutcolouringFormula(outcolouringFormula);
			}
		}
	}

	/**
	 * Returns the renderingFormula.
	 * 
	 * @return the renderingFormula.
	 */
	public RenderingFormulaRuntimeFormula getRenderingFormula() {
		return renderingFormula;
	}

	private void setRenderingFormula(final RenderingFormulaRuntimeFormula renderingFormula) {
		if (this.renderingFormula != null) {
			this.renderingFormula.dispose();
		}
		this.renderingFormula = renderingFormula;
		renderingFormulaChanged = true;
	}

	/**
	 * Returns the transformingFormula.
	 * 
	 * @return the transformingFormula.
	 */
	public TransformingFormulaRuntimeFormula getTransformingFormula() {
		return transformingFormula;
	}

	private void setTransformingFormula(final TransformingFormulaRuntimeFormula transformingFormula) {
		if (this.transformingFormula != null) {
			this.transformingFormula.dispose();
		}
		this.transformingFormula = transformingFormula;
		transformingFormulaChanged = true;
	}

	/**
	 * Returns an incolouring formula.
	 * 
	 * @param index the formula index.
	 * @return the incolouring formula.
	 */
	public IncolouringFormulaRuntimeElement getIncolouringFormula(final int index) {
		return incolouringFormulaListElement.getElement(index);
	}

	/**
	 * Returns an incolouring formula index.
	 * 
	 * @param formulaElement the incolouring formula.
	 * @return the index.
	 */
	public int indexOfIncolouringFormula(final IncolouringFormulaRuntimeElement formulaElement) {
		return incolouringFormulaListElement.indexOfElement(formulaElement);
	}

	/**
	 * Returns the number of incolouring formulas.
	 * 
	 * @return the number of incolouring formulas.
	 */
	public int getIncolouringFormulaCount() {
		return incolouringFormulaListElement.getElementCount();
	}

	private void appendIncolouringFormula(final IncolouringFormulaRuntimeElement formula) {
		incolouringFormulaListElement.appendElement(formula);
	}

	private void insertIncolouringFormulaAfter(final int index, final IncolouringFormulaRuntimeElement formula) {
		incolouringFormulaListElement.insertElementAfter(index, formula);
	}

	private void insertIncolouringFormulaBefore(final int index, final IncolouringFormulaRuntimeElement formula) {
		incolouringFormulaListElement.insertElementBefore(index, formula);
	}

	private void removeIncolouringFormula(final int index) {
		incolouringFormulaListElement.getElement(index).dispose();
		incolouringFormulaListElement.removeElement(index);
	}

	/**
	 * Returns an outcolouring formula.
	 * 
	 * @param index the formula index.
	 * @return the outcolouring formula.
	 */
	public OutcolouringFormulaRuntimeFormula getOutcolouringFormula(final int index) {
		return outcolouringFormulaListElement.getElement(index);
	}

	/**
	 * Returns an outcolouring formula index.
	 * 
	 * @param formula the outcolouring formula.
	 * @return the index.
	 */
	public int indexOfOutcolouringFormula(final OutcolouringFormulaRuntimeFormula formula) {
		return outcolouringFormulaListElement.indexOfElement(formula);
	}

	/**
	 * Returns the number of outcolouring formulas.
	 * 
	 * @return the number of outcolouring formulas.
	 */
	public int getOutcolouringFormulaCount() {
		return outcolouringFormulaListElement.getElementCount();
	}

	private void appendOutcolouringFormula(final OutcolouringFormulaRuntimeFormula formula) {
		outcolouringFormulaListElement.appendElement(formula);
	}

	private void insertOutcolouringFormulaAfter(final int index, final OutcolouringFormulaRuntimeFormula formula) {
		outcolouringFormulaListElement.insertElementAfter(index, formula);
	}

	private void insertOutcolouringFormulaBefore(final int index, final OutcolouringFormulaRuntimeFormula formula) {
		outcolouringFormulaListElement.insertElementBefore(index, formula);
	}

	private void removeOutcolouringFormula(final int index) {
		outcolouringFormulaListElement.getElement(index).dispose();
		outcolouringFormulaListElement.removeElement(index);
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#isChanged()
	 */
	@Override
	public boolean isChanged() {
		boolean fractalChanged = false;
		if (renderingFormula.isChanged()) {
			renderingFormulaChanged = true;
			fractalChanged = true;
		}
		if (transformingFormula.isChanged()) {
			transformingFormulaChanged = true;
			fractalChanged = true;
		}
		if (incolouringFormulaListElement.isChanged()) {
			incolouringFormulaChanged = true;
			fractalChanged = true;
		}
		if (outcolouringFormulaListElement.isChanged()) {
			outcolouringFormulaChanged = true;
			fractalChanged = true;
		}
		return super.isChanged() || fractalChanged;
	}

	/**
	 * @return the incolouringFormulaChanged
	 */
	public boolean isIncolouringFormulaChanged() {
		final boolean value = incolouringFormulaChanged;
		incolouringFormulaChanged = false;
		return value;
	}

	/**
	 * @return the outcolouringFormulaChanged
	 */
	public boolean isOutcolouringFormulaChanged() {
		final boolean value = outcolouringFormulaChanged;
		outcolouringFormulaChanged = false;
		return value;
	}

	/**
	 * @return the renderingFormulaChanged
	 */
	public boolean isRenderingFormulaChanged() {
		final boolean value = renderingFormulaChanged;
		renderingFormulaChanged = false;
		return value;
	}

	/**
	 * @return the transformingFormulaChanged
	 */
	public boolean isTransformingFormulaChanged() {
		final boolean value = transformingFormulaChanged;
		transformingFormulaChanged = false;
		return value;
	}

	private class IncolouringFormulaListElementListener implements ValueChangeListener, ListConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case ELEMENT_ADDED: {
					appendIncolouringFormula(new IncolouringFormulaRuntimeElement((IncolouringFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					incolouringFormulaChanged = true;
					break;
				}
				case ELEMENT_INSERTED_AFTER: {
					insertIncolouringFormulaAfter(((Integer) e.getParams()[1]).intValue(), new IncolouringFormulaRuntimeElement((IncolouringFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					incolouringFormulaChanged = true;
					break;
				}
				case ELEMENT_INSERTED_BEFORE: {
					insertIncolouringFormulaBefore(((Integer) e.getParams()[1]).intValue(), new IncolouringFormulaRuntimeElement((IncolouringFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					incolouringFormulaChanged = true;
					break;
				}
				case ELEMENT_REMOVED: {
					removeIncolouringFormula(((Integer) e.getParams()[1]).intValue());
					fireChanged();
					incolouringFormulaChanged = true;
					break;
				}
				default: {
					break;
				}
			}
		}
	}

	private class OutcolouringFormulaListElementListener implements ValueChangeListener, ListConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case ELEMENT_ADDED: {
					appendOutcolouringFormula(new OutcolouringFormulaRuntimeFormula((OutcolouringFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					outcolouringFormulaChanged = false;
					break;
				}
				case ELEMENT_INSERTED_AFTER: {
					insertOutcolouringFormulaAfter(((Integer) e.getParams()[1]).intValue(), new OutcolouringFormulaRuntimeFormula((OutcolouringFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					outcolouringFormulaChanged = false;
					break;
				}
				case ELEMENT_INSERTED_BEFORE: {
					insertOutcolouringFormulaBefore(((Integer) e.getParams()[1]).intValue(), new OutcolouringFormulaRuntimeFormula((OutcolouringFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					outcolouringFormulaChanged = false;
					break;
				}
				case ELEMENT_REMOVED: {
					removeOutcolouringFormula(((Integer) e.getParams()[1]).intValue());
					fireChanged();
					outcolouringFormulaChanged = false;
					break;
				}
				default: {
					break;
				}
			}
		}
	}

	private class RenderingFormulaElementListener implements ValueChangeListener, ValueConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case VALUE_CHANGED: {
					setRenderingFormula(new RenderingFormulaRuntimeFormula((RenderingFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					renderingFormulaChanged = true;
					break;
				}
				default: {
					break;
				}
			}
		}
	}

	private class TransformingFormulaElementListener implements ValueChangeListener, ValueConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case VALUE_CHANGED: {
					setTransformingFormula(new TransformingFormulaRuntimeFormula((TransformingFormulaConfigElement) e.getParams()[0]));
					fireChanged();
					transformingFormulaChanged = true;
					break;
				}
				default: {
					break;
				}
			}
		}
	}
}
