/*
 * $Id:ColorRendererRuntimeElement.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.color;

import net.sf.jame.core.config.RuntimeElement;
import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.core.extension.ConfigurableExtensionReference;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.extension.ExtensionNotFoundException;
import net.sf.jame.mandelbrot.MandelbrotRegistry;
import net.sf.jame.mandelbrot.color.extension.ColorRendererExtensionConfig;
import net.sf.jame.mandelbrot.color.extension.ColorRendererExtensionRuntime;
import net.sf.jame.twister.util.ExtensionConfigElementEvents;

/**
 * @author Andrea Medeghini
 */
public class ColorRendererRuntimeElement extends RuntimeElement {
	private ColorRendererExtensionRuntime<?> rendererRuntime;
	private ColorRendererConfigElement rendererElement;
	private ExtensionListener extensionListener;

	/**
	 * Constructs a new renderer element.
	 * 
	 * @param registry
	 * @param rendererElement
	 */
	public ColorRendererRuntimeElement(final ColorRendererConfigElement rendererElement) {
		if (rendererElement == null) {
			throw new IllegalArgumentException("rendererElement is null");
		}
		this.rendererElement = rendererElement;
		createRuntime(rendererElement.getReference());
		extensionListener = new ExtensionListener();
		rendererElement.getExtensionElement().addChangeListener(extensionListener);
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#dispose()
	 */
	@Override
	public void dispose() {
		if ((rendererElement != null) && (extensionListener != null)) {
			rendererElement.getExtensionElement().removeChangeListener(extensionListener);
		}
		extensionListener = null;
		if (rendererRuntime != null) {
			rendererRuntime.dispose();
			rendererRuntime = null;
		}
		rendererElement = null;
		super.dispose();
	}

	@SuppressWarnings("unchecked")
	private void createRuntime(final ConfigurableExtensionReference<ColorRendererExtensionConfig> reference) {
		try {
			if (reference != null) {
				final ColorRendererExtensionRuntime rendererRuntime = MandelbrotRegistry.getInstance().getColorRendererExtension(reference.getExtensionId()).createExtensionRuntime();
				final ColorRendererExtensionConfig rendererConfig = reference.getExtensionConfig();
				rendererRuntime.setConfig(rendererConfig);
				setRendererRuntime(rendererRuntime);
			}
			else {
				setRendererRuntime(null);
			}
		}
		catch (final ExtensionNotFoundException e) {
			e.printStackTrace();
		}
		catch (final ExtensionException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @return the rendererRuntime
	 */
	public ColorRendererExtensionRuntime<?> getRendererRuntime() {
		return rendererRuntime;
	}

	private void setRendererRuntime(final ColorRendererExtensionRuntime<?> rendererRuntime) {
		if (this.rendererRuntime != null) {
			this.rendererRuntime.dispose();
		}
		this.rendererRuntime = rendererRuntime;
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#isChanged()
	 */
	@Override
	public boolean isChanged() {
		final boolean rendererChanged = (rendererRuntime != null) && rendererRuntime.isChanged();
		return super.isChanged() || rendererChanged;
	}

	private class ExtensionListener implements ValueChangeListener, ExtensionConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		@SuppressWarnings("unchecked")
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case EXTENSION_REFERENCE_CHANGED: {
					createRuntime((ConfigurableExtensionReference<ColorRendererExtensionConfig>) e.getParams()[0]);
					fireChanged();
					break;
				}
				default: {
					break;
				}
			}
		}
	}
}
