package zinger.io.file;

import java.io.*;
import java.util.*;
import java.util.prefs.*;

public class FileWatcher extends Thread
{
    protected final Set listeners = new HashSet(1);

    private final Set fileSet = Collections.synchronizedSet(new HashSet());
    private File[] files;
	
	private Preferences userPref;
    private long delay;

    private boolean keepRunning = true;

    public FileWatcher()
    {
		this(true);
    }
	
	public FileWatcher(boolean persistPreferences)
	{
		this.delay = 1000L;
		if(persistPreferences)
		{
			this.userPref = Preferences.userNodeForPackage(this.getClass());
			this.delay = this.userPref.getLong("delay", this.delay);
		}
        this.start();
	}

    public void setDelay(final long delay)
    {
		this.delay = delay;
		if(this.userPref != null)
			this.userPref.putLong("delay", this.delay);
    }

    public long getDelay()
    {
        return this.delay;
    }

    public void addFiles(final File[] files)
    {
        for(int i = 0; i < files.length; ++i)
        {
            this.fileSet.add(files[i]);
        }
        this.files = null;
    }

    public void removeFiles(final File[] files)
    {
        for(int i = 0; i < files.length; ++i)
        {
            this.fileSet.remove(files[i]);
        }
        this.files = null;
    }

    public void addFile(final File file)
    {
        this.fileSet.add(file);
        this.files = null;
    }

    public void removeFile(final File file)
    {
        this.fileSet.remove(file);
        this.files = null;
    }

    public void addFileListener(final FileListener listener)
    {
        synchronized(this.listeners)
        {
            this.listeners.add(listener);
        }
    }

    public void removeFileListener(final FileListener listener)
    {
        synchronized(this.listeners)
        {
            this.listeners.remove(listener);
        }
    }

    protected void notifyFileListeners(final FileEvent[] events)
    {
        synchronized(this.listeners)
        {
            FileListener fl;
            for(final Iterator iterator = this.listeners.iterator(); iterator.hasNext();)
            {
                fl = (FileListener)iterator.next();
                for(int i = 0; i < events.length; i++)
                {
                    if(events[i] == null)
                    {
                        continue; // null elements in the events array allowed for quicker state analysis
                    }
                    fl.handleFileEvent(events[i]);
                }
            }
        }
    }

    public final void run()
    {
        long[] lastModifiedDates = null;
        long lastModified;
        long[] lengths = null;
        long length;
        FileEvent[] events = null;
        boolean changes;
        boolean fileSetChanged;
        while(keepRunning)
        {
            if(this.files == null)
            {
                this.files = (File[])this.fileSet.toArray(new File[this.fileSet.size()]);
                lastModifiedDates = null;
                lengths = null;
                events = null;
                fileSetChanged = true;
            }
            else
            {
                fileSetChanged = false;
            }
            if(events == null)
            {
                events = new FileEvent[this.files.length];
			}
			if(lastModifiedDates == null)
			{
                lastModifiedDates = new long[this.files.length];
			}
			if(lengths == null)
			{
                lengths = new long[this.files.length];
			}
            changes = false;
            for(int i = 0; i < this.files.length; i++)
            {
                events[i] = null;
                lastModified = this.files[i].lastModified();
                length = this.files[i].length();
                if(this.files[i].exists())
                {
                    if(lastModifiedDates[i] <= 0 && !fileSetChanged)
                    {
                        events[i] = new FileEvent(this, this.files[i], FileEvent.CREATED);
                    }
                    else if(lastModifiedDates[i] != lastModified || lengths[i] != length)
                    {
                        events[i] = new FileEvent(this, this.files[i], FileEvent.MODIFIED);
                    }
                }
                else if(lastModifiedDates[i] >0)
                {
                    events[i] = new FileEvent(this, this.files[i], FileEvent.DELETED);
                }
                lastModifiedDates[i] = lastModified;
                lengths[i] = length;
                if(events[i] != null)
                {
                    changes = true;
                }
            }
            if(changes)
            {
                this.notifyFileListeners(events);
            }
            try
            {
                Thread.sleep(this.delay);
            }
            catch(final InterruptedException ex)
            {
                ex.printStackTrace();
                break;
            }
            catch(final IllegalArgumentException ex)
            {
                ex.printStackTrace();
                break;
            }
        }
    }

    protected void finalize() throws Throwable
    {
        this.keepRunning = false;
        super.finalize();
    }
}
