/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.util;

import org.ckkloverdos.log.StdLog;
import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.*;

/**
 * An extensible classloader that delegates class loading to a list of other class loaders.
 * @author Christos KK Loverdos
 */
public class CompositeClassLoader extends ClassLoader implements IToStringAware
{
    private List loaders = new ArrayList();

    /**
     * Constructs a new instance with the given parent.
     * @param parent
     */
    public CompositeClassLoader(ClassLoader parent)
    {
        super(parent);
    }

    /**
     * Constructs a new instance with the context class loader as the parent.
     */
    public CompositeClassLoader()
    {
        this(ClassUtil.getContextClassLoader());
    }

    /**
     * Constructs a new instance with the given list of class loaders.
     * @param loaders
     */
    public CompositeClassLoader(List loaders)
    {
        this();
        this.loaders.addAll(loaders);
    }


    /**
     * Constructs a new instance with the given array of class loaders.
     * @param loaders
     */
    public CompositeClassLoader(ClassLoader[] loaders)
    {
        this();
        this.loaders.addAll(Arrays.asList(loaders));
    }

    /**
     * Adds the given class loader to the end of the internal list of class loaders.
     * @param l
     */
    public void add(ClassLoader l)
    {
        if(!loaders.contains(l))
        {
            loaders.add(l);
        }
    }

    /**
     * Creates a new {@link java.net.URLClassLoader} and adds it to the end of the internal
     * list of class loaders.
     * @param url
     */
    public void add(URL url)
    {
        add(new URL[]{url});
    }

    /**
     * Creates a new {@link java.net.URLClassLoader} and adds it to the end of the internal
     * list of class loaders.
     * @param urls
     */
    public void add(URL[] urls)
    {
        add(new URLClassLoader(urls));
    }


    /**
     * Creates a new {@link java.net.URLClassLoader} and adds it to the end of the internal
     * list of class loaders. The URIs are first translated to URLs.
     *
     * @param uris
     */
    public void add(URI[] uris) throws MalformedURLException
    {
        uris = (URI[]) Util.safe(uris, URI.class);
        URL[] urls = new URL[uris.length];
        for(int i = 0; i < uris.length; i++)
        {
            urls[i] = uris[i].toURL();
        }
        add(urls);
    }
    /**
     * Creates a new {@link java.net.URLClassLoader} and adds it to the end of the internal
     * list of class loaders.
     * @param uri
     */
    public void add(URI uri) throws MalformedURLException
    {
            add(uri.toURL());
    }

    public void add(File f)
    {
        try
        {
            add(f.toURI().toURL());
        }
        catch(MalformedURLException e)
        {
            StdLog.error(e);
        }
    }

    public void addFile(String filename)
    {
        add(new File(filename));
    }

    public boolean remove(ClassLoader l)
    {
        return loaders.remove(l);
    }

    public ClassLoader[] getClassLoaders()
    {
        return (ClassLoader[]) loaders.toArray(new ClassLoader[loaders.size()]);
    }

    /**
     * Sets this class loader as the {@link Thread#setContextClassLoader(ClassLoader) context} class loader.
     */
    public void setAsContextClassLoader()
    {
        Thread.currentThread().setContextClassLoader(this);
    }

    protected Class findClass(String s) throws ClassNotFoundException
    {
        for(Iterator iLoaders = loaders.iterator(); iLoaders.hasNext();)
        {
            ClassLoader cl = (ClassLoader) iLoaders.next();
            try
            {
                return cl.loadClass(s);
            }
            catch(ClassNotFoundException e)
            {
                StdLog.warn(e.getMessage());
            }
        }

        return findSystemClass(s);
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ToString ts2 = ts.save();
        ts
            .setUsingTypeNames(false)
            .add(loaders)
            .restore(ts2);
    }
}
