/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.filter;

import org.ckkloverdos.string.ICharSequence;

import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * An implementation of {@link org.ckkloverdos.filter.IFilter} for which the filtering
 * is based on a regular expression.
 * This assumes that the client code always passes an implementation of either
 * {@link CharSequence} or {@link org.ckkloverdos.string.ICharSequence} as the first
 * paramater to {@link #accept(Object, Object)}, otherwise
 * 
 * @author Christos KK Loverdos
 */
public class REFilter implements IFilter
{
    public static final Object HINT_FIND  = new Object();
    public static final Object HINT_MATCH = new Object();

    private Matcher matcher;

    public REFilter(String pattern)
	{
		this(pattern, 0);
	}
	
	public REFilter(String pattern, int flags)
	{
		this(Pattern.compile(pattern, flags));
	}
	
	public REFilter(Pattern pattern)
	{
        this.matcher = pattern.matcher("");
    }

    /**
     * Tests if <code>object</code> can be {@link java.util.regex.Matcher#find() found} or
     * {@link java.util.regex.Matcher#matches() matched} according to the regular expression
     * provided in one of the constructors.
     *
     * If <code>hints</code> is {@link #HINT_MATCH} then {@link java.util.regex.Matcher#matches()}
     * is used, otherwise {@link java.util.regex.Matcher#find()} is used.
     * 
     * @param object
     * @param hints
     */
    public boolean accept(Object object, Object hints)
	{
        CharSequence cs;
        if(object instanceof ICharSequence)
        {
            cs = ((ICharSequence) object).getSequence();
        }
        else
        {
            cs = (CharSequence) object;
        }

        matcher.reset(cs);

        if(HINT_MATCH == hints)
        {
            return matcher.matches();
        }
        return matcher.find();
	}
}
