/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.collection;

import org.ckkloverdos.log.StdLog;
import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

/**
 * Fast dynamic array of <code>int</code> primitives.
 * 
 * @author Christos KK Loverdos
 */
public final class IntArray implements IToStringAware
{
    private int[] array;
    private int nextPosition;

    public IntArray()
    {
        this.array = new int[10];
        this.nextPosition = 0;
    }
    
    public IntArray(int initialCapacity)
    {
        initialCapacity = (initialCapacity <= 0 ? 10 : initialCapacity);
        this.array = new int[initialCapacity];
        this.nextPosition = 0;
    }

    public IntArray(int[] array)
    {
        this.array = array;
        this.nextPosition = this.array.length;
    }

    public void add(int i)
    {
        if(nextPosition == array.length)
        {
            int[] newArray = new int[((nextPosition << 1) + nextPosition) >>> 1];
            System.arraycopy(array, 0, newArray, 0, array.length);
            array = newArray;
        }
        array[nextPosition++] = i;
    }

    public int length()
    {
        return nextPosition;
    }

    public void removeByIndex(int index)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        System.arraycopy(array, index + 1, array, index, nextPosition - index - 1);
        nextPosition--;
    }

    /**
     * Returns the underlying <code>int</code> array. Note that you need to
     * consult {@link #length()} in order to get the actual length of the
     * array, since the underlying storage may be bigger than expected.
     */
    public int[] array()
    {
        return array;
    }

    /**
     * Ensures that the underlying storage has at least the given <code>length</code>.
     * @param length
     */
    public void ensureCapacity(int length)
    {
        if(length > array.length)
        {
            int[] newArray = new int[length];
            System.arraycopy(array, 0, newArray, 0, array.length);
            array = newArray;
        }
    }

    public void set(int index, int value)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        array[index] = value;
    }

    public void increase(int index)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        array[index]++;
    }

    public void decrease(int index)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        array[index]--;
    }

    public void increaseBy(int index, int value)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        array[index] += value;
    }

    public void decreaseBy(int index, int value)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        array[index] -= value;
    }

    public int get(int index)
    {
        if(index < 0 || index >= nextPosition)
        {
            throw new ArrayIndexOutOfBoundsException("Bad index " + index);
        }
        return array[index];
    }

    public int first()
    {
        return array[0];
    }

    public int last()
    {
        return array[nextPosition - 1];
    }

    public void push(int i)
    {
        if(nextPosition == array.length)
        {
            int[] newArray = new int[((nextPosition << 1) + nextPosition) >>> 1];
            System.arraycopy(array, 0, newArray, 0, array.length);
            array = newArray;
        }
        array[nextPosition++] = i;
    }

    public int peek()
    {
        return array[nextPosition - 1];
    }

    public int pop()
    {
        int i = array[nextPosition - 1];
        nextPosition--;
        return i;
    }

    public void compact()
    {
        if(nextPosition < array.length)
        {
            int[] newArray = new int[nextPosition];
            System.arraycopy(array, 0, newArray, 0, nextPosition);
            array = newArray;
        }
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        for(int i=0; i<nextPosition; i++)
        {
            ts.add("" + i, array[i]);
        }
    }

    public static void main(String[] args)
    {
        IntArray a = new IntArray(0);
        a.add(0);
        a.add(1);
        a.add(2);
        StdLog.log("length = " + a.length() + ", a = " + a);
        a.removeByIndex(2);
        StdLog.log("length = " + a.length() + ", a = " + a);
        a.push(3);
        StdLog.log("length = " + a.length() + ", a = " + a);
        StdLog.log("a.first() = " + a.first());
        StdLog.log("a.last() = " + a.last());
    }
}
