/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.beans;

import org.ckkloverdos.type.java.JavaType;
import org.ckkloverdos.type.java.JavaTypeRegistry;
import org.ckkloverdos.type.java.NameAndType;
import org.ckkloverdos.log.StdLog;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.util.HashMap;
import java.util.Map;

/**
 * A generic representation of a JavaBean property.
 * It provides utility methods to retrieve its accessor, either
 * as property names ({@link #getPropertyAccessor(boolean)}) or as getter method calls ({@link #getReadMethodAccesor(boolean)}).
 *
 * <p> This is useful in a scripting or code generation environment.
 * 
 * @author Christos KK Loverdos
 */
public class JavaBeanProperty
{
    private String name;
    private Class c;
    private JavaType jt;
    private boolean loaded;
    private Map properties;
    private JavaBeanProperty parent;
    private PropertyDescriptor pd;
    private JavaTypeRegistry typeRegistry;

    /**
     * Constructs a <code>JavaBeanProperty</code> for the given class <code>c</code>.
     * The name of the property is the empty string.
     * @param c
     */
    public JavaBeanProperty(Class c)
    {
        this(c, null);
    }

    /**
     * Constructs a <code>JavaBeanProperty</code> for the given class <code>c</code>,
     * using the <code>typeRegistry</code> to resolve {@link  JavaType types}.
     * The name of the property is the empty string.
     * @param c
     * @param typeRegistry
     */
    public JavaBeanProperty(Class c, JavaTypeRegistry typeRegistry)
    {
        this("", c, typeRegistry);
    }

    /**
     * Constructs a <code>JavaBeanProperty</code> for the given class <code>c</code>,
     * using the <code>typeRegistry</code> to resolve {@link  JavaType types} and
     * <code>name</code> as the property name.
     * @param name
     * @param c
     * @param typeRegistry
     */
    public JavaBeanProperty(String name, Class c, JavaTypeRegistry typeRegistry)
    {
        this(name, c, null, null, typeRegistry);
    }
    
    private JavaBeanProperty(String name, Class c, JavaBeanProperty parent, PropertyDescriptor pd, JavaTypeRegistry typeRegistry)
    {
        this.name = name;
        this.c = c;
        this.properties = new HashMap();
        this.parent = parent;
        this.pd = pd;
        this.typeRegistry = typeRegistry;
    }

    /**
     * Returns the {@link org.ckkloverdos.type.java.JavaTypeRegistry} given in
     * {@link #JavaBeanProperty(Class, org.ckkloverdos.type.java.JavaTypeRegistry) this}
     * or the {@link #JavaBeanProperty(String, Class, org.ckkloverdos.type.java.JavaTypeRegistry) other}
     * constructor.
     *
     */
    public JavaTypeRegistry getTypeRegistry()
    {
        return typeRegistry;
    }

    private void load()
    {
        if(loaded)
        {
            return;
        }
        else
        {
            try
            {
                BeanInfo bi = Introspector.getBeanInfo(c);
                PropertyDescriptor[] pds = bi.getPropertyDescriptors();

                for(int i = 0; i < pds.length; i++)
                {
                    PropertyDescriptor pd = pds[i];
                    String pname = pd.getName();
                    Class ptype = pd.getPropertyType();
                    JavaBeanProperty p = new JavaBeanProperty(pname, ptype, this, pd, typeRegistry);
                    properties.put(pname, p);
                }

                loaded = true;
            }
            catch(IntrospectionException e)
            {
                StdLog.error(e);
            }
        }
    }

    /**
     * Returns the sub-properties.
     */
    public JavaBeanProperty[] getProperties()
    {
        return (JavaBeanProperty[]) properties.values().toArray(new JavaBeanProperty[0]);
    }
    
    /**
     * Returns the parent property.
     */
    public JavaBeanProperty getParent()
    {
        return parent;
    }

    /**
     * Returns the non-qualified name of this property.
     */
    public String getName()
    {
        return name;
    }

    /**
     * Returns the type of this property.
     */
    public JavaType getType()
    {
        if(null != typeRegistry)
        {
            return typeRegistry.getByClass(c);
        }
        if(null == jt)
        {
            jt = new JavaType(c);
        }
        return jt;
    }
    
    public NameAndType getNameType()
    {
        return new NameAndType(getName(), getType());
    }

    public NameAndType getFullNameType()
    {
        return getFullNameType(true);
    }
    
    public NameAndType getFullNameType(boolean includeTopParent)
    {
        return new NameAndType(getFullName(includeTopParent), getType());
    }

    /**
     * Returns the fully qualified name of this property. If this instance has been
     * created from a root property using successive {@link #getProperty(String)}
     * calls, then the whole property hierarchy will appear in the full name.
     */
    public String getFullName()
    {
        return getFullName(true);
    }

    /**
     * Returns the fully qualified name of this property, with the option to include
     * the top parent name in it.
     * 
     * @param includeTopParent
     */
    public String getFullName(boolean includeTopParent)
    {
        String name = this.name;
        JavaBeanProperty parent = this.parent;
        while(null != parent)
        {
            if(parent.name.length() > 0)
            {
                if(includeTopParent || null != parent.parent)
                {
                    name = parent.name + "." + name;
                }
            }
            parent = parent.parent;
        }
        return name;
    }

    /**
     * Returns the java class of this property.
     */
    public Class getJavaClass()
    {
        return c;
    }

    /**
     * Returns the method name used to read this property.
     */
    public String getReadMethodName()
    {
        return pd.getReadMethod().getName();
    }

    /**
     * Returns <code>true</code> iff <code>name</code> is a sub-property of the property represented by this instance.
     * @param name
     */
    public boolean hasProperty(String name)
    {
        load();
        return null != getProperty(name);
    }

    /**
     * Returns an accessor for this property, based on direct property access. This method could be useful in scripting.
     * @param fullyQualified
     */
    public String getPropertyAccessor(boolean fullyQualified)
    {
        return getPropertyAccessor(fullyQualified, true);
    }

    /**
     * Returns an accessor for this property, based on direct property access. This method could be useful in scripting.
     * @param fullyQualified
     * @param includeTopParent
     */
    public String getPropertyAccessor(boolean fullyQualified, boolean includeTopParent)
    {
        if(fullyQualified)
        {
            return getFullName(includeTopParent);
        }
        return getName();
    }

    /**
     * Returns an accessor for this property, based on read method calls. This method could be useful in scripting.
     * @param fullyQualified
     */
    public String getReadMethodAccesor(boolean fullyQualified)
    {
        return getReadMethodAccesor(fullyQualified, true);
    }

    /**
     * Returns an accessor for this property, based on read method calls. This method could be useful in scripting.
     * @param fullyQualified
     * @param includeTopParent
     */
    public String getReadMethodAccesor(boolean fullyQualified, boolean includeTopParent)
    {
        String accessor = getReadMethodName() + "()";
        JavaBeanProperty parent = this.parent;        
        if(fullyQualified)
        {
            String pname = parent.name;
            if(pname.length() > 0)
            {
                if(null == parent.parent && includeTopParent)
                {
                    accessor = pname + "." + accessor;
                }
                else
                {
                    accessor = parent.getReadMethodAccesor(fullyQualified) + "." + accessor;
                }
            }
         }
        return accessor;
    }

    /**
     * Returns the <code>name</code>d sub-property of the property represented by this instance.
     * @param name
     */
    public JavaBeanProperty getProperty(String name)
    {
        load();

        if(properties.containsKey(name))
        {
            return (JavaBeanProperty) properties.get(name);
        }
        else if(-1 == name.indexOf('.'))
        {
            return null;
        }
        return getNestedProperty(name);
    }

    private JavaBeanProperty getNestedProperty(String name)
    {
        load();

        String[] parts = name.split("\\.");
        String partialNestedName = "";
        
        JavaBeanProperty jbp = this;
        for(int i = 0; i < parts.length; i++)
        {
            String part = parts[i];
            if(!jbp.hasProperty(part))
            {
                return null;
            }

            JavaBeanProperty jbppart = (JavaBeanProperty) jbp.properties.get(part);
            if("".equals(partialNestedName))
            {
                partialNestedName = part;
            }
            else
            {
                partialNestedName += "." + part;
            }

            // cache fully qualified names
            this.properties.put(partialNestedName, new JavaBeanProperty(jbppart.name, jbppart.c, jbp, jbppart.pd, jbppart.typeRegistry));
            jbp = jbppart;
        }

        return jbp;
    }

    public String toString()
    {
        String repr = getFullName();
        if(repr.length() > name.length())
        {
            int pos = repr.lastIndexOf(name);
            repr = "[" + repr.substring(0, pos) + "]" + repr.substring(pos);
        }
        String par = null == parent ? "" : parent.getFullName();

        return repr + ": " + c.getName() + ", parent = " + par;
    }
}
