program MastrMnd;

uses
    Crt, MannDoor, mCrt, mStrings, mUtils, SysUtils, Windows;

const
  PegColours: Array[0..2] of Byte = (DARKGRAY, LIGHTBLUE, WHITE); {Blank, Right Colour Right Place, Right Colour Wrong Place}
  PieceColours: Array[0..6] of Byte =
  (DARKGRAY, LIGHTGREEN, LIGHTCYAN, LIGHTRED, LIGHTMAGENTA, YELLOW, WHITE);{Blank, 6 Colours}
  ProgVer = 'Mastermind v1.00';

type
  TPegs = Array[1..4] of Byte;
  TPieces = Array[1..4] of Byte;

  TLine = record
    Peg: TPegs;
    Piece: TPieces;
  end;

var
   Answer: TPieces;              {The 4 colours used in the answer}
   Colour: Byte;                 {The current colour (1..6)}
   CurLine, CurPiece: Byte;      {Current position on board}
   GameOver: Boolean;            {Did we win or lose?}
   Lines: Array[1..10] of TLine; {Our guess history}

procedure DrawAnswer; forward;
procedure RedrawScreen; forward;

{ Update the message line with a new MSG }
procedure ChangeMsg(Msg: String);
begin
     mGotoXY(26, 22);
     mTextAttr(31);
     mWrite(PadRight(Msg, ' ', 54));
end;

{ Check TEMPLINE against the ANSWER, looking for right colour in right spot
  Return CNT which is the number of black pegs found }
function CheckBlackPegs(var TempLine: TLine): Byte;
var
   I, Cnt: Integer;
begin
     Cnt := 0;
     for I := 1 to 4 do
     begin
          if (TempLine.Piece[I] = Answer[I]) then
          begin
               Inc(Cnt);
               TempLine.Piece[I] := 0;
               TempLine.Peg[Cnt] := 1;
          end;
     end;
     CheckBlackPegs := Cnt;
end;

{ Check to see if we ran out of guesses }
procedure CheckLost;
begin
     if (CurLine = 10) then
     begin
          GameOver := True;
          ChangeMsg('Game Over, You Lose!');
          DrawAnswer;
     end;
end;

{ Check to see that the current line is valid
  It isn't if there are duplicate or blank pieces }
function CheckValid: Boolean;
var
   I, J: Integer;
begin
     CheckValid := True;
     for I := 1 to 4 do
     begin
          if (Lines[CurLine].Piece[I] = 0) then
          begin
               ChangeMsg('You Must Pick Four Colours');
               CheckValid := False;
               Exit;
          end;
          for J := 1 to 4 do
          begin
               if (Lines[CurLine].Piece[I] = Lines[CurLine].Piece[J]) and (I <> J) then
               begin
                    ChangeMsg('You Must Pick Four Unique Colours');
                    CheckValid := False;
                    Exit;
               end;
          end;
     end;
end;


{ Check TEMPLINE against the ANSWER, looking for right colour in wrong spot
  Return CNT - BLACK which is the number of white pegs found }
function CheckWhitePegs(var TempLine: TLine): Byte;
var
   I, J, Black, Cnt: Integer;
begin
     Cnt := 4;
     for I := 1 to 4 do
     begin
          if (TempLine.Peg[I] = 0) then
          begin
               Cnt := I - 1;
               Break;
          end;
     end;

     Black := Cnt;
     if (Cnt < 4) then
     begin
          for I := 1 to 4 do
          begin
               for J := 1 to 4 do
               begin
                    if (TempLine.Piece[I] = Answer[J]) then
                    begin
                         Inc(Cnt);
                         TempLine.Piece[I] := 0;
                         TempLine.Peg[Cnt] := 2;
                    end;
               end;
          end;
     end;
     CheckWhitePegs := Cnt - Black;
end;

{ Check to see if all four pegs are black }
function CheckWon: Boolean;
var
   I: Integer;
   Res: Boolean;
begin
     Res := True;
     for I := 1 to 4 do
     begin
          if (Lines[CurLine].Peg[I] <> 1) then
             Res := False;
     end;
     if (Res) then
     begin
          GameOver := True;
          ChangeMsg('Congratulations, You Win!');
          DrawAnswer;
     end;
     CheckWon := Res;
end;

{ Draw the answer line }
procedure DrawAnswer;
var
   I: Integer;
begin
     for I := 1 to 4 do
     begin
          mGotoXY(4 + ((I - 1) * 4), 2);
          mTextAttr(PieceColours[Answer[I]]);
          mWrite(#219#219);
     end;
end;

{ Highlight the currently selected colour }
procedure DrawColour;
begin
     mGotoXY(32, 8 + Colour);
     mTextAttr(PieceColours[Colour] + (7 * 16));
     mWrite(#254#254);
end;

{ Draw the pegs for the current line }
procedure DrawPegs;
var
   I: Integer;
begin
     for I := 1 to 4 do
     begin
          mGotoXY(14 + (I - 1), 22 - ((CurLine - 1) * 2));
          mTextAttr(PegColours[Lines[CurLine].Peg[I]]);
          mWrite(#249);
     end;
end;

{ Highlight the currently selected piece }
procedure DrawPiece;
begin
     mGotoXY(3 + ((CurPiece - 1) * 2), 22 - ((CurLine - 1) * 2));
     mTextAttr(7);
     mWrite('[ ]');
     mGotoXY(4 + ((CurPiece - 1) * 2), 22 - ((CurLine - 1) * 2));
     mTextAttr(PieceColours[Lines[CurLine].Piece[CurPiece]]);
     mWrite(#254);
end;

{ Un-highlight the current colour }
procedure EraseColour;
begin
     mGotoXY(32, 8 + Colour);
     mTextAttr(PieceColours[Colour]);
     mWrite(#254#254);
end;

{ Un-highlight the current piece }
procedure ErasePiece;
begin
     mGotoXY(3 + ((CurPiece - 1) * 2), 22 - ((CurLine - 1) * 2));
     mTextAttr(0);
     mWrite('[ ]');
     mGotoXY(4 + ((CurPiece - 1) * 2), 22 - ((CurLine - 1) * 2));
     mTextAttr(PieceColours[Lines[CurLine].Piece[CurPiece]]);
     mWrite(#254);
end;

{ Generate a random ANSWER line }
procedure GenerateAnswer;
var
   A: Array[1..6] of Byte;
   I, Num: Integer;
begin
     for I := 1 to 6 do
         A[I] := I;

     for I := 1 to 4 do
     begin
          repeat
                Num := Random(6) + 1;
          until (A[Num] <> 0);
          A[Num] := 0;
          Answer[I] := Num;
     end;
end;

{ Guess the current line
  First validate, then check for black pegs, then white pegs, then
  check for a win, then for a loss }
procedure GuessLine;
var
   Black, White: Byte;
   TempLine: TLine;
begin
     if (CheckValid) then
     begin
          TempLine := Lines[CurLine];
          Black := CheckBlackPegs(TempLine);
          White := CheckWhitePegs(TempLine);
          Lines[CurLine].Peg := TempLine.Peg;
          if Not(CheckWon) then
             CheckLost;
          DrawPegs;
          if (GameOver) then
          begin
               EraseColour;
               ErasePiece;
          end else
          begin
               ChangeMsg('You Scored ' + IntToStr(Black) + ' Black Peg(s) and ' + IntToStr(White) + ' White Peg(s)');
               ErasePiece;
               CurPiece := 1;
               Inc(CurLine);
               DrawPiece;
          end;
     end;
end;

{ Move the current colour up or down (negative = up, positive = down) }
procedure MoveColour(Offset: ShortInt);
begin
     EraseColour;
     Colour := Colour + Offset;
     if (Colour > 6) then
        Colour := 1;
     if (Colour < 1) then
        Colour := 6;
     DrawColour;
end;

{ Move the current piece left or right (negative=left, positive = right) }
procedure MovePiece(Offset: ShortInt);
begin
     ErasePiece;
     CurPiece := CurPiece + Offset;
     if (CurPiece > 4) then
        CurPiece := 1;
     if (CurPiece < 1) then
        CurPiece := 4;
     DrawPiece;
end;

{ Start a new game }
procedure NewGame;
begin
     Colour := 1;
     GameOver := False;
     CurLine := 1;
     CurPiece := 1;
     FillChar(Lines, SizeOf(Lines), 0);
     RedrawScreen;
     GenerateAnswer;
end;

{ Place the currently selected colour at the current place on the board }
procedure PlacePiece;
begin
     Lines[CurLine].Piece[CurPiece] := Colour;
     MovePiece(+1);
end;

{ Redraw the pieces and pegs on the board }
procedure RedrawBoard;
var
   X, Y: Integer;
begin
     for Y := 1 to 10 do
     begin
          for X := 1 to 4 do
          begin
               mGotoXY(4 + ((X - 1) * 2), 22 - ((Y - 1) * 2));
               mTextAttr(PieceColours[Lines[Y].Piece[X]]);
               mWrite(#254);
               mGotoXY(14 + (X - 1), 22 - ((Y - 1) * 2));
               mTextAttr(PegColours[Lines[Y].Peg[X]]);
               mWrite(#249);
          end;
     end;
end;

{ Redraw the entire screen }
procedure RedrawScreen;
begin
     mDisplayFile(AppPath + 'MAIN.ANS');
     RedrawBoard;
     DrawPiece;
     DrawColour;
end;

{ A somewhat custom chat procedure
  We use the default procedure from MannDoor, but we redraw the screen
  and change the message when exiting }
procedure _OnChat; far;
begin
     OnChat;
     RedrawScreen;
     ChangeMsg('Chat Mode Terminated');
end;

procedure _OnHangup; far;
begin
     ChangeMsg('Caller Dropped Carrier');
     Sleep(2500);
     Halt;
end;

procedure _OnLocalLogin; far;
var
   S: String;
begin
     mClrScr;
     mCrlf;
     mWriteLn('|0F  ' + ProgVer + ' - Local Login');
     mCrlf;
     mWrite('|07  Enter Your Name: ');
     S := mInput('SYSOP', CHARS_ALPHA + ' ', #0, 40, 40, 31);
     DropInfo.RealName := S;
     DropInfo.Alias := S;
end;

procedure _OnStatusBar; far;
begin
     case StatusBar.Current of
          1: begin
                  FastWrite(Right('Left: ' + SecToHMS(mTimeLeft) + ' '), 1, 25, 31);
                  FastWrite(DropInfo.RealName + ' (' + DropInfo.Alias + ') ', 1, 25, 31);
             end;
          2: begin
                  FastWrite(Right('Idle: ' + SecToMS(mTimeIdle) + ' '), 1, 25, 31);
                  FastWrite(DropInfo.RealName + ' (' + DropInfo.Alias + ') ', 1, 25, 31);
             end;
          3: begin
                  FastWrite(PadRight(#254 + '                      ' + #254 + '             ' + #254 +
                  '                ' + #254 + '             ' + #254, ' ', 80), 1, 25, 30);
                  FastWrite('F1: Toggle StatusBar', 3, 25, 31);
                  FastWrite('Alt-C: Chat', 26, 25, 31);
                  FastWrite('Alt-H: Hang-Up', 40, 25, 31);
                  FastWrite('Alt-K: Kick', 57, 25, 31);
             end;
     end;
end;

procedure _OnSysOpHangup; far;
begin
     ChangeMsg('You Are Unworthy (The SysOp Has Disconnected You)');
     Sleep(2500);
     Tel.Close(True);
     Halt;
end;

procedure _OnSysOpKick; far;
begin
     ChangeMsg('You Are Unworthy (The SysOp Has Kicked You)');
     Sleep(2500);
     Halt;
end;

procedure _OnTimeOut; far;
begin
     ChangeMsg('Come Back When You''re Awake (Idle Limit Exceeded)');
     Sleep(2500);
     Halt;
end;

procedure _OnTimeOutWarning(AMinutes: Byte); far;
begin
     ChangeMsg(IntToStr(AMinutes) + ' Minutes Until An Idle Kick Occurs');
end;

procedure _OnTimeUp; far;
begin
     ChangeMsg('Come Back When You Have More TIme (Ran Out Of Time');
     Sleep(2500);
     Halt;
end;

procedure _OnTimeUpWarning(AMinutes: Byte); far;
begin
     ChangeMsg(IntToStr(AMinutes) + ' Minutes Remaining This Call');
end;

procedure _OnUsage; far;
begin
     mClrScr;
     mCrlf;
     mWriteLn(' USAGE: ' + AppName + ' <parameters>');
     mCrlf;
     mWriteLn(' Parameter List:');
     mCrlf;
     mWriteLn(' -Dxxx   Full Path And FileName Of Dropfile');
     mWriteLn(' -Hxxx   Socket Handle (Win32) or Com Port (DOS)');
     mWriteLn(' -L      Run In Local Mode');
     mWriteLn(' -Nxxx   Node Number To Use');
     mCrlf;
     mWriteLn(' Examples:');
     mCrlf;
     mWriteLn(' ' + AppName + ' -L');
     mWriteLn('  - Run In Local Mode');
     mWriteLn(' ' + AppName + ' -DC:\MANNBBS\NODE1\DOOR32.SYS');
     mWriteLn('  - Read Connection Information From Dropfile');
     mWriteLn(' ' + AppName + ' -H4 -N2');
     mWriteLn('  - Run Using Socket Handle/Com 4 On Node 2');
     mCrlf;
     Halt;
end;

{ The start of the program }
var
   Ch: Char;
begin
     mOnChat := _OnChat;
     mOnHangup := _OnHangup;
     mOnLocalLogin := _OnLocalLogin;
     mOnStatusBar := _OnStatusBar;
     mOnSysopHangup := _OnSysopHangup;
     mOnSysopKick := _OnSysopKick;
     mOnTimeOut := _OnTimeOut;
     mOnTimeOutWarning := _OnTimeOutWarning;
     mOnTimeUp := _OnTimeUp;
     mOnTimeUpWarning := _OnTimeUpWarning;
     mOnUsage := _OnUsage;
     StatusBar.Count := 3;
     mStartUp;

     Randomize;
     NewGame;
     ChangeMsg('Welcome to ' + ProgVer);
     repeat
           Ch := UpCase(mReadKey);
           if (GameOver) and Not(Ch in ['N', 'Q']) then
              Continue;
           case Ch of
                '2': MoveColour(+1);
                '4': MovePiece(-1);
                '6': MovePiece(+1);
                '8': MoveColour(-1);
                'N': if (GameOver) then
                        NewGame
                     else
                         ChangeMsg('Finish This Game First');
                #13: PlacePiece;
                #32: GuessLine;
           end;
     until (Ch = 'Q');
     mGotoXY(1, 25);
     mCrlf;
end.
