/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* mdm_lnx.c - Linux modem routines */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/errno.h>

#include "bbs_func.h"
#include "output.h"
#include "modem.h"
#include "logfile.h"
#include "userbase.h"

HFILE hCom = 0; /* Com port handle */
char devname[MAX_NAME_LEN];
int real_carrier, carrier = 1;
int reqx, reqy; /* Response to esc[2n command */
int close_handle = 0;

static struct termio savetio;

int tty_raw(void)
{
    struct termio tio;

    if (ioctl(hCom,TCGETA,&savetio) != 0) return 0;
    tio=savetio;
    tio.c_iflag=0;
    tio.c_oflag=0;
    tio.c_cflag &= ~(CSTOPB | PARENB | PARODD);
    tio.c_cflag |= CS8 | CREAD | CLOCAL | HUPCL | CRTSCTS;
    tio.c_cflag &= ~CBAUD;
    tio.c_cflag |= B38400;
    tio.c_lflag=0;
    tio.c_cc[VMIN]=1;
    tio.c_cc[VTIME]=0;

    return ioctl(hCom,TCSETAW,&tio) == 0;
}

int tty_old(void)
{
    return ioctl(hCom,TCSETAW,&savetio) == 0;
}

int init_modem(unsigned long handle, char *device)
{
    strcpy(devname, device);

    if (device[0] != '\0' || handle != 0)
    {
        close_handle = handle == 0;

        if (handle == 0 && device[0] != '\0')
        {
            /* Open device */
            handle = open(device, O_RDWR);
        }

        hCom = handle;
    }

    if (!tty_raw())
    {
        printf("Can't set device to raw mode.\n");
        write_log("Can't set device to raw mode.");
    }

    carrier = 1;
    real_carrier = 1;

    return hCom != 0;
}

void deinit_modem(void)
{
    if (hCom == 0) return;

    tty_old();

    if (close_handle) close(hCom);
}

int mdm_dataout(void *data, int size)
{
    if (hCom == 0) return size;

    return write(hCom,data,size);
}

int mdm_dataout_nowait(void *data, int size)
{
    int num;

    if (hCom == 0) return size;

    if (ioctl(hCom, TIOCOUTQ, &num) != 0) return 0;
    if (num > 1024) return 0;

    if (size > 1024-num) size = 1024-num;
    return write(hCom, data, size);
}

int mdm_strout(char *data)
{
    if (hCom == 0) return 1;

    return write(hCom,data,strlen(data));
}

int mdm_chrout(char chr)
{
    if (hCom == 0) return 1;

    return write(hCom,&chr,1);
}

int mdm_kbhit(void)
{
    int numchar;

    if (hCom == 0) return 0;

    if (mdm_waiting) return 1;
    if (ioctl(hCom, FIONREAD, &numchar) < 0) return 0;

    return numchar > 0;
}

unsigned char mdm_charin(void)
{
    unsigned char hitch;

    if (read(hCom, &hitch, 1) != 1) return 0;
    return hitch;
}

int wait_modem(void)
{
    static fd_set fdset;
    struct timeval wait;

    if (hCom == 0) return 0;

    wait.tv_sec = 1;
    wait.tv_usec = 0;

    FD_ZERO(&fdset);
    FD_SET(hCom, &fdset);

    return select(hCom+1, &fdset, NULL, NULL, &wait) >= 1;
}

int mdm_datain(void *data, int size)
{
    int numchar, readed;

    if (ioctl(hCom, FIONREAD, &numchar) < 0) return 0;

    if (numchar == 0) return 0;

    if (numchar > size) numchar = size;
    readed = read(hCom, data, numchar);
    if (readed < 0)
    {
        printf("mdm_datain() : readed < %d\n", readed);
        write_log("mdm_datain() : readed < %d", readed);
        readed = 0;
    }
    return readed;
}

void modem_setdtr(int dtr_on)
{
    struct termios tios;

    if (hCom == 0) return;

    if (tcgetattr(hCom, &tios)) return;
    if (dtr_on)
        if (cfsetospeed(&tios, 38400)) return;
    else
        if (cfsetospeed(&tios, 0)) return;
    if (tcsetattr(hCom, TCSANOW, &tios)) return;

    if (!dtr_on) sleep(1);
}

int carr_det(void)
{
    int tmp = 0;

    if (!carrier) return 0;

    if (hCom == 0) return 1;
    if(ioctl(hCom, TIOCMGET, &tmp) < 0) return 1;

    carrier = (tmp & TIOCM_CD) != 0;
    if (!carrier)
    {
        printf("Carrier lost.\n");
        sleep(3);
        if(ioctl(hCom, TIOCMGET, &tmp) < 0) return 0;
        carrier = (tmp & TIOCM_CD) != 0;
        if (carrier)
            printf("On a second though: It didn't.\n");
        else
            write_log("Carrier lost.");
    }
    real_carrier = carrier;
    return (tmp & TIOCM_CD) != 0;
}
